package ru.yandex.calendar.frontend.ews;

import java.util.regex.Pattern;

import com.microsoft.schemas.exchange.services._2006.types.EmailAddressType;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.ews.subscriber.ExchangeSubscriber;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;

@Slf4j
public class ExchangeEmailManager {
    private static final String YA_TEAM_DOMAIN = "@yandex-team.ru";

    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private ExchangeSubscriber exchangeSubscriber;
    @Autowired
    private EventInvitationManager eventInvitationManager;

    /**
     * Clever routine that accepts any email (including @ld).
     * If subscription does not exist, returns email as is.
     * @param email @yandex-team, @msft, @ld
     * @return @yandex-team, @msft, @ld
     */
    public Email getExchangeEmailByAnyEmailOrGetAsIs(Email email) {
        try {
            return getExchangeEmailByEmail(email);
        } catch (EwsSubjectNotSubscribedException e) {
            log.warn("Subject not subscribed -- ok, using email as is: {}", email);
            return email;
        } catch (Exception e) {
            log.warn("Could not get exchange email by email ({})", email, e);
            throw EwsUtils.translate(e);
        }
    }

    public Tuple2List<Email, Email> getExchangeOrAsIsEmailsSafeByEmails(ListF<Email> emails) {
        MapF<Email, UidOrResourceId> subjectIdByEmail = eventInvitationManager.getParticipantIdsByEmails(emails)
                .filterMap(t -> t.get2().getSubjectId().map(id -> Tuple2.tuple(t.get1(), id)))
                .toMap(Tuple2::get1, Tuple2::get2);

        MapF<UidOrResourceId, Email> subscriptionEmailBySubjectId =
                exchangeSubscriber.findSubscriptionsBySubjectIds(subjectIdByEmail.values())
                        .filter(s -> s.getEmail().isPresent())
                        .toMap(s -> UidOrResourceId.fromOptions(s.getUid(), s.getResourceId()), s -> s.getEmail().get());

        return emails.zipWith(e -> subjectIdByEmail.getO(e).flatMapO(subscriptionEmailBySubjectId::getO).getOrElse(e));
    }

    /**
     * NOTE: requires ews subscription
     * @param email @yandex-team, @msft, @ld
     * @return @yandex-team, @msft, @ld
     */
    public Email getExchangeEmailByEmail(Email email) {
        Email emailForSubject = getEmailByExchangeEmail(email, true);
        YtEwsSubscription subscription = exchangeSubscriber.findSubscriptionByEmail(emailForSubject);
        return getExchangeEmailBySubscription(subscription);
    }

    public Email getExchangeEmailFromSubscriptionOrBySubjectId(UidOrResourceId subjectId) {
        return exchangeSubscriber.findSubscriptionOBySubjectId(subjectId).map(s -> s.getEmail().get())
                .getOrElse(() -> getExchangeEmailBySubjectId(subjectId));
    }

    public Email getExchangeEmailBySubscription(YtEwsSubscription subscription) {
        return subscription.getEmail()
            .toOptional()
            .orElseGet(() -> getExchangeEmailBySubjectId(EwsUtils.getSubjectId(subscription)));
    }

    public Option<Email> resolveDomainIfNeeded(Email email) {
        if (Cf.list("winadminhdsandbox.yandex.net", "msft.yandex-team.ru", "ld.yandex.ru").containsTs(email.getDomain().getDomain())) {
            return Option.x(userManager.getYtUserEmailByLogin(email.getLocalPart()));
        } else {
            return Option.of(email);
        }
    }

    public Option<Email> resolveDomainIfNeeded(EmailAddressType emailAddress) {
        try {
            return resolveDomainIfNeeded(new Email(emailAddress.getEmailAddress()));
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            return Option.empty();
        }
    }

    /**
     * this is only fallback, correctness is not guaranteed
     * @return @yandex-team, @ld, @msft
     */
    @Deprecated
    private Email getExchangeEmailBySubjectId(UidOrResourceId subjectId) {
        if (subjectId.isUser()) {
            return userManager.getLdEmailByUid(subjectId.getUid());
        } else if (subjectId.isResource()) {
            return resourceRoutines.getExchangeEmail(resourceRoutines.loadById(subjectId.getResourceId()));
        } else {
            throw new IllegalStateException("Unknown subjectId type: " + subjectId);
        }
    }

    /**
     * ignores @ld/@yandex-team difference.
     * XXX may become needless when subscription's primary email is made not null
     */
    public static boolean emailsNormalizedByLdDomainEqual(Email email1, Email email2) {
        return UserManager.normalize(email1).equalsIgnoreCase(UserManager.normalize(email2));
    }

    private static final Pattern EXCHANGE_DOMAIN =
        Pattern.compile("@(msft\\.yandex-team\\.ru|winadminhdsandbox\\.yandex\\.net)");

    public static String getEmailValueByExchangeEmailValue(String value, boolean noteLd) {
        val result = EXCHANGE_DOMAIN.matcher(value).replaceFirst(YA_TEAM_DOMAIN);
        return noteLd ? result.replace("@ld.yandex.ru", YA_TEAM_DOMAIN) : result;
    }

    public static Email getEmailByExchangeEmail(Email exchangeEmail, boolean noteLd) {
        return new Email(getEmailValueByExchangeEmailValue(exchangeEmail.getEmail(), noteLd));
    }

    public Function<YtEwsSubscription, Email> getExchangeEmailBySubscriptionF() {
        return this::getExchangeEmailBySubscription;
    }
}
