package ru.yandex.calendar.frontend.ews;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;

/**
 * Windows has its own reference to timezones by their names.
 * We should know it, for example, when change timezone in exchange items.
 *
 * This class performs conversion of timezone to- and from timezone name.
 *
 * Details and source mapping can be found here:
 *     http://jexp.ru/index.php/Java_Tutorial/Development/TimeZone
 *     http://www.java2s.com/Tutorial/Java/0120__Development/Timezoneconversionroutines.htm
 *
 * Unfortunately, this mapping seems somewhat poor
 * (for instance, there's no mapping for "Europe/Istanbul"),
 * so that only most popular timezones present here.
 * If anybody finds more rich mapping, please update it here.
 *
 * @see <a href="https://jira.yandex-team.ru/browse/CAL-4239">CAL-4239</a>
 * @see <a href="https://www.bedework.org/svn/calendarapi/trunk/calCore/resources/properties/timezonesAliases.properties">timezonesAliases.properties</a>
 *
 * @author ssytnik
 */
public class WindowsTimeZones {
    private static final Pattern timezoneWithOffsetPattern = Pattern.compile(
            "^\\((?:UTC|GMT)(?:([+-])(\\d+)(?::(\\d+))?)?\\)\\s+(.+?)(?: \\(RTZ[^)]+\\))?$");

    private static final MapF<String, DateTimeZone> winName2zone;
    private static final MapF<DateTimeZone, String> zone2winName;
    /**
     * ssytnik@: NOTE: mapping to existing {@link #winName2zone} entry it is NOT guaranteed.
     * Offsets like (GMT+03:00), (UTC+04:00) or (GMT) are cut, because they can be inaccurate.
     */
    private static final MapF<String, String> localizedName2winName;

    static {
        Tuple2List<String, DateTimeZone> winName2zoneList = Tuple2List.arrayList();

        winName2zoneList.add("(UTC) Coordinated Universal Time", DateTimeZone.forID("UTC"));
        winName2zoneList.add("UTC", DateTimeZone.forID("UTC"));
        winName2zoneList.add("Romance Standard Time", DateTimeZone.forID("Europe/Paris"));
        winName2zoneList.add("Central Europe Standard Time", DateTimeZone.forID("Europe/Prague"));
        winName2zoneList.add("W. Central Africa Standard Time", DateTimeZone.forID("Africa/Luanda"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Helsinki"));
        winName2zoneList.add("GTB Standard Time", DateTimeZone.forID("Europe/Athens"));
        winName2zoneList.add("Israel Standard Time", DateTimeZone.forID("Asia/Jerusalem"));
        winName2zoneList.add("Arab Standard Time", DateTimeZone.forID("Asia/Riyadh"));
        winName2zoneList.add("Arabic Standard Time", DateTimeZone.forID("Asia/Baghdad"));
        winName2zoneList.add("E. Africa Standard Time", DateTimeZone.forID("Africa/Nairobi"));
        winName2zoneList.add("Iran Standard Time", DateTimeZone.forID("Asia/Tehran"));
        winName2zoneList.add("Afghanistan Standard Time", DateTimeZone.forID("Asia/Kabul"));
        winName2zoneList.add("India Standard Time", DateTimeZone.forID("Asia/Calcutta"));
        winName2zoneList.add("Myanmar Standard Time", DateTimeZone.forID("Asia/Rangoon"));
        winName2zoneList.add("Nepal Standard Time", DateTimeZone.forID("Asia/Katmandu"));
        winName2zoneList.add("Sri Lanka Standard Time", DateTimeZone.forID("Asia/Colombo"));
        winName2zoneList.add("China Standard Time", DateTimeZone.forID("Asia/Hong_Kong"));
        winName2zoneList.add("AUS Central Standard Time", DateTimeZone.forID("Australia/Darwin"));
        winName2zoneList.add("Cen. Australia Standard Time", DateTimeZone.forID("Australia/Adelaide"));
        winName2zoneList.add("Vladivostok Standard Time", DateTimeZone.forID("Asia/Vladivostok"));
        winName2zoneList.add("West Pacific Standard Time", DateTimeZone.forID("Pacific/Guam"));
        winName2zoneList.add("E. South America Standard Time", DateTimeZone.forID("America/Sao_Paulo"));
        winName2zoneList.add("Greenland Standard Time", DateTimeZone.forID("America/Godthab"));
        winName2zoneList.add("Newfoundland Standard Time", DateTimeZone.forID("America/St_Johns"));
        winName2zoneList.add("Pacific SA Standard Time", DateTimeZone.forID("America/Santiago"));
        winName2zoneList.add("SA Western Standard Time", DateTimeZone.forID("America/Caracas"));
        winName2zoneList.add("SA Pacific Standard Time", DateTimeZone.forID("America/Bogota"));
        winName2zoneList.add("US Eastern Standard Time", DateTimeZone.forID("America/Indianapolis"));
        winName2zoneList.add("Central America Standard Time", DateTimeZone.forID("America/Regina"));
        winName2zoneList.add("Mexico Standard Time", DateTimeZone.forID("America/Mexico_City"));
        winName2zoneList.add("Canada Central Standard Time", DateTimeZone.forID("America/Regina"));
        winName2zoneList.add("US Mountain Standard Time", DateTimeZone.forID("America/Phoenix"));
        winName2zoneList.add("GMT Standard Time", DateTimeZone.forID("Europe/London"));
        winName2zoneList.add("Ekaterinburg Standard Time", DateTimeZone.forID("Asia/Yekaterinburg"));
        winName2zoneList.add("West Asia Standard Time", DateTimeZone.forID("Asia/Karachi"));
        winName2zoneList.add("Central Asia Standard Time", DateTimeZone.forID("Asia/Dhaka"));
        winName2zoneList.add("N. Central Asia Standard Time", DateTimeZone.forID("Asia/Novosibirsk"));
        winName2zoneList.add("North Asia Standard Time", DateTimeZone.forID("Asia/Krasnoyarsk"));
        winName2zoneList.add("SE Asia Standard Time", DateTimeZone.forID("Asia/Bangkok"));
        winName2zoneList.add("North Asia East Standard Time", DateTimeZone.forID("Asia/Ulaanbaatar"));
        winName2zoneList.add("Singapore Standard Time", DateTimeZone.forID("Asia/Singapore"));
        winName2zoneList.add("Taipei Standard Time", DateTimeZone.forID("Asia/Taipei"));
        winName2zoneList.add("W. Australia Standard Time", DateTimeZone.forID("Australia/Perth"));
        winName2zoneList.add("Korea Standard Time", DateTimeZone.forID("Asia/Seoul"));
        winName2zoneList.add("Tokyo Standard Time", DateTimeZone.forID("Asia/Tokyo"));
        winName2zoneList.add("Yakutsk Standard Time", DateTimeZone.forID("Asia/Yakutsk"));
        winName2zoneList.add("Central European Standard Time", DateTimeZone.forID("Europe/Belgrade"));
        winName2zoneList.add("W. Europe Standard Time", DateTimeZone.forID("Europe/Berlin"));
        winName2zoneList.add("Tasmania Standard Time", DateTimeZone.forID("Australia/Hobart"));
        winName2zoneList.add("AUS Eastern Standard Time", DateTimeZone.forID("Australia/Sydney"));
        winName2zoneList.add("E. Australia Standard Time", DateTimeZone.forID("Australia/Brisbane"));
        winName2zoneList.add("Central Pacific Standard Time", DateTimeZone.forID("Pacific/Guadalcanal"));
        winName2zoneList.add("Dateline Standard Time", DateTimeZone.forID("Etc/GMT-12"));
        winName2zoneList.add("Fiji Standard Time", DateTimeZone.forID("Pacific/Fiji"));
        winName2zoneList.add("Samoa Standard Time", DateTimeZone.forID("Pacific/Samoa"));
        winName2zoneList.add("Hawaiian Standard Time", DateTimeZone.forID("Pacific/Honolulu"));
        winName2zoneList.add("Alaskan Standard Time", DateTimeZone.forID("America/Anchorage"));
        winName2zoneList.add("Pacific Standard Time", DateTimeZone.forID("America/Los_Angeles"));
        winName2zoneList.add("Mexico Standard Time 2", DateTimeZone.forID("America/Chihuahua"));
        winName2zoneList.add("Mountain Standard Time", DateTimeZone.forID("America/Denver"));
        winName2zoneList.add("Central Standard Time", DateTimeZone.forID("America/Chicago"));
        winName2zoneList.add("Eastern Standard Time", DateTimeZone.forID("America/New_York"));
        winName2zoneList.add("E. Europe Standard Time", DateTimeZone.forID("Europe/Bucharest"));
        winName2zoneList.add("Egypt Standard Time", DateTimeZone.forID("Africa/Cairo"));
        winName2zoneList.add("South Africa Standard Time", DateTimeZone.forID("Africa/Harare"));
        winName2zoneList.add("Atlantic Standard Time", DateTimeZone.forID("America/Halifax"));
        winName2zoneList.add("SA Eastern Standard Time", DateTimeZone.forID("America/Buenos_Aires"));
        winName2zoneList.add("Azores Standard Time", DateTimeZone.forID("Atlantic/Azores"));
        winName2zoneList.add("Cape Verde Standard Time", DateTimeZone.forID("Atlantic/Cape_Verde"));
        winName2zoneList.add("Russian Standard Time", DateTimeZone.forID("Europe/Moscow"));
        winName2zoneList.add("New Zealand Standard Time", DateTimeZone.forID("Pacific/Auckland"));
        winName2zoneList.add("Tonga Standard Time", DateTimeZone.forID("Pacific/Tongatapu"));
        winName2zoneList.add("Arabian Standard Time", DateTimeZone.forID("Asia/Muscat"));
        winName2zoneList.add("Caucasus Standard Time", DateTimeZone.forID("Asia/Tbilisi"));
        winName2zoneList.add("Morocco Standard Time", DateTimeZone.forID("Africa/Casablanca"));
        winName2zoneList.add("Pacific Standard Time (Mexico)", DateTimeZone.forID("America/Tijuana"));
        winName2zoneList.add("Magadan Standard Time", DateTimeZone.forID("Asia/Magadan"));
        winName2zoneList.add("Mid-Atlantic Standard Time", DateTimeZone.forID("America/Noronha"));
        winName2zoneList.add("Turkey Standard Time", DateTimeZone.forID("Europe/Istanbul"));
        winName2zoneList.add("Greenwich Standard Time", DateTimeZone.forID("GMT"));

        winName2zoneList.add("Belarus Standard Time", DateTimeZone.forID("Europe/Minsk"));

        winName2zoneList.add("Kaliningrad Standard Time", DateTimeZone.forID("Europe/Kaliningrad"));

        //https://st.yandex-team.ru/GREG-1053
        winName2zoneList.add("Russia Time Zone 3", DateTimeZone.forID("Europe/Samara"));
        winName2zoneList.add("Saratov Standard Time", DateTimeZone.forID("Europe/Saratov"));

        winName2zone = winName2zoneList.toMap();

        winName2zoneList.add("Russian Standard Time", DateTimeZone.forID("Europe/Simferopol"));

        // XXX (UTC+04:00) Izhevsk, Samara name after Exchange server update
        winName2zoneList.add("Central Asia Standard Time", DateTimeZone.forID("Asia/Almaty"));
        winName2zoneList.add("North Asia East Standard Time", DateTimeZone.forID("Asia/Irkutsk"));
        winName2zoneList.add("Vladivostok Standard Time", DateTimeZone.forID("Asia/Sakhalin"));
        winName2zoneList.add("Central Standard Time", DateTimeZone.forID("America/Managua"));
        winName2zoneList.add("W. Central Africa Standard Time", DateTimeZone.forID("Africa/Lagos"));
        winName2zoneList.add("South Africa Standard Time", DateTimeZone.forID("Africa/Johannesburg"));
        winName2zoneList.add("Central Europe Standard Time", DateTimeZone.forID("Europe/Zurich"));
        winName2zoneList.add("Central Europe Standard Time", DateTimeZone.forID("CET"));
        // XXX , (UTC+11:00) Chokurdakh (RTZ10) after Exchange server update

        winName2zoneList.add("UTC", DateTimeZone.forID("Atlantic/Reykjavik"));
        winName2zoneList.add("Central Europe Standard Time", DateTimeZone.forID("Europe/Budapest"));
        winName2zoneList.add("Central European Standard Time", DateTimeZone.forID("Europe/Tallinn"));
        winName2zoneList.add("Central European Standard Time", DateTimeZone.forID("Europe/Luxembourg"));
        winName2zoneList.add("Central European Standard Time", DateTimeZone.forID("Europe/Warsaw"));
        winName2zoneList.add("Romance Standard Time", DateTimeZone.forID("Europe/Brussels"));
        winName2zoneList.add("Romance Standard Time", DateTimeZone.forID("Europe/Madrid"));
        winName2zoneList.add("Romance Standard Time", DateTimeZone.forID("Europe/Copenhagen"));
        winName2zoneList.add("W. Europe Standard Time", DateTimeZone.forID("Europe/Amsterdam"));
        winName2zoneList.add("W. Europe Standard Time", DateTimeZone.forID("Europe/Rome"));
        winName2zoneList.add("Caucasus Standard Time", DateTimeZone.forID("Asia/Baku"));
        winName2zoneList.add("Caucasus Standard Time", DateTimeZone.forID("Asia/Yerevan"));
        winName2zoneList.add("China Standard Time", DateTimeZone.forID("Asia/Shanghai"));
        winName2zoneList.add("Eastern Standard Time", DateTimeZone.forID("America/Detroit"));
        winName2zoneList.add("GMT Standard Time", DateTimeZone.forID("Europe/Dublin"));
        winName2zoneList.add("West Asia Standard Time", DateTimeZone.forID("Indian/Maldives"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Tallinn"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Kiev"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Riga"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Sofia"));
        winName2zoneList.add("FLE Standard Time", DateTimeZone.forID("Europe/Vilnius"));

        //https://support.microsoft.com/ru-ru/help/4468323/dst-and-time-zone-changes-in-windows-for-morocco-and-volgograd
        winName2zoneList.add("Volgograd Standard Time", DateTimeZone.forID("Europe/Volgograd"));

        zone2winName = winName2zoneList.invert().toMap();


        localizedName2winName = Cf.hashMap();

        localizedName2winName.put("Kabul", "Afghanistan Standard Time");
        localizedName2winName.put("Alaska", "Alaskan Standard Time");
        localizedName2winName.put("Kuwait, Riyadh", "Arab Standard Time");
        localizedName2winName.put("Abu Dhabi, Muscat", "Arabian Standard Time");
        localizedName2winName.put("Baghdad", "Arabic Standard Time");
        localizedName2winName.put("Atlantic Time (Canada)", "Atlantic Standard Time");
        localizedName2winName.put("Darwin", "AUS Central Standard Time");
        localizedName2winName.put("Canberra, Melbourne, Sydney", "AUS Eastern Standard Time");
        localizedName2winName.put("Azores", "Azores Standard Time");
        localizedName2winName.put("Saskatchewan", "Canada Central Standard Time");
        localizedName2winName.put("Cape Verde Is.", "Cape Verde Standard Time");
        localizedName2winName.put("Baku, Tbilisi, Yerevan", "Caucasus Standard Time");
        localizedName2winName.put("Adelaide", "Cen. Australia Standard Time");
        localizedName2winName.put("Central America", "Central America Standard Time");
        localizedName2winName.put("Astana, Dhaka", "Central Asia Standard Time");
        localizedName2winName.put("Belgrade, Bratislava, Budapest, Ljubljana, Prague", "Central Europe Standard Time");
        localizedName2winName.put("Sarajevo, Skopje, Warsaw, Zagreb", "Central European Standard Time");
        localizedName2winName.put("Magadan, Solomon Is., New Caledonia", "Central Pacific Standard Time");
        localizedName2winName.put("Central Time (US & Canada)", "Central Standard Time");
        localizedName2winName.put("Beijing, Chongqing, Hong Kong, Urumqi", "China Standard Time");
        localizedName2winName.put("International Date Line West", "Dateline Standard Time");
        localizedName2winName.put("Nairobi", "E. Africa Standard Time");
        localizedName2winName.put("Brisbane", "E. Australia Standard Time");
        localizedName2winName.put("Bucharest", "E. Europe Standard Time");
        localizedName2winName.put("Brasilia", "E. South America Standard Time");
        localizedName2winName.put("Eastern Time (US & Canada)", "Eastern Standard Time");
        localizedName2winName.put("Cairo", "Egypt Standard Time");
        localizedName2winName.put("Ekaterinburg", "Ekaterinburg Standard Time");
        localizedName2winName.put("Fiji, Kamchatka, Marshall Is.", "Fiji Standard Time");
        localizedName2winName.put("Helsinki, Kyiv, Riga, Sofia, Tallinn, Vilnius", "FLE Standard Time");
        localizedName2winName.put("Greenwich Mean Time : Dublin, Edinburgh, Lisbon, London", "GMT Standard Time");
        localizedName2winName.put("Greenland", "Greenland Standard Time");
        localizedName2winName.put("Monrovia, Reykjavik", "Greenwich Standard Time");
        localizedName2winName.put("Casablanca", "Morocco Standard Time");
        localizedName2winName.put("Athens, Beirut, Istanbul, Minsk", "GTB Standard Time");
        localizedName2winName.put("Hawaii", "Hawaiian Standard Time");
        localizedName2winName.put("Chennai, Kolkata, Mumbai, New Delhi", "India Standard Time");
        localizedName2winName.put("Tehran", "Iran Standard Time");
        localizedName2winName.put("Jerusalem", "Israel Standard Time");
        localizedName2winName.put("Petropavlovsk-Kamchatsky - Old", "Kamchatka Standard Time");
        localizedName2winName.put("Anadyr, Petropavlovsk-Kamchatsky", "Kamchatka Standard Time");
        localizedName2winName.put("Seoul", "Korea Standard Time");
        localizedName2winName.put("Guadalajara, Mexico City, Monterrey", "Mexico Standard Time");
        localizedName2winName.put("Chihuahua, La Paz, Mazatlan", "Mexico Standard Time 2");
        localizedName2winName.put("Mid-Atlantic", "Mid-Atlantic Standard Time");
        localizedName2winName.put("Mountain Time (US & Canada)", "Mountain Standard Time");
        localizedName2winName.put("Rangoon", "Myanmar Standard Time");
        localizedName2winName.put("Almaty, Novosibirsk", "N. Central Asia Standard Time");
        localizedName2winName.put("Novosibirsk", "N. Central Asia Standard Time");
        localizedName2winName.put("Kathmandu", "Nepal Standard Time");
        localizedName2winName.put("Auckland, Wellington", "New Zealand Standard Time");
        localizedName2winName.put("Newfoundland", "Newfoundland Standard Time");
        localizedName2winName.put("Irkutsk, Ulaan Bataar", "North Asia East Standard Time");
        localizedName2winName.put("Krasnoyarsk", "North Asia Standard Time");
        localizedName2winName.put("Santiago", "Pacific SA Standard Time");
        localizedName2winName.put("Pacific Time (US & Canada); Tijuana", "Pacific Standard Time");
        localizedName2winName.put("Brussels, Copenhagen, Madrid, Paris", "Romance Standard Time");
        localizedName2winName.put("Moscow, St. Petersburg, Volgograd", "Russian Standard Time");
        localizedName2winName.put("Moscow, St. Petersburg", "Russian Standard Time");
        localizedName2winName.put("Volgograd", "Volgograd Standard Time");

        localizedName2winName.put("Minsk", "Belarus Standard Time");
        localizedName2winName.put("Kaliningrad, Minsk", "Kaliningrad Standard Time");
        localizedName2winName.put("Kaliningrad", "Kaliningrad Standard Time");
        localizedName2winName.put("Buenos Aires, Georgetown", "SA Eastern Standard Time");
        localizedName2winName.put("Bogota, Lima, Quito", "SA Pacific Standard Time");
        localizedName2winName.put("Caracas, La Paz", "SA Western Standard Time");
        localizedName2winName.put("Midway Island, Samoa", "Samoa Standard Time");
        localizedName2winName.put("Bangkok, Hanoi, Jakarta", "SE Asia Standard Time");
        localizedName2winName.put("Kuala Lumpur, Singapore", "Singapore Standard Time");
        localizedName2winName.put("Harare, Pretoria", "South Africa Standard Time");
        localizedName2winName.put("Sri Jayawardenepura", "Sri Lanka Standard Time");
        localizedName2winName.put("Taipei", "Taipei Standard Time");
        localizedName2winName.put("Hobart", "Tasmania Standard Time");
        localizedName2winName.put("Osaka, Sapporo, Tokyo", "Tokyo Standard Time");
        localizedName2winName.put("Nuku'alofa", "Tonga Standard Time");
        localizedName2winName.put("Indiana (East)", "US Eastern Standard Time");
        localizedName2winName.put("Arizona", "US Mountain Standard Time");
        localizedName2winName.put("Vladivostok", "Vladivostok Standard Time");
        localizedName2winName.put("Perth", "W. Australia Standard Time");
        localizedName2winName.put("West Central Africa", "W. Central Africa Standard Time");
        localizedName2winName.put("Amsterdam, Berlin, Bern, Rome, Stockholm, Vienna", "W. Europe Standard Time");
        localizedName2winName.put("Islamabad, Karachi, Tashkent", "West Asia Standard Time");
        localizedName2winName.put("Guam, Port Moresby", "West Pacific Standard Time");
        localizedName2winName.put("Yakutsk", "Yakutsk Standard Time");

        //https://st.yandex-team.ru/GREG-1053
        localizedName2winName.put("Izhevsk, Samara", "Russia Time Zone 3");
        localizedName2winName.put("Saratov", "Saratov Standard Time");

        localizedName2winName.put("Кабул", "Afghanistan Standard Time");
        localizedName2winName.put("Аляска", "Alaskan Standard Time");
        localizedName2winName.put("Кувейт, Эр-Рияд", "Arab Standard Time");
        localizedName2winName.put("Абу-Даби, Мускат", "Arabian Standard Time");
        localizedName2winName.put("Багдад", "Arabic Standard Time");
        localizedName2winName.put("Буэнос-Айрес", "Argentina Standard Time");
        localizedName2winName.put("Атлантическое время (Канада)", "Atlantic Standard Time");
        localizedName2winName.put("Дарвин", "AUS Central Standard Time");
        localizedName2winName.put("Канберра, Мельбурн, Сидней", "AUS Eastern Standard Time");
        localizedName2winName.put("Баку", "Azerbaijan Standard Time");
        localizedName2winName.put("Азорские о-ва", "Azores Standard Time");
        localizedName2winName.put("Дакка", "Bangladesh Standard Time");
        localizedName2winName.put("Саскачеван", "Canada Central Standard Time");
        localizedName2winName.put("О-ва Зеленого Мыса", "Cape Verde Standard Time");
        localizedName2winName.put("Ереван", "Caucasus Standard Time");
        localizedName2winName.put("Аделаида", "Cen. Australia Standard Time");
        localizedName2winName.put("Центральная Америка", "Central America Standard Time");
        localizedName2winName.put("Астана", "Central Asia Standard Time");
        localizedName2winName.put("Куяба", "Central Brazilian Standard Time");
        localizedName2winName.put("Белград, Братислава, Будапешт, Любляна, Прага", "Central Europe Standard Time");
        localizedName2winName.put("Варшава, Загреб, Сараево, Скопье", "Central European Standard Time");
        localizedName2winName.put("Соломоновы о-ва, Нов. Каледония", "Central Pacific Standard Time");
        localizedName2winName.put("Центральное время (США и Канада)", "Central Standard Time");
        localizedName2winName.put("Гвадалахара, Мехико, Монтеррей", "Central Standard Time (Mexico)");
        localizedName2winName.put("Гонконг, Пекин, Урумчи, Чунцин", "China Standard Time");
        localizedName2winName.put("Линия перемены дат", "Dateline Standard Time");
        localizedName2winName.put("Найроби", "E. Africa Standard Time");
        localizedName2winName.put("Брисбен", "E. Australia Standard Time");

        localizedName2winName.put("Минск", "Belarus Standard Time");
        localizedName2winName.put("Калининград, Минск", "Kaliningrad Standard Time");
        localizedName2winName.put("Калининград", "Kaliningrad Standard Time");
        localizedName2winName.put("Бразилия", "E. South America Standard Time");
        localizedName2winName.put("Восточное время (США и Канада)", "Eastern Standard Time");
        localizedName2winName.put("Каир", "Egypt Standard Time");
        localizedName2winName.put("Екатеринбург", "Ekaterinburg Standard Time");
        localizedName2winName.put("Фиджи", "Fiji Standard Time");
        localizedName2winName.put("Вильнюс, Киев, Рига, София, Таллин, Хельсинки", "FLE Standard Time");
        localizedName2winName.put("Тбилиси", "Georgian Standard Time");
        localizedName2winName.put("Дублин, Лиссабон, Лондон, Эдинбург", "GMT Standard Time");
        localizedName2winName.put("Гренландия", "Greenland Standard Time");
        localizedName2winName.put("Монровия, Рейкьявик", "Greenwich Standard Time");
        localizedName2winName.put("Афины, Бухарест, Стамбул", "GTB Standard Time");
        localizedName2winName.put("Гавайи", "Hawaiian Standard Time");
        localizedName2winName.put("Колката, Мумбаи, Нью-Дели, Ченнай", "India Standard Time");
        localizedName2winName.put("Тегеран", "Iran Standard Time");
        localizedName2winName.put("Иерусалим", "Israel Standard Time");
        localizedName2winName.put("Амман", "Jordan Standard Time");
        localizedName2winName.put("Петропавловск-Камчатский — устаревшее", "Kamchatka Standard Time");
        localizedName2winName.put("Анадырь, Петропавловск-Камчатский", "Kamchatka Standard Time");
        localizedName2winName.put("Сеул", "Korea Standard Time");
        localizedName2winName.put("Магадан", "Magadan Standard Time");
        localizedName2winName.put("Порт-Луи", "Mauritius Standard Time");
        localizedName2winName.put("Среднеатлантическое время", "Mid-Atlantic Standard Time");
        localizedName2winName.put("Бейрут", "Middle East Standard Time");
        localizedName2winName.put("Монтевидео", "Montevideo Standard Time");
        localizedName2winName.put("Касабланка", "Morocco Standard Time");
        localizedName2winName.put("Горное время (США и Канада)", "Mountain Standard Time");
        localizedName2winName.put("Ла-Пас, Мазатлан, Чихуахуа", "Mountain Standard Time (Mexico)");
        localizedName2winName.put("Янгон", "Myanmar Standard Time");
        localizedName2winName.put("Новосибирск", "N. Central Asia Standard Time");
        localizedName2winName.put("Виндхук", "Namibia Standard Time");
        localizedName2winName.put("Катманду", "Nepal Standard Time");
        localizedName2winName.put("Веллингтон, Окленд", "New Zealand Standard Time");
        localizedName2winName.put("Ньюфаундленд", "Newfoundland Standard Time");
        localizedName2winName.put("Иркутск", "North Asia East Standard Time");
        localizedName2winName.put("Красноярск", "North Asia Standard Time");
        localizedName2winName.put("Сантьяго", "Pacific SA Standard Time");
        localizedName2winName.put("Тихоокеанское время (США и Канада)", "Pacific Standard Time");
        localizedName2winName.put("Нижняя Калифорния", "Pacific Standard Time (Mexico)");
        localizedName2winName.put("Исламабад, Карачи", "Pakistan Standard Time");
        localizedName2winName.put("Асунсьон", "Paraguay Standard Time");
        localizedName2winName.put("Брюссель, Копенгаген, Мадрид, Париж", "Romance Standard Time");
        localizedName2winName.put("Волгоград, Москва, Санкт-Петербург", "Russian Standard Time");
        localizedName2winName.put("Москва, Санкт-Петербург", "Russian Standard Time");
        localizedName2winName.put("Волгоградская", "Volgograd Standard Time");
        localizedName2winName.put("Кайенна, Форталеза", "SA Eastern Standard Time");
        localizedName2winName.put("Богота, Кито, Лима", "SA Pacific Standard Time");
        localizedName2winName.put("Джорджтаун, Ла-Пас, Манаус, Сан-Хуан", "SA Western Standard Time");
        localizedName2winName.put("Самоа", "Samoa Standard Time");
        localizedName2winName.put("Бангкок, Джакарта, Ханой", "SE Asia Standard Time");
        localizedName2winName.put("Куала-Лумпур, Сингапур", "Singapore Standard Time");
        localizedName2winName.put("Хараре, Претория", "South Africa Standard Time");
        localizedName2winName.put("Шри-Джаявардене-пура-Котте", "Sri Lanka Standard Time");
        localizedName2winName.put("Дамаск", "Syria Standard Time");
        localizedName2winName.put("Тайбэй", "Taipei Standard Time");
        localizedName2winName.put("Хобарт", "Tasmania Standard Time");
        localizedName2winName.put("Осака, Саппоро, Токио", "Tokyo Standard Time");
        localizedName2winName.put("Нукуалофа", "Tonga Standard Time");
        localizedName2winName.put("Улан-Батор", "Ulaanbaatar Standard Time");
        localizedName2winName.put("Индиана (восток)", "US Eastern Standard Time");
        localizedName2winName.put("Аризона", "US Mountain Standard Time");
        localizedName2winName.put("Время в формате UTC", "UTC");
        localizedName2winName.put("Время в формате UTC +12", "UTC+12");
        localizedName2winName.put("Время в формате UTC -02", "UTC-02");
        localizedName2winName.put("Время в формате UTC -11", "UTC-11");
        localizedName2winName.put("Каракас", "Venezuela Standard Time");
        localizedName2winName.put("Владивосток", "Vladivostok Standard Time");
        localizedName2winName.put("Перт", "W. Australia Standard Time");
        localizedName2winName.put("Западная Центральная Африка", "W. Central Africa Standard Time");
        localizedName2winName.put("Амстердам, Берлин, Берн, Вена, Рим, Стокгольм", "W. Europe Standard Time");
        localizedName2winName.put("Ташкент", "West Asia Standard Time");
        localizedName2winName.put("Гуам, Порт-Морсби", "West Pacific Standard Time");
        localizedName2winName.put("Якутск", "Yakutsk Standard Time");

        // hack for old exchange events
        localizedName2winName.put("Москва, Санкт-Петербург, Волгоград", "Russian Standard Time");

        //https://st.yandex-team.ru/GREG-1053
        localizedName2winName.put("Ижевск, Самара", "Russia Time Zone 3");
        localizedName2winName.put("Саратов", "Saratov Standard Time");

        /*for (String winName : localizedName2winName.values()) {
            Check.C.isTrue(winName2zone.containsKeyTs(winName));
        }*/
    }

    public static Option<String> getWinNameByZone(DateTimeZone zone0) {
        DateTimeZone zone = DateTimeZone.forID(zone0.getID()); // just-in-case safety (is 'equals' here perfect?)
        return zone.isFixed() && zone.getOffset(0) != 0 ? Option.of("UTC") : zone2winName.getO(zone);
    }

    public static Option<DateTimeZone> getZoneByWinName(@Nullable String winName0) {
        if (winName0 == null) {
            return Option.empty();
        }

        String winName;
        Matcher m = timezoneWithOffsetPattern.matcher(winName0);
        if (m.matches()) {
            // TODO calculate offset based on sign (1), hours (2) and minutes (3)
            String localizedName = m.group(4);
            winName = localizedName2winName.getOrElse(localizedName, winName0);
        } else {
            winName = localizedName2winName.getOrElse(winName0, winName0);
        }
        return winName2zone.getO(winName);
    }
}
