package ru.yandex.calendar.frontend.ews;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.GenericBeanDao;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscriptionFields;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscriptionHelper;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.db.CalendarJdbcDaoSupport;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;

/**
 * @author akirakozov
 */
public class YtEwsSubscriptionDao extends CalendarJdbcDaoSupport {

    @Autowired
    private GenericBeanDao genericBeanDao;


    @RunWithRandomTest
    public ListF<Long> findSubscribedResourceIds() {
        String q = "SELECT resource_id FROM yt_ews_subscription WHERE resource_id IS NOT NULL";
        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public ListF<Long> findSubscribedUserUids() {
        String q = "SELECT uid FROM yt_ews_subscription WHERE uid IS NOT NULL";
        return getJdbcTemplate().queryForList(q, Long.class);
    }

    @RunWithRandomTest
    public Option<YtEwsSubscription> findSubscriptionByExchangeId(String subscriptionId) {
        String q = "SELECT * FROM yt_ews_subscription WHERE subscription_id = ?";
        return getJdbcTemplate().queryForOption(q, YtEwsSubscription.class, subscriptionId);
    }

    @RunWithRandomTest
    public Option<YtEwsSubscription> findUserSubscription(PassportUid uid) {
        String q = "SELECT * FROM yt_ews_subscription WHERE uid = ?";
        return getJdbcTemplate().queryForOption(q, YtEwsSubscription.class, uid);
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findUsersSubscriptions(ListF<PassportUid> uids) {
        return genericBeanDao.loadBeans(
                YtEwsSubscriptionHelper.INSTANCE, YtEwsSubscriptionFields.UID.column().inSet(uids));
    }

    @RunWithRandomTest
    public Option<YtEwsSubscription> findResourceSubscription(long resourceId) {
        String q = "SELECT * FROM yt_ews_subscription WHERE resource_id = ?";
        return getJdbcTemplate().queryForOption(q, YtEwsSubscription.class, resourceId);
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findResourcesSubscriptions(ListF<Long> resourceIds) {
        return genericBeanDao.loadBeans(
                YtEwsSubscriptionHelper.INSTANCE, YtEwsSubscriptionFields.RESOURCE_ID.column().inSet(resourceIds));
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findExpiredSubscripitons(Instant expiredTsBound) {
        String q =
                "SELECT * FROM yt_ews_subscription" +
                " WHERE last_ping_ts < ? OR last_ping_ts IS NULL";
        return getJdbcTemplate().queryForList(q, YtEwsSubscription.class, expiredTsBound);
    }

    @RunWithRandomTest(possible=EmptyResultDataAccessException.class)
    public void updateSubscriptionLastPingTs(String subscriptionId) {
        String q = "UPDATE yt_ews_subscription SET last_ping_ts = ? WHERE subscription_id = ?";
        getJdbcTemplate().updateRow(q, AuxDateTime.NOWTS(), subscriptionId);
    }

    @RunWithRandomTest
    public void removeUserSubscription(PassportUid uid) {
        String q = "DELETE FROM yt_ews_subscription WHERE uid = ?";
        getJdbcTemplate().update(q, uid);
    }

    @RunWithRandomTest
    public int removeDismissedUsersSubscriptions() {
        return getJdbcTemplate().update(""
                + "DELETE FROM yt_ews_subscription s USING settings_yt syt"
                + " WHERE syt.is_dismissed = TRUE and s.uid = syt.uid");
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findAllSubscriptions() {
        String q = "SELECT * FROM yt_ews_subscription";
        return getJdbcTemplate().queryForList(q, YtEwsSubscription.class);
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findAllPullSubscriptions() {
        return genericBeanDao.loadBeans(
                YtEwsSubscriptionHelper.INSTANCE,
                YtEwsSubscriptionFields.PULL_SUBSCRIPTION_ID.column().isNotNull());
    }

    @RunWithRandomTest
    public ListF<YtEwsSubscription> findSubscribedResources() {
        String q = "SELECT * FROM yt_ews_subscription WHERE resource_id IS NOT NULL";
        return getJdbcTemplate().queryForList(q, YtEwsSubscription.class);
    }

    public void updateYtEwsSubscription(YtEwsSubscription ytEwsSubscription) {
        genericBeanDao.updateBean(ytEwsSubscription);
    }

    public void saveYtEwsSubscription(YtEwsSubscription ytEwsSubscription) {
        genericBeanDao.insertBean(ytEwsSubscription);
    }

    @RunWithRandomTest
    public void deleteYtEwsSubscriptionsByResourceIds(ListF<Long> subscribedResourceIds) {
        SqlCondition condition = YtEwsSubscriptionFields.RESOURCE_ID.column().inSet(subscribedResourceIds);
        genericBeanDao.deleteBeans(YtEwsSubscriptionHelper.INSTANCE, condition);
    }

}
