package ru.yandex.calendar.frontend.ews.compare;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemTypeType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.val;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeEmailManager;
import ru.yandex.calendar.frontend.ews.YtEwsSubscriptionDao;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.DecisionWithSource;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;

/**
 * Compares timetables between calendar and exchange
 * XXX support non-subscribed subjects, see ExchangeZscript#getExchangeEmailByEmail
 * @author ssytnik
 */
public class EwsComparator {
    private static final Logger logger = LoggerFactory.getLogger(EwsComparator.class);

    private static final Duration COMPARISON_EXTENSION = Duration.standardHours(2);

    @Value("${ews.domain}")
    String ewsDomain;

    @Autowired
    private YtEwsSubscriptionDao ytEwsSubscriptionDao;
    @Autowired
    private ExchangeEmailManager exchangeEmailManager;
    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private LayerRoutines layerRoutines;

    private boolean ignoreWhenPrefix;

    /**
     * Calls {@link #compareRandomSubjects(int, InstantInterval)} with interval [-1 day; +1 day]
     */
    public ListF<EwsCompareResult> compareRandomSubjects(int count) {
        return compareRandomSubjects(count, getDefaultInterval());
    }

    /**
     * Compares timetables for random users using existing subscriptions
     */
    public ListF<EwsCompareResult> compareRandomSubjects(int count, final InstantInterval interval) {
        return Random2.R.shuffle(ytEwsSubscriptionDao.findAllSubscriptions()).take(count).map(compareF(interval));
    }

    public ListF<EwsCompareResult> compareAllResources() {
        return ytEwsSubscriptionDao.findSubscribedResources().map(compareF(getDefaultInterval()));
    }

    public ListF<EwsCompareResult> compareAllResourcesSafe() {
        return ytEwsSubscriptionDao.findSubscribedResources().filterMap(compareSafeOF(getDefaultInterval()));
    }

    public EwsCompareResultBrief compareEwsLoginsToday() {
        return compareEwsLogins(getIntervalDays(1));
    }

    private int getDiffsWithKnownReason(EwsCompareResult compareResult) {
        return getNotStoppedEventsInExchangeCount(compareResult);
    }

    public EwsCompareResultBrief compareEwsLogins(InstantInterval interval) {
        return settingsRoutines.getAllEwserEmails().map(email -> compare(email, interval))
                .map(r -> r.toBrief(this::getDiffsWithKnownReason, this::getCalendarUniqueEventsWithEwsSyncCount,
                        this::getCommonEventsWithEwsSyncCount))
                .reduceRight(EwsCompareResultBrief::plus);
    }

    public EwsCompareResult compare(Email email) {
        return compare(email, getDefaultInterval());
    }

    public static InstantInterval getDefaultInterval() {
        return getIntervalDays(3);
    }

    public static InstantInterval getIntervalDays(int days) {
        DateTime moscowToday = new DateTime(MoscowTime.TZ);
        return new InstantInterval(
                moscowToday.minusDays(days/2).withTimeAtStartOfDay(),
                moscowToday.plusDays(days - days/2).withTimeAtStartOfDay());
    }

    public EwsCompareResult compare(Email email, InstantInterval interval) {
        Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(email);
        Email emailForSubject = ExchangeEmailManager.getEmailByExchangeEmail(email, true);
        UidOrResourceId subjectId = userManager.getSubjectIdByEmail(emailForSubject).get();
        return compare(exchangeEmail, subjectId, interval);
    }

    public EwsCompareResult compare(YtEwsSubscription subscription, InstantInterval interval) {
        Email exchangeEmail = exchangeEmailManager.getExchangeEmailBySubscription(subscription);
        UidOrResourceId subjectId = EwsUtils.getSubjectId(subscription);
        return compare(exchangeEmail, subjectId, interval);
    }

    private EwsCompareResult compare(Email exchangeEmail, UidOrResourceId subjectId, InstantInterval interval) {
        return compare(exchangeEmail, subjectId, interval, true);
    }

    public EwsCompareResult compareWithoutFilterForTests(
            Email exchangeEmail, UidOrResourceId subjectId, InstantInterval interval)
    {
        return compare(exchangeEmail, subjectId, interval, false);
    }

    public EwsCompareResultBrief compareEwsLoginsWithoutDecisions(InstantInterval interval) {
        return settingsRoutines.getAllEwserEmails().map(email -> {
                    Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(email);
                    Email emailForSubject = ExchangeEmailManager.getEmailByExchangeEmail(email, true);
                    UidOrResourceId subjectId = userManager.getSubjectIdByEmail(emailForSubject).get();
                    return compare(exchangeEmail, subjectId, interval, true, true);
                })
                .map(r -> r.toBrief(this::getDiffsWithKnownReason, this::getCalendarUniqueEventsWithEwsSyncCount,
                        this::getCommonEventsWithEwsSyncCount))
                .reduceRight(EwsCompareResultBrief::plus);
    }

    private EwsCompareResult compare(
            Email exchangeEmail, UidOrResourceId subjectId, InstantInterval interval,
            boolean filterOutInsignificant)
    {
        return compare(exchangeEmail, subjectId, interval, filterOutInsignificant, false);
    }

    private SetF<EventKey> getKeysFromExtendedInterval(
            InstantInterval interval, Function<InstantInterval, CollectionF<EventKey>> resolver)
    {
        InstantInterval extendedInterval = interval
                .withStart(interval.getStart().minus(COMPARISON_EXTENSION))
                .withEnd(interval.getEnd().plus(COMPARISON_EXTENSION));

        return resolver.apply(extendedInterval).unique().filter(key -> interval.contains(key.getInterval()));
    }

    private EwsCompareResult compare(
            Email exchangeEmail, UidOrResourceId subjectId, InstantInterval interval,
            boolean filterOutInsignificant, boolean ignoreDecisionDesync)
    {
        logger.info("Running compare for " + subjectId + " (" + exchangeEmail + ") at interval " + interval);

        boolean isEwser = subjectId.getUidO().map(settingsRoutines::getIsEwser).getOrElse(false);

        SetF<EventKey> exchangeKeys = getKeysFromExtendedInterval(interval, i -> getExchangeKeys(exchangeEmail, i));
        SetF<EventKey> calendarKeys = getKeysFromExtendedInterval(interval, i -> getCalendarKeys(subjectId, i));

        if (ignoreDecisionDesync) {
            exchangeKeys = exchangeKeys.map(EventKey::withoutAttendeesDecisions).unique();
            calendarKeys = calendarKeys.map(EventKey::withoutAttendeesDecisions).unique();
        }

        SetF<EventKey> commonKeys = exchangeKeys.intersect(calendarKeys);
        SetF<EventKey> exchangeUniqueKeys = exchangeKeys.minus(commonKeys);
        SetF<EventKey> calendarUniqueKeys = calendarKeys.minus(commonKeys);

        EwsCompareResult result = new EwsCompareResult(
                exchangeEmail, subjectId, interval,
                commonKeys, exchangeUniqueKeys, calendarUniqueKeys);

        if (filterOutInsignificant) {
            result = filterOutInsignificantDesync(result, isEwser);
        }

        return result;
    }

    private boolean attendeesDecisionsDiffer(EventKey exchangeKey, EventKey calendarKey) {
        for (Tuple2<Email, DecisionWithSource> attendee : calendarKey.attendeesDecisions.entries()) {
            Email email = attendee._1;
            Decision calendarDecision = attendee._2.getDecision();

            boolean decisionIsSetInExchange =
                    attendee._2.getSource().map(ActionSource::isFromExchangeOrMail).getOrElse(false);

            Option<Decision> exchangeDecision =
                    exchangeKey.attendeesDecisions.getO(email).map(DecisionWithSource::getDecision);

            if (!(exchangeDecision.isSome(calendarDecision) || decisionIsSetInExchange)) {
                return true;
            }
        }
        return false;
    }

    private String suppressFwPrefix(String eventName) {
        if (eventName.startsWith("FW: ")) {
            eventName = eventName.substring(4);
        }
        return eventName;
    }

    private boolean noSignificantDifference(
            EventKey exchangeKey, EventKey calendarKey, Email subject, boolean isEwser)
    {
        boolean isEwsOrganizerInCalendar = calendarKey.ewsOrganizerO.isSome(subject);
        boolean isOrganizerInExchange = exchangeKey.ewsOrganizerO.isSome(subject);

        boolean sameName = exchangeKey.name.equals(calendarKey.name)
                || suppressFwPrefix(exchangeKey.name).equals(calendarKey.name);

        boolean sameDecisionAndAvailability = exchangeKey.subjectDecision.equals(calendarKey.subjectDecision)
                && exchangeKey.availability.equals(calendarKey.availability);

        boolean sameEwsExportStatus = isOrganizerInExchange == isEwsOrganizerInCalendar;

        boolean correctEwsDependingParams = !isEwser || (sameDecisionAndAvailability && sameEwsExportStatus);

        return exchangeKey.startTs.equals(calendarKey.startTs)
                && exchangeKey.endTs.equals(calendarKey.endTs)
                && sameName
                && correctEwsDependingParams
                && (exchangeKey.sequence <= calendarKey.sequence)
                && (!isEwsOrganizerInCalendar || !attendeesDecisionsDiffer(exchangeKey, calendarKey));
    }

    public EwsCompareResult filterOutInsignificantDesync(EwsCompareResult input, boolean isEwser) {
        SetF<EventKey> newCommonKeys = Cf.toHashSet(input.getCommonKeys());
        SetF<EventKey> newExchangeUniqueKeys = Cf.hashSet();
        SetF<EventKey> newCalendarUniqueKeys = Cf.toHashSet(input.getCalendarUniqueKeys());

        for (EventKey exchangeKey : input.getExchangeUniqueKeys()) {
            if (exchangeKey.externalIdO.isPresent()) {
                Option<EventKey> calendarKeyO =
                        newCalendarUniqueKeys.find(key -> (exchangeKey.externalIdO.equals(key.externalIdO)
                                && exchangeKey.startTs.equals(key.startTs)));
                if (calendarKeyO.isPresent()
                        && noSignificantDifference(exchangeKey, calendarKeyO.get(), input.getExchangeEmail(), isEwser))
                {
                    newCommonKeys.add(calendarKeyO.get());
                    newCalendarUniqueKeys.removeTs(calendarKeyO.get());
                } else {
                    newExchangeUniqueKeys.add(exchangeKey);
                }
            } else {
                newExchangeUniqueKeys.add(exchangeKey);
            }
        }

        newCommonKeys = Cf.toSet(newCommonKeys);
        newExchangeUniqueKeys = Cf.toSet(newExchangeUniqueKeys);
        newCalendarUniqueKeys = Cf.toSet(newCalendarUniqueKeys).filter(key -> key.subjectDecision != Decision.NO);
        return new EwsCompareResult(input.getExchangeEmail(), input.getSubjectId(), input.getInterval(),
                newCommonKeys, newExchangeUniqueKeys, newCalendarUniqueKeys);
    }

    public int getNotStoppedEventsInExchangeCount(EwsCompareResult compareResult) {
        return compareResult.getExchangeUniqueKeys()
                .count(key -> shouldBeStoppedInExchange(compareResult.getSubjectId(), key));
    }

    public int getCalendarUniqueEventsWithEwsSyncCount(EwsCompareResult compareResult) {
        return compareResult.getCalendarUniqueKeys().count(key -> key.ewsOrganizerO.isPresent());
    }

    public int getCommonEventsWithEwsSyncCount(EwsCompareResult compareResult) {
        return compareResult.getCommonKeys().count(key -> key.ewsOrganizerO.isPresent());
    }

    public boolean shouldBeStoppedInExchange(UidOrResourceId subjectId, EventKey key) {
        return key.externalIdO.filterMap(externalId ->
                eventRoutines.findMasterEventBySubjectIdAndExternalId(subjectId, externalId))
                .map(eventDbManager::getEventAndRepetitionByEvent).filterMap(EventAndRepetition.getRepetitionF())
                .filterMap(Repetition::getDueTs).map(dueTs -> dueTs.isBefore(Instant.now())).getOrElse(false);
    }

    private Function<YtEwsSubscription, EwsCompareResult> compareF(final InstantInterval interval) {
        return a -> compare(a, interval);
    }

    private Function<YtEwsSubscription, Option<EwsCompareResult>> compareSafeOF(final InstantInterval interval) {
        return a -> {
            try {
                return Option.of(compare(a, interval));
            } catch (Exception e) {
                logger.error(e, e);
                return Option.empty();
            }
        };
    }

    @Getter
    @AllArgsConstructor
    public static class EventKey extends DefaultObject {
        private static final DateTimeFormatter TS_FORMAT = DateTimeFormat.forPattern("HH:mm dd-MM-yyyy");

        private Instant startTs;
        private Instant endTs;
        private String name;
        private Option<String> externalIdO;
        private Option<Instant> recurrenceIdO;
        private MapF<Email, DecisionWithSource> attendeesDecisions;
        private Option<Email> ewsOrganizerO;
        private Decision subjectDecision;
        private Availability availability;
        private int sequence;
        private Optional<String> exchangeId;

        public static Function<EventKey, Instant> getStartTsF() {
            return a -> a.startTs;
        }

        public EventKey withoutAttendeesDecisions() {
            return new EventKey(startTs, endTs, name, externalIdO,
                    recurrenceIdO, Cf.map(), ewsOrganizerO, subjectDecision, availability, sequence, exchangeId);
        }

        public InstantInterval getInterval() {
            return new InstantInterval(startTs, endTs);
        }

        @Override
        public String toString() {
            String startStr = TS_FORMAT.print(startTs);
            String endStr = TS_FORMAT.print(endTs);
            String extIdStr = externalIdO.isPresent() ? externalIdO.get() : "<no ext. id>";
            String recurStr = recurrenceIdO.isPresent() ? "/" + recurrenceIdO.get() : "";
            String ewsStr = String.valueOf(ewsOrganizerO.isPresent());
            return "[" + name + " @ " + startStr + "-" + endStr + ", " + extIdStr + recurStr + " Ews: " + ewsStr + "]";
        }
    }


    private ListF<EventKey> getExchangeKeys(Email exchangeEmail, InstantInterval interval) {
        ListF<UnindexedFieldURIType> fields = Cf.list(
                UnindexedFieldURIType.CALENDAR_UID,
                UnindexedFieldURIType.ITEM_SUBJECT,
                UnindexedFieldURIType.CALENDAR_MY_RESPONSE_TYPE,
                UnindexedFieldURIType.CALENDAR_CALENDAR_ITEM_TYPE,
                UnindexedFieldURIType.CALENDAR_START,
                UnindexedFieldURIType.CALENDAR_END,
                UnindexedFieldURIType.CALENDAR_RECURRENCE_ID,
                UnindexedFieldURIType.CALENDAR_IS_CANCELLED,
                UnindexedFieldURIType.CALENDAR_IS_MEETING,
                UnindexedFieldURIType.CALENDAR_REQUIRED_ATTENDEES,
                UnindexedFieldURIType.CALENDAR_OPTIONAL_ATTENDEES,
                UnindexedFieldURIType.CALENDAR_RESOURCES,
                UnindexedFieldURIType.CALENDAR_APPOINTMENT_SEQUENCE_NUMBER,
                UnindexedFieldURIType.CALENDAR_LEGACY_FREE_BUSY_STATUS);

        ListF<String> extendedFields = Cf.list(EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID);

        ListF<String> exchangeIds = ewsProxyWrapper.findInstanceEventIds(exchangeEmail, interval);
        if (exchangeIds.isEmpty()) {  // CAL-8179
            exchangeIds = ewsProxyWrapper.findInstanceEventIds(
                    new Email(exchangeEmail.getLocalPart() + "@" + ewsDomain), interval);
        }

        ListF<CalendarItemType> items = ewsProxyWrapper.getEvents(exchangeIds, fields, extendedFields);
        return items.filterMap(getEventKeyByItemOF(exchangeEmail));
    }

    // copy-paste from ExchangeEventDataConverter + changes
    private Option<EventKey> getEventKeyByItemO(Email exchangeEmail, CalendarItemType calItem) {
        if (EwsUtils.isCancelled(calItem)) {
            return Option.empty();
        }

        // Exchange can set recurrenceId for OCCURRENCE and EXCEPTION. We mind it for EXCEPTION only.
        final Option<Instant> recurrenceIdO;
        if (calItem.getCalendarItemType() == CalendarItemTypeType.EXCEPTION) {
            recurrenceIdO = EwsUtils.toInstantO(calItem.getRecurrenceId());
        } else if (calItem.getCalendarItemType() == CalendarItemTypeType.SINGLE) {
            recurrenceIdO = EwsUtils.convertExtendedProperties(calItem.getExtendedProperty()).getRecurrenceId();
        } else {
            recurrenceIdO = Option.empty();
        }

        MapF<Email, DecisionWithSource> attendeesDecisions = ExchangeEventDataConverter.getAllAttendees(calItem)
                .toTuple2List(
                        a -> exchangeEmailManager.resolveDomainIfNeeded(a.getMailbox()),
                        a -> EwsUtils.getDecisionSafe(a.getResponseType()))
                .plus1(Tuple2.tuple(exchangeEmailManager.resolveDomainIfNeeded(exchangeEmail),
                        EwsUtils.getDecisionSafe(calItem.getMyResponseType())))
                .filterBy1(Option::isPresent).map1(emailO -> unifyYtEmail(emailO.get())).
                        map2(DecisionWithSource::withoutSource).toMap();

        return Option.of(new EventKey(
                EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getStart()),
                EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getEnd()),
                prepareKeyName(calItem.getSubject()),
                StringUtils.notEmptyO(calItem.getUID()),
                recurrenceIdO,
                attendeesDecisions,
                Option.when(EwsUtils.isOrganizedOrNonMeeting(calItem), exchangeEmail),
                Decision.findByExchangeReponseType(calItem.getMyResponseType()).getOrElse(Decision.UNDECIDED),
                ExchangeEventDataConverter.getAvailability(calItem.getLegacyFreeBusyStatus())
                        .getOrElse(Availability.MAYBE),
                Option.ofNullable(calItem.getAppointmentSequenceNumber()).getOrElse(0),
                Optional.of(calItem.getItemId().getId())
        ));
    }

    private Function<CalendarItemType, Option<EventKey>> getEventKeyByItemOF(Email exchangeEmail) {
        return a -> getEventKeyByItemO(exchangeEmail, a);
    }


    private SetF<EventKey> getCalendarKeys(UidOrResourceId subjectId, InstantInterval interval) {
        Option<PassportUid> uidO = Option.empty();
        EventGetProps egp = EventGetProps.any();
        ActionSource actionSource = ActionSource.UNKNOWN;
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(interval);

        ListF<EventInstanceInfo> infos;
        if (subjectId.isUser()) {
            PassportUid uid = subjectId.getUid();

            Option<Long> defaultLayerId = layerRoutines.getDefaultLayerId(uid);

            ListF<LayerUserWithRelations> layerUsers = layerRoutines
                    .getLayerUsersWithRelationsByUid(uid, Option.empty());

            ListF<Long> absenceLayerIds = layerUsers
                    .filterMap(lu -> Option.when(lu.getLayer().getType().isAbsence(), lu.getLayerId()));

            ListF<Long> createdLayerIds = layerUsers
                    .filterMap(lu -> Option.when(lu.layerCreatorIs(uid), lu.getLayerId()));

            infos = eventRoutines.getSortedInstancesOnLayer(
                    uidO, egp, LayerIdPredicate.list(createdLayerIds, false), limits, actionSource)
                    .filter(info -> info.getEventParticipants().getEventUser(uid).isPresent()
                            && (defaultLayerId.equals(info.getLayerId())
                            || info.getLayerId().exists(absenceLayerIds::containsTs)
                            || info.getEventParticipants().getEventUser(uid).get().isAttendeeOrOrganizerOrSubscriber())
                            && info.getEvent().getCreationSource() != ActionSource.WEB_ICS);

        } else if (subjectId.isResource()) {
            ListF<Long> resourceIds = Cf.list(subjectId.getResourceId());
            infos = eventRoutines.getSortedInstancesOnResource(uidO, egp, resourceIds, limits, actionSource);

        } else {
            throw new IllegalStateException("Unknown type of subjectId: " + subjectId);
        }
        MapF<Email, Email> exchangeEmailMap = exchangeEmailManager.getExchangeOrAsIsEmailsSafeByEmails(
                infos.filterMap(e -> e.getEventWithRelations().getOrganizerEmailIfMeeting())).toMap();

        return infos.zipWith(i ->
                i.getEventWithRelations().getOrganizerEmailIfMeeting().flatMapO(exchangeEmailMap::getO))
                .filterMap(t -> getEventKeyByInstanceInfoO(subjectId, t._1, t._2)).unique();
    }

    // TODO note shared layers (absent in exchange)
    private MapF<Email, DecisionWithSource> getNotMeetingAttendeesDecision(UidOrResourceId subjectId, Participants participants) {
        if (participants.isNotMeetingStrict()) {
            val exchangeEmail = subjectId.isUser()
                    ? userManager.getEmailByUid(subjectId.getUid())
                    : Optional.of(resourceRoutines.getExchangeEmailById(subjectId.getResourceId()));

            if (exchangeEmail.isPresent()) {
                return Cf.map(exchangeEmail.get(), DecisionWithSource.withoutSource(Decision.YES));
            }
        }
        return Cf.map();
    }

    private Option<EventKey> getEventKeyByInstanceInfoO(
            UidOrResourceId subjectId, EventInstanceInfo info, Option<Email> organizerEmailO)
    {
        if (info.getEvent().getType() != EventType.USER) {
            return Option.empty();
        }

        Participants participants = info.getEventWithRelations().getParticipants();

        MapF<Email, DecisionWithSource> attendeesDecisions = participants.isMeeting()
                ? participants.getAllAttendees().toMap(
                        p -> unifyYtEmail(exchangeEmailManager.resolveDomainIfNeeded(p.getEmail()).get()),
                        DecisionWithSource::new)
                : getNotMeetingAttendeesDecision(subjectId, participants);

        Decision subjectDecision = Decision.NO;
        Availability subjectAvailability = Availability.AVAILABLE;

        if (subjectId.isResource()) {
            subjectDecision = Decision.YES;
            subjectAvailability = Availability.BUSY;
        } else {
            Option<EventUserWithRelations> eventUserWithRelations =
                    info.getEventParticipants().getEventUser(subjectId.getUid());
            if (eventUserWithRelations.isPresent()) {
                EventUser eventUser = eventUserWithRelations.get().getEventUser();
                subjectDecision = eventUser.getDecision();
                subjectAvailability = eventUser.getAvailability();
            }
        }

        return Option.of(new EventKey(
                info.getInterval().getStart(),
                info.getInterval().getEnd(),
                prepareKeyName(info.getEvent().getName()),
                StringUtils.notEmptyO(info.getEventWithRelations().getMainEvent().getExternalId()),
                info.getEvent().getRecurrenceId(), attendeesDecisions,
                info.getMainEvent().getIsExportedWithEws().getOrElse(false) ? organizerEmailO : Option.empty(),
                subjectDecision,
                subjectAvailability,
                info.getEvent().getSequence(),
                Optional.empty()
        ));
    }

    private String prepareKeyName(String eventName) {
        String notEmptyEventName = StringUtils.defaultIfEmpty(eventName, UStringLiteral.NO_NAME);
        return ignoreWhenPrefix ? discardWhenPrefix(notEmptyEventName) : notEmptyEventName;
    }

    private static final String genitiveMonths =
        "(?:" + UStringLiteral.GENITIVE_MONTHS.mkString("|") + ")";
    private static final Pattern eventNameWithWhenPrefix = Pattern.compile(
        "^(?:Сегодня|Завтра|(?:\\d{2} " + genitiveMonths + ")) в \\d{2}:\\d{2} \"(.*)\"$"
    );

    protected static String discardWhenPrefix(String s) {
        Matcher m = eventNameWithWhenPrefix.matcher(s);
        return m.matches() ? m.group(1) : s;
    }

    private Email unifyYtEmail(Email email) {
        if (email.getDomain().getDomain().startsWith("yandex-team")) {
            return new Email(email.getLocalPart() + "@yandex-team.ru");
        } else {
            return email;
        }
    }

    @Required
    @Value("${ews-comparator.ignore.when.prefix:-false}")
    public void setIgnoreWhenPrefix(boolean ignoreWhenPrefix) {
        this.ignoreWhenPrefix = ignoreWhenPrefix;
    }

}
