package ru.yandex.calendar.frontend.ews.exp;

import javax.xml.datatype.XMLGregorianCalendar;

import com.microsoft.schemas.exchange.services._2006.types.AbsoluteMonthlyRecurrencePatternType;
import com.microsoft.schemas.exchange.services._2006.types.AbsoluteYearlyRecurrencePatternType;
import com.microsoft.schemas.exchange.services._2006.types.AttendeeType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.DailyRecurrencePatternType;
import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekIndexType;
import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekType;
import com.microsoft.schemas.exchange.services._2006.types.EndDateRecurrenceRangeType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import com.microsoft.schemas.exchange.services._2006.types.LegacyFreeBusyType;
import com.microsoft.schemas.exchange.services._2006.types.MonthNamesType;
import com.microsoft.schemas.exchange.services._2006.types.NoEndRecurrenceRangeType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfAttendeesType;
import com.microsoft.schemas.exchange.services._2006.types.RecurrenceType;
import com.microsoft.schemas.exchange.services._2006.types.RelativeMonthlyRecurrencePatternType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import com.microsoft.schemas.exchange.services._2006.types.WeeklyRecurrencePatternType;
import lombok.val;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.frontend.ews.EwsException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.WindowsTimeZones;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.EventChangesInfoForExchange;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class EventToCalendarItemConverter {
    private static final Logger logger = LoggerFactory.getLogger(EventToCalendarItemConverter.class);

    @Value("${ews.domain}")
    private String ewsDomain;

    @Autowired
    private EnvironmentType environmentType;
    @Autowired
    private ResourceRoutines resourceRoutines;

    public CalendarItemType convertToCalendarItem(
            EventWithRelations eventWithRelations, RepetitionInstanceInfo repetitionInfo)
    {
        CalendarItemType calendarItem = new CalendarItemType();
        DateTimeZone tz = eventWithRelations.getTimezone();
        Event event = eventWithRelations.getEvent();

        String timezoneName = getTimezoneName(tz, false);
        calendarItem.setStartTimeZone(EwsUtils.createTimezoneDefinitionType(timezoneName));
        calendarItem.setEndTimeZone(EwsUtils.createTimezoneDefinitionType(timezoneName));

        // date time data
        calendarItem.setStart(event.getIsAllDay()
                ? EwsUtils.localDateTimeToXmlGregorianCalendar(new LocalDateTime(event.getStartTs(), tz))
                : EwsUtils.instantToXMLGregorianCalendar(event.getStartTs(), tz));
        calendarItem.setEnd(event.getIsAllDay()
                ? EwsUtils.localDateTimeToXmlGregorianCalendar(new LocalDateTime(event.getEndTs(), tz))
                : EwsUtils.instantToXMLGregorianCalendar(event.getEndTs(), tz));

        if (event.isFieldSet(EventFields.IS_ALL_DAY)) {
            calendarItem.setIsAllDayEvent(event.getIsAllDay());
        }
        // info data
        calendarItem.setSubject(event.getName());
        if (event.isFieldSet(EventFields.LOCATION)) {
            calendarItem.setLocation(constructLocationText(event.getLocation(), eventWithRelations));
        }
        if (event.isFieldSet(EventFields.DESCRIPTION)) {
            calendarItem.setBody(EwsUtils.createTextBody(event.getDescription()));
        }
        if(event.isFieldSet(EventFields.CONFERENCE_URL) && event.getConferenceUrl().isPresent()){
            calendarItem.setMeetingWorkspaceUrl(event.getConferenceUrl().get());
        }

        // synchronization data
        calendarItem.setUID(eventWithRelations.getExternalId());

        Participants participants = eventWithRelations.getParticipants();
        if (participants.isMeeting()) {
            calendarItem.setRequiredAttendees(convertParticipantsList(participants.getAllUserAttendeesButNotOrganizer()));
            calendarItem.setResources(convertParticipantsList(participants.getResourceParticipants()));
        }
        // There are problems with setting exchange organizer:
        // http://social.technet.microsoft.com/Forums/en-US/exchangesvrdevelopment/thread/3183b77b-b190-4e42-8d12-2f1e47317369
        //calendarItem.setOrganizer(convertOrganizer(participants.getOrganizer()));
        if (repetitionInfo.getRepetition().isPresent()) {
            calendarItem.setRecurrence(convertRecurrence(event, repetitionInfo, tz));
        }
        //
        calendarItem.setReminderIsSet(false);
        calendarItem.setLegacyFreeBusyStatus(LegacyFreeBusyType.BUSY);
        calendarItem.getExtendedProperty().add(EwsUtils.createSourceExtendedProperty());
        if (participants.isMeeting()) {
            calendarItem.getExtendedProperty().add(
                    EwsUtils.createOrganizerExtendedProperty(participants.getOrganizer().getEmail()));
        }
        Option<Instant> recurrenceId = event.getFieldValueO(EventFields.RECURRENCE_ID).filterNotNull();
        if (recurrenceId.isPresent()) {
            calendarItem.getExtendedProperty().add(EwsUtils.createRecurrenceIdExtendedProperty(recurrenceId.get()));
        }
        return calendarItem;
    }


    // update

    // TODO ssytnik: check: can simplify using single CalendarItemType

    /**
     * @param updatedEvent event with relations to be updated (relations may be already updated)
     * @param updatedRepetitionInfo repetition object for the event
     * @param eventChangesInfo description of changes in event
     * @return information for Exchange
     */
    public ListF<ItemChangeDescriptionType> convertToChangeDescriptions(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetitionInfo,
            EventChangesInfoForExchange eventChangesInfo)
    {
        DateTimeZone tz = updatedEvent.getTimezone();

        ListF<ItemChangeDescriptionType> changes = Cf.arrayList();
        final Event eventChanges = eventChangesInfo.getEventChanges();

        Event event = updatedEvent.getEvent();

        // date time data
        if (eventChanges.isFieldSet(EventFields.START_TS)) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.setStart(updatedEvent.getEvent().getIsAllDay()
                    ? EwsUtils.localDateTimeToXmlGregorianCalendar(new LocalDateTime(updatedEvent.getStartTs(), tz))
                    : EwsUtils.instantToXMLGregorianCalendar(eventChanges.getStartTs(), tz));
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.CALENDAR_START));
        }
        if (eventChanges.isFieldSet(EventFields.END_TS)) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.setEnd(updatedEvent.getEvent().getIsAllDay()
                    ? EwsUtils.localDateTimeToXmlGregorianCalendar(new LocalDateTime(updatedEvent.getEndTs(), tz))
                    : EwsUtils.instantToXMLGregorianCalendar(eventChanges.getEndTs(), tz));
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.CALENDAR_END));
        }
        if (eventChanges.isFieldSet(EventFields.IS_ALL_DAY)) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.setIsAllDayEvent(eventChanges.getIsAllDay());
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.CALENDAR_IS_ALL_DAY_EVENT));
            changes.addAll(convertToChangeDescriptions(tz, false)); // CAL-6152
        }

        // data info
        if (eventChanges.isFieldSet(EventFields.NAME)) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.setSubject(eventChanges.getName());
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.ITEM_SUBJECT));
        }
        if (eventChanges.isFieldSet(EventFields.LOCATION) || eventChangesInfo.isParticipantsChanged()) {   // CAL-6280 & CAL-7947 & CAL-8136
            CalendarItemType calendarItem = new CalendarItemType();
            String eventLocation =
                    eventChanges.isFieldSet(EventFields.LOCATION) ? eventChanges.getLocation() : event.getLocation();
            calendarItem.setLocation(constructLocationText(eventLocation, updatedEvent));
            changes.add(EwsUtils.createSetItemField(calendarItem, UnindexedFieldURIType.CALENDAR_LOCATION));
        }
        if (eventChanges.isFieldSet(EventFields.DESCRIPTION)) {
            CalendarItemType calendarItem = new CalendarItemType();
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.ITEM_BODY));
            calendarItem.setBody(EwsUtils.createTextBody(eventChanges.getDescription()));
        }
        if (eventChanges.isFieldSet(EventFields.CONFERENCE_URL)) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.setMeetingWorkspaceUrl(eventChanges.getConferenceUrl().orElse(""));
            changes.add(EwsUtils.createSetItemField(
                    calendarItem, UnindexedFieldURIType.CALENDAR_MEETING_WORKSPACE_URL));
        }

        // attendees
        if (eventChangesInfo.isParticipantsChanged()) {
            Participants participants = updatedEvent.getParticipants();
            if (participants.isMeeting()) {
                changes.add(setParticipantFieldOrDeleteIfEmpty(participants.getAllUserAttendeesButNotOrganizer(),
                        UnindexedFieldURIType.CALENDAR_REQUIRED_ATTENDEES));

                changes.add(setParticipantFieldOrDeleteIfEmpty(participants.getResourceParticipants(),
                        UnindexedFieldURIType.CALENDAR_RESOURCES));

                CalendarItemType calendarItem = new CalendarItemType();
                calendarItem.getExtendedProperty().add(
                        EwsUtils.createOrganizerExtendedProperty(participants.getOrganizer().getEmail()));
                changes.add(EwsUtils.createSetItemField(calendarItem, EwsUtils.EXTENDED_PROPERTY_ORGANIZER));

            } else {
                changes.add(EwsUtils.createDeleteItemField(UnindexedFieldURIType.CALENDAR_REQUIRED_ATTENDEES));
                changes.add(EwsUtils.createDeleteItemField(UnindexedFieldURIType.CALENDAR_RESOURCES));
            }
            // Optional attendees are not yet supported. On import they are merged with requiredAttendees/resources
            changes.add(EwsUtils.createDeleteItemField(UnindexedFieldURIType.CALENDAR_OPTIONAL_ATTENDEES));
        }
        // repetition. XXX ssytnik: something must be wrong here, see https://jira.yandex-team.ru/browse/CAL-4189
        boolean isRecurrenceId = event.getRecurrenceId().isPresent();
        boolean hasRepetitionChanges = eventChangesInfo.getRepetitionChanges().isNotEmpty();
        if (!isRecurrenceId && hasRepetitionChanges) {
            boolean isWithRepetition = updatedRepetitionInfo.getRepetition().isPresent();
            logger.info(
                    "event is recurrenceId? " + isRecurrenceId + " " +
                    "hasRepetitionChanges? " + hasRepetitionChanges + " " +
                    "is with repetition? " + isWithRepetition);
            if (isWithRepetition) {
                CalendarItemType calendarItem = new CalendarItemType();
                calendarItem.setRecurrence(convertRecurrence(event, updatedRepetitionInfo, tz));
                changes.add(EwsUtils.createSetItemField(
                        calendarItem, UnindexedFieldURIType.CALENDAR_RECURRENCE));
            } else {
                 changes.add(EwsUtils.createDeleteItemField(
                         UnindexedFieldURIType.CALENDAR_RECURRENCE));
            }
        }
        Option<Instant> recurrenceId = eventChanges.getFieldValueO(EventFields.RECURRENCE_ID).filterNotNull();
        if (recurrenceId.isPresent()) {
            CalendarItemType calendarItem = new CalendarItemType();
            calendarItem.getExtendedProperty().add(EwsUtils.createRecurrenceIdExtendedProperty(recurrenceId.get()));
            changes.add(EwsUtils.createSetItemField(calendarItem, EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID));
        }

        return changes;
    }

    private ItemChangeDescriptionType setParticipantFieldOrDeleteIfEmpty(
            ListF<? extends ParticipantInfo> participants, UnindexedFieldURIType field)
    {
        if (participants.isEmpty()) {
            return EwsUtils.createDeleteItemField(field);
        }

        CalendarItemType calendarItem = new CalendarItemType();

        Function1V<NonEmptyArrayOfAttendeesType> setter;

        switch (field) {
            case CALENDAR_RESOURCES:
                setter = calendarItem::setResources;
                break;
            case CALENDAR_REQUIRED_ATTENDEES:
                setter = calendarItem::setRequiredAttendees;
                break;
            case CALENDAR_OPTIONAL_ATTENDEES:
                setter = calendarItem::setOptionalAttendees;
                break;
            default:
                throw new IllegalArgumentException("Unsupported attendee field: " + field);
        }

        setter.apply(convertParticipantsList(participants));
        return EwsUtils.createSetItemField(calendarItem, field);
    }

    public ListF<ItemChangeDescriptionType> convertToChangeDescriptions(DateTimeZone tz0, boolean isSafely) {
        String timezoneName = getTimezoneName(tz0, isSafely);

        CalendarItemType startTimezoneCalendarItem = new CalendarItemType();
        startTimezoneCalendarItem.setStartTimeZone(EwsUtils.createTimezoneDefinitionType(timezoneName));

        CalendarItemType endTimezoneCalendarItem = new CalendarItemType();
        endTimezoneCalendarItem.setEndTimeZone(EwsUtils.createTimezoneDefinitionType(timezoneName));

        return Cf.list(
                EwsUtils.createSetItemField(startTimezoneCalendarItem, UnindexedFieldURIType.CALENDAR_START_TIME_ZONE),
                EwsUtils.createSetItemField(endTimezoneCalendarItem, UnindexedFieldURIType.CALENDAR_END_TIME_ZONE));
    }

    /**
     * @return actual timezone after apply (may differ from <code>tz</code> when <code>isSafely</code> = false)
     */
    public String getTimezoneName(DateTimeZone tz0, boolean isSafely) {
        String defaultTzName = "UTC";
        Option<String> tzName = WindowsTimeZones.getWinNameByZone(tz0);

        if (tzName.isPresent()) {
            logger.info("Using timezone: " + tz0);
        } else {
            if (isSafely) {
                logger.warn("Could not convert timezone " + tz0 + ", got " + defaultTzName);
            } else {
                throw new EwsException("unsupported timezone for export: " + tz0);
            }
        }
        return tzName.getOrElse(defaultTzName);
    }

    private NonEmptyArrayOfAttendeesType convertParticipantsList(ListF<? extends ParticipantInfo> participants) {
        if (participants.isEmpty()) {
            return null;
        }

        NonEmptyArrayOfAttendeesType attendeesArray = new NonEmptyArrayOfAttendeesType();
        boolean devOrTest = Cf.list(EnvironmentType.DEVELOPMENT, EnvironmentType.TESTS).containsTs(environmentType);
        for (ParticipantInfo attendee : participants) {
            // TODO: set decision?
            AttendeeType attendeeType = new AttendeeType();
            Email email = attendee.getEmail();

            if (devOrTest && email.getDomain().sameAs("yandex-team.ru")) {
                boolean isResource = attendee instanceof ResourceParticipantInfo;
                boolean isEwsUser = attendee.getSettingsIfYandexUser().exists(s -> s.getYt().exists(SettingsYt::getIsEwser));
                if (isResource || isEwsUser) {
                    email = new Email(attendee.getEmail().getLocalPart() + "@" + ewsDomain);
                }
            }

            attendeeType.setMailbox(EwsUtils.createEmailAddressType(email));
            attendeesArray.getAttendee().add(attendeeType);
        }
        return attendeesArray;
    }

    private RecurrenceType convertRecurrence(Event event, RepetitionInstanceInfo repetitionInfo, DateTimeZone tz) {
        RecurrenceType recurrence = new RecurrenceType();

        Repetition repetition = repetitionInfo.getRepetition().get();
        DateTime startDateTime = new DateTime(event.getStartTs(), tz);
        RegularRepetitionRule regularRepetitionRule = RegularRepetitionRule.find(repetition);
        final Integer rEach = RepetitionUtils.calcREach(repetition);

        switch (regularRepetitionRule) {
        case DAILY:
            DailyRecurrencePatternType dailyRecurrence = new DailyRecurrencePatternType();
            dailyRecurrence.setInterval(rEach);
            recurrence.setDailyRecurrence(dailyRecurrence);
            break;
        case WEEKLY:
            WeeklyRecurrencePatternType weeklyRecurrence = new WeeklyRecurrencePatternType();
            String str = repetition.getRWeeklyDays().getOrElse("");
            ListF<String> weekDays = Cf.x(StringUtils.split(str, ","));
            ListF<DayOfWeekType> daysOfWeek = Cf.x(weeklyRecurrence.getDaysOfWeek());
            for (String weekDay : weekDays) {
                daysOfWeek.add(WeekdayConv.calsToExchange(weekDay));
            }
            weeklyRecurrence.setInterval(rEach);
            recurrence.setWeeklyRecurrence(weeklyRecurrence);
            break;
        case MONTHLY_NUMBER:
            AbsoluteMonthlyRecurrencePatternType absMonthlyRecurence = new AbsoluteMonthlyRecurrencePatternType();
            absMonthlyRecurence.setInterval(rEach);
            absMonthlyRecurence.setDayOfMonth(startDateTime.getDayOfMonth());
            recurrence.setAbsoluteMonthlyRecurrence(absMonthlyRecurence);
            break;
        case MONTHLY_DAY_WEEKNO:
            RelativeMonthlyRecurrencePatternType relMonthlyRecurrence = new RelativeMonthlyRecurrencePatternType();
            relMonthlyRecurrence.setInterval(rEach);
            DayOfWeekType dayOfWeek = WeekdayConv.jodaToExchange(startDateTime.getDayOfWeek());
            relMonthlyRecurrence.setDaysOfWeek(dayOfWeek);
            if (repetition.getRMonthlyLastweek().getOrNull()) {
                relMonthlyRecurrence.setDayOfWeekIndex(DayOfWeekIndexType.LAST);
            } else {
                int weekOfMonth = AuxDateTime.getWeekOfMonth(startDateTime);
                DayOfWeekIndexType dayOfWeekIndex = EwsUtils.getDayOfWeekIndexTypeByWeekOfMonth(weekOfMonth);
                relMonthlyRecurrence.setDayOfWeekIndex(dayOfWeekIndex);
            }
            recurrence.setRelativeMonthlyRecurrence(relMonthlyRecurrence);
            break;
        case YEARLY:
            AbsoluteYearlyRecurrencePatternType yearlyRecurrence = new AbsoluteYearlyRecurrencePatternType();
            MonthNamesType monthName = EwsUtils.getMonthNamesTypeByJodaMonthOfYear(startDateTime.getMonthOfYear());
            yearlyRecurrence.setMonth(monthName);
            yearlyRecurrence.setDayOfMonth(startDateTime.getDayOfMonth());
            recurrence.setAbsoluteYearlyRecurrence(yearlyRecurrence);
            break;
        default:
            throw new IllegalStateException();
        }
        XMLGregorianCalendar startDateForExchange = EwsUtils.localDateToXMLGregorianCalendar(
                event.getStartTs().toDateTime(tz).toLocalDate(), tz);

        final Option<Instant> dueTs = repetition.getDueTs();
        if (dueTs.isPresent()) {
            XMLGregorianCalendar dueDate = convertDueDate(dueTs.get(), tz);
            EndDateRecurrenceRangeType endDateRecurrenceRange = new EndDateRecurrenceRangeType();
            endDateRecurrenceRange.setStartDate(startDateForExchange);
            endDateRecurrenceRange.setEndDate(dueDate);
            recurrence.setEndDateRecurrence(endDateRecurrenceRange);
        } else {
            NoEndRecurrenceRangeType noEndRecurrenceRange = new NoEndRecurrenceRangeType();
            noEndRecurrenceRange.setStartDate(startDateForExchange);
            recurrence.setNoEndRecurrence(noEndRecurrenceRange);
        }
        return recurrence;
    }

    public static XMLGregorianCalendar convertDueDate(Instant dueTs, DateTimeZone tz) {
        return EwsUtils.localDateToXMLGregorianCalendar(EventRoutines.convertDueTsToUntilDate(dueTs, tz), tz);
    }

    /**
     * @param eventLocation textual location
     * @param event changed event
     * @return list of meeting rooms if there is any one or textual location otherwise
     */
    private String constructLocationText(String eventLocation, EventWithRelations event) {
        val meetingRooms = resourceRoutines.locationStringWithResourceNames(
                event.getEvent().getCreatorUid(), event.getParticipants()).getOrElse("");
        return meetingRooms.isEmpty() ? eventLocation : meetingRooms;
    }
}
