package ru.yandex.calendar.frontend.ews.exp;

import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.microsoft.schemas.exchange.services._2006.types.BodyType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemCreateOrDeleteOperationType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import com.microsoft.schemas.exchange.services._2006.types.MessageDispositionType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.frontend.ews.EwsConflictingMeetingsNotCalculated;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.exp.async.YtEwsExportingEventDao;
import ru.yandex.calendar.frontend.ews.proxy.EwsActionLogData;
import ru.yandex.calendar.frontend.ews.proxy.EwsCallLogEventJson;
import ru.yandex.calendar.frontend.ews.proxy.EwsCallOperation;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.proxy.ExchangeIdLogData;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.worker.task.EwsExportEventResourceChangesOnUpdateTask;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventHelper;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.YtEwsExportingEvent;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventChangesInfoForExchange;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.MainEventInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.meeting.ExchangeMails;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

import static java.util.Collections.emptyList;
import static java.util.function.Predicate.not;

public class EwsExportRoutines {
    private static final Logger logger = LoggerFactory.getLogger(EwsExportRoutines.class);

    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private EventToCalendarItemConverter eventToCalendarItemConverter;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EwsMonitoring ewsMonitoring;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private YtEwsExportingEventDao ytEwsExportingEventDao;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;
    @Autowired
    private EventsLogger eventsLogger;
    @Autowired
    private EwsAliveHandler ewsAliveHandler;


    public void exportToExchangeIfNeeded(MainEventInfo main, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;

        EventWithRelations master = main.getMasterEventInfos().single().getEventWithRelations();

        RepetitionInstanceInfo repetition = main.getMasterEventInfos().single().getRepetitionInstanceInfo()
                .withoutPastRecurrencesAndExdates(actionInfo.getNow());

        exportToExchangeIfNeededOnCreate(master, repetition, Option.empty(), actionInfo);

        ListF<EventInfo> recurrences = main.getRecurrenceEventInfos()
                .filter(e -> e.getRepetitionInstanceInfo().goesOnAfter(actionInfo.getNow()));

        recurrences.forEach(recurrence -> {
            Option<InstantInterval> instance = RepetitionUtils.getInstanceIntervalStartingAt(
                    repetition.withoutRecurrences(), recurrence.getRecurrenceId().get());

            InstantInterval interval = recurrence.getRepetitionInstanceInfo().getEventInterval();

            Event fields = recurrence.getEvent().getFields(
                    EventFields.START_TS, EventFields.END_TS, EventFields.IS_ALL_DAY,
                    EventFields.NAME, EventFields.DESCRIPTION, EventFields.LOCATION, EventFields.URL);

            EventChangesInfo.EventChangesInfoFactory factory = new EventChangesInfo.EventChangesInfoFactory();

            factory.setEventChanges(EventHelper.INSTANCE.findChanges(master.getEvent(), fields));
            factory.setEventInstanceStartEndTsChanged(!instance.isSome(interval));

            ListF<ParticipantInfo> attendees = recurrence.getParticipants().getAllAttendeesButNotOrganizerSafe();

            ParticipantsData invitations = recurrence.getParticipants().getOrganizerSafe()
                    .map(org -> ParticipantsData.merge(org.toData(), attendees.map(ParticipantInfo::toData)))
                    .getOrElse(ParticipantsData::notMeeting);

            factory.setEventParticipantsChangesInfo(eventInvitationManager.participantsChanges(
                    Option.empty(), master.getParticipants(),
                    ParticipantsOrInvitationsData.participantsData(invitations)));

            OccurrenceId occurrence = new OccurrenceId(
                    recurrence.getMainEvent().getExternalId(),
                    instance.getOrElse(interval), recurrence.getRecurrenceId().get());

            exportToExchangeIfNeededOnUpdate(
                    recurrence.getEventWithRelations(), recurrence.getRepetitionInstanceInfo(),
                    Option.of(occurrence), factory.create(), Option.empty(), actionInfo);
        });
    }

    public void exportToExchangeIfNeededOnCreate(long eventId, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (!ensureActionSource(actionInfo.getActionSource())) return;

        EventWithRelations event = eventDbManager.getEventWithRelationsById(eventId);
        RepetitionInstanceInfo repetitionInfo = repetitionRoutines.getRepetitionInstanceInfo(event);

        exportToExchangeIfNeededOnCreate(event, repetitionInfo, actionInfo);
    }

    public void exportToExchangeIfNeededOnCreate(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo)
    {
        exportToExchangeIfNeededOnCreate(event, repetitionInfo, Option.empty(), actionInfo);
    }

    public void exportToExchangeIfNeededOnCreate(EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
                                                 Option<ExchangeMails> mails, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (!ensureActionSource(actionInfo.getActionSource())) return;

        if (exportToExchangeIfNeededOnCreateForOrganizer(event, repetitionInfo, mails.toOptional(), actionInfo)) {
            return;
        }

        val resources = mapToBriefInfo(event.getResourceParticipantsYaTeamAndSyncWithExchange());

        if (event.getRecurrenceId().isPresent()) {
            val masterEvent = eventDao.findMasterEventByMainId(event.getMainEventId()).singleO();

            val resourcesFromSyncedLayers = mapToBriefInfo(
                    resourceDao.findYaTeamResourceLayersThatSyncWithExchangeWithEvents(masterEvent.map(Event.getIdF())));
            val resourcesByAsyncWithExchange = StreamEx.of(resourcesFromSyncedLayers)
                        .append(resources)
                        .partitioningBy(ResourceParticipantBriefInfo::isAsyncWithExchange);

            submitAsyncExport(resourcesByAsyncWithExchange.get(true), event.getMainEvent(), actionInfo);

            val resourcesNotAsync = resourcesByAsyncWithExchange.get(false);
            if (resourcesNotAsync.isEmpty()) {
                return;
            }

            Validate.some(masterEvent, "master event is required to modify occurrence in ews");

            val eventChangesInfo = eventChangesFinder
                    .getEventChangesInfoForExchangeForNewRecurrence(masterEvent.get(), event.getEvent());

            val eventChangesByIds = StreamEx.of(resourcesNotAsync)
                    .groupingBy(ResourceParticipantBriefInfo::getEventId);
            val masterEventChanges = eventChangesByIds.getOrDefault(masterEvent.get().getId(), emptyList());
            val eventChanges = eventChangesByIds.getOrDefault(event.getId(), emptyList());

            val resourcesChanges = ResourceParticipantChangesInfo.byMasterAndNewRecurrence(masterEventChanges, eventChanges);

            Duration instanceDuration = EventRoutines.getInstantInterval(event.getEvent()).getDuration();
            InstantInterval instanceInterval = new InstantInterval(event.getRecurrenceId().get(), instanceDuration);
            OccurrenceId occurrenceId = new OccurrenceId(event.getExternalId(), instanceInterval, event.getRecurrenceId().get());

            createOrUpdateRecurrence(occurrenceId, eventChangesInfo, event, resourcesChanges, actionInfo);
        } else if (!resources.isEmpty()) {
            val resourcesByIsAsync = partitioningByAsyncWithExchange(resources);
            submitAsyncExport(resourcesByIsAsync.get(true), event.getMainEvent(), actionInfo);
            createMeetings(resourcesByIsAsync.get(false), event, repetitionInfo, actionInfo);
        }
    }

    private Map<Boolean, List<ResourceParticipantBriefInfo>> partitioningByAsyncWithExchange(List<ResourceParticipantBriefInfo> resources) {
        return StreamEx.of(resources)
                .partitioningBy(ResourceParticipantBriefInfo::isAsyncWithExchange);
    }

    public void exportToExchangeIfNeededOnUpdate(
            long eventId, EventChangesInfo eventChangesInfo,
            Option<OccurrenceId> occurrenceIdO, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (!ensureActionSource(actionInfo.getActionSource())) return;

        if (!eventChangesInfo.wasChange()) {
            return;
        }

        EventWithRelations event = eventDbManager.getEventWithRelationsById(eventId);
        RepetitionInstanceInfo repetitionInfo = repetitionRoutines.getRepetitionInstanceInfo(event);

        exportToExchangeIfNeededOnUpdate(event, repetitionInfo, occurrenceIdO, eventChangesInfo, Option.empty(), actionInfo);
    }

    public void exportToExchangeIfNeededOnUpdate(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition,
            Option<OccurrenceId> occurrenceIdO, EventChangesInfo eventChangesInfo,
            Option<ExchangeMails> mails, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (!eventChangesInfo.wasChange()) {
            return;
        }
        if (!mails.isPresent() && !eventChangesInfo.wasNonPerUserFieldsChange()) {
            mails = Option.of(ExchangeMails.NONE);
        }
        if (!ensureActionSource(actionInfo.getActionSource())) {
            return;
        }

        val participantsChanges = eventChangesInfo.getEventParticipantsChangesInfo();
        val removedParticipants = participantsChanges.getRemovedParticipants();
        val newResources = participantsChanges.getNewResources();

        val changes = eventChangesInfo.toEventChangesInfoForExchange();

        val removed = StreamEx.of(removedParticipants)
                .select(ResourceParticipantInfo.class)
                .filter(ResourceParticipantInfo.isYaTeamAndSyncWithExchangeF())
                .map(resourceParticipantInfo -> new ResourceParticipantBriefInfo(resourceParticipantInfo,
                        resourceRoutines.getExchangeEmail(resourceParticipantInfo.getResource())))
                .toImmutableList();

        exportToExchangeIfNeededOnUpdate(updatedEvent, updatedRepetition, occurrenceIdO.toOptional(), changes,
                eventChangesInfo.wasNonPerUserFieldsChange(), removed, newResources,
                mails.toOptional(), actionInfo);
    }

    public void exportToExchangeIfNeededOnUpdate(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition,
            Optional<OccurrenceId> occurrenceIdO, EventChangesInfoForExchange changes,
            boolean wasNonPerUserChanges, List<ResourceParticipantBriefInfo> removedParticipants, List<Long> newResources,
            Optional<ExchangeMails> mails, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        val wasExportedForOrganizer = exportToExchangeIfNeededOnUpdateForOrganizer(
                updatedEvent, updatedRepetition, occurrenceIdO, changes, mails, actionInfo);
        if (wasExportedForOrganizer || !wasNonPerUserChanges) {
            return;
        }

        val resources = mapToBriefInfo(updatedEvent.getResourceParticipantsYaTeamAndSyncWithExchange());
        val resourcesChanges = ResourceParticipantChangesInfo
                .byParticipantChangesInfo(resources, removedParticipants, newResources);

        val resourcesByAsyncWithExchange = partitioningByAsyncWithExchange(resources);

        val asyncResources = StreamEx.of(resourcesChanges.getAllResources())
                .filter(ResourceParticipantBriefInfo::isAsyncWithExchange)
                .append(resourcesByAsyncWithExchange.get(true))
                .toImmutableList();

        submitAsyncExport(asyncResources, updatedEvent.getMainEvent(), actionInfo);

        val notAsyncResources = resourcesByAsyncWithExchange.get(false);
        val notAsyncResourceChanges = resourcesChanges.filterNotAsync();

        if (notAsyncResources.isEmpty() && notAsyncResourceChanges.getRemovedResources().isEmpty()) {
            return;
        }

        if (occurrenceIdO.isPresent()) {
            createOrUpdateRecurrence(occurrenceIdO.get(), changes, updatedEvent, notAsyncResourceChanges, actionInfo);
        } else {
            updateOrCreateMeetings(changes, notAsyncResourceChanges, updatedEvent, updatedRepetition, actionInfo);
        }
    }

    public void exportToExchangeIfNeededOnUpdate(EventChangesInfoForExchange changes, Optional<OccurrenceId> occurrenceId, long eventId,
                                                 ResourceParticipantChangesInfo resourcesChanges, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        val updatedEvent = eventDbManager.getEventWithRelationsById(eventId);
        val updatedRepetition = repetitionRoutines.getRepetitionInstanceInfo(updatedEvent);

        val removedParticipants = resourcesChanges.getRemovedResources();
        val newResources = StreamEx.of(resourcesChanges.getNewResources())
                .map(ResourceParticipantBriefInfo::getResourceId)
                .toImmutableList();

        exportToExchangeIfNeededOnUpdate(updatedEvent, updatedRepetition, occurrenceId, changes,
                true, removedParticipants, newResources, Optional.empty(), actionInfo);
    }

    public void scheduleNewExportTask(EventChangesInfo eventChangesInfo, EventWithRelations updatedEvent, Optional<OccurrenceId> occurrenceId, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        val resources = mapToBriefInfo(updatedEvent.getResourceParticipantsYaTeamAndSyncWithExchange());
        val removed = StreamEx.of(eventChangesInfo.getEventParticipantsChangesInfo().getRemovedParticipants())
                .select(ResourceParticipantInfo.class)
                .filter(ResourceParticipantInfo.isYaTeamAndSyncWithExchangeF())
                .map(resourceParticipantInfo -> new ResourceParticipantBriefInfo(resourceParticipantInfo,
                        resourceRoutines.getExchangeEmail(resourceParticipantInfo.getResource())))
                .toImmutableList();

        val newResources = eventChangesInfo.getEventParticipantsChangesInfo().getNewResources();
        val resourcesChanges = ResourceParticipantChangesInfo.byParticipantChangesInfo(resources, removed, newResources);

        if (resources.isEmpty() && resourcesChanges.getRemovedResources().isEmpty()) {
            return;
        }

        val exportTask = new EwsExportEventResourceChangesOnUpdateTask(updatedEvent.getId(), resourcesChanges, occurrenceId, actionInfo);

        bazingaTaskManager.schedule(exportTask);
    }

    public void changeTimezoneInExchangeExistingMasterEvents(
            long recurringMasterEventId, DateTimeZone newTz, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        Event event = eventDao.findEventById(recurringMasterEventId);
        Validate.V.some(event.getRepetitionId(), "can change timezone at repeating event only");
        Validate.V.none(event.getRecurrenceId(), "timezone can be changed for recurring master only");

        String externalId = mainEventDao.findExternalIdByMainEventId(event.getMainEventId());

        ListF<ResourceParticipantInfo> affectedResources = resourceDao
                .findYaTeamResourceLayersThatSyncWithExchangeWithEvents(Cf.list(recurringMasterEventId))
                .filter(ResourceParticipantInfo.hasExchangeIdF());

        ListF<ItemChangeDescriptionType> changeDescriptions =
                eventToCalendarItemConverter.convertToChangeDescriptions(newTz, false);

        for (ResourceParticipantInfo resourceParticipant : affectedResources) {
            val logData = new EwsActionLogData(
                    UidOrResourceId.resource(resourceParticipant), new EventIdLogDataJson(externalId, event), actionInfo);

            Option<String> exchangeIdO = ewsProxyWrapper.updateItem(
                    EwsModifyingItemId.fromExchangeId(resourceParticipant.getExchangeId().get()),
                    changeDescriptions, logData);

            if (exchangeIdO.isPresent()) {
                eventResourceDao.saveUpdateExchangeId(
                        resourceParticipant.getResourceId(), event.getId(), exchangeIdO.get(), actionInfo);
            }
        }
    }

    private void submitAsyncExport(List<ResourceParticipantBriefInfo> resources, MainEvent event, ActionInfo actionInfo) {
        if (!resources.isEmpty()) {
            submitAsyncExportByEventExternalIds(List.of(event.getExternalId()), actionInfo);
        }
    }

    public void submitAsyncExportByEventExternalIds(List<String> externalIds, final ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        val events = StreamEx.of(externalIds)
                .map(id -> buildYtEwsExportingEvent(id, actionInfo.getNow()))
                .collect(CollectorsF.toList());
        ytEwsExportingEventDao.replaceEvents(events);
    }

    private static YtEwsExportingEvent buildYtEwsExportingEvent(String externalId, Instant now) {
        val event = new YtEwsExportingEvent();
        event.setExternalId(externalId);
        event.setLastSubmitTs(now);
        event.setNextAttemptTs(now);
        return event;
    }

    // http://wiki.yandex-team.ru/Calendar/exchange/exportRecurrenceId
    private void createOrUpdateRecurrence(
            OccurrenceId occurrenceId, EventChangesInfoForExchange eventChangesInfo,
            EventWithRelations event, ResourceParticipantChangesInfo resourcesChanges, ActionInfo actionInfo)
    {
        Validate.some(event.getRecurrenceId());

        val forCreate = resourcesChanges.getNewResources();
        val forRemove = resourcesChanges.getRemovedResources();
        val forUpdate = resourcesChanges.getNotChangedResources();

        RepetitionInstanceInfo repetitionInfo = RepetitionInstanceInfo.noRepetition(
                new InstantInterval(event.getEvent().getStartTs(), event.getEvent().getEndTs()));

        CalendarItemType itemToCreate =
                eventToCalendarItemConverter.convertToCalendarItem(event, repetitionInfo);
        ListF<ItemChangeDescriptionType> changesToUpdate =
                eventToCalendarItemConverter.convertToChangeDescriptions(event, repetitionInfo, eventChangesInfo);

        ListF<ExchangeIdLogData> createdExchangeIds = Cf.list();
        try {
            createdExchangeIds = doCreateMeetings(itemToCreate, forCreate, event, repetitionInfo, actionInfo);
            doUpdateMeetings(Optional.of(occurrenceId), changesToUpdate, forUpdate, event, actionInfo);
            doCancelOrDeclineOccurrence(occurrenceId, forRemove, event, actionInfo);

        } catch (Throwable t) {
            ewsProxyWrapper.cancelMeetingsSafe(createdExchangeIds, actionInfo);
            throw ExceptionUtils.throwException(t);
        }
    }

    private ListF<ExchangeIdLogData> createMeetings(
            List<ResourceParticipantBriefInfo> resourceParticipants,
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo)
    {
        Validate.none(event.getRecurrenceId());

        CalendarItemType calendarItem = eventToCalendarItemConverter.convertToCalendarItem(event, repetitionInfo);
        ListF<Instant> exdates = repetitionInfo.getExdates().map(Rdate.getStartTsF());

        ListF<ExchangeIdLogData> createdExchangeIds = Cf.list();
        try {
            createdExchangeIds = doCreateMeetings(calendarItem, resourceParticipants, event, repetitionInfo, actionInfo);
            doCancelOrDeclineOccurrences(exdates, resourceParticipants, event, actionInfo);
            return createdExchangeIds;

        } catch (Throwable t) {
            ewsProxyWrapper.cancelMeetingsSafe(createdExchangeIds, actionInfo);
            throw ExceptionUtils.throwException(t);
        }
    }

    private void updateOrCreateMeetings(
            EventChangesInfoForExchange eventChangesInfo, ResourceParticipantChangesInfo resourcesChanges,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetitionInfo, ActionInfo actionInfo)
    {
        Validate.none(updatedEvent.getRecurrenceId());

        val hasExchangeId = StreamEx.of(resourcesChanges.getNotChangedResources())
                .partitioningBy(res -> res.getExchangeId().isPresent());

        val forCreate = StreamEx.of(resourcesChanges.getNewResources())
                .append(hasExchangeId.get(false))
                .toImmutableList();
        val forRemove = resourcesChanges.getRemovedResources();
        val forUpdate = hasExchangeId.get(true);

        ListF<ItemChangeDescriptionType> changeDescriptions = eventToCalendarItemConverter
                .convertToChangeDescriptions(updatedEvent, updatedRepetitionInfo, eventChangesInfo);

        ListF<ExchangeIdLogData> createdExchangeIds = Cf.list();
        try {
            createdExchangeIds = createMeetings(forCreate, updatedEvent, updatedRepetitionInfo, actionInfo);

            // Note: No conflicts check / ewsMonitoring.reportMeetingConflict() call
            doUpdateMeetings(Optional.empty(), changeDescriptions, forUpdate, updatedEvent, actionInfo);
            doCancelOrDeclineOccurrences(eventChangesInfo.getNewExdates(), forUpdate, updatedEvent, actionInfo);

            doCancelOrDeclineMeetings(forRemove, updatedEvent, actionInfo);

        } catch (Throwable t) {
            ewsProxyWrapper.cancelMeetingsSafe(createdExchangeIds, actionInfo);
            throw ExceptionUtils.throwException(t);
        }
    }

    private void doUpdateMeetings(
            Optional<OccurrenceId> occurrenceIdO, ListF<ItemChangeDescriptionType> changeDescriptions,
            List<ResourceParticipantBriefInfo> resourceParticipants,
            EventWithRelations updatedEvent, ActionInfo actionInfo) {
        for (ResourceParticipantBriefInfo resourceParticipant : resourceParticipants) {
            val email = resourceParticipant.getExchangeEmail();
            val resourceId = resourceParticipant.getResourceId();
            val exchangeId = resourceParticipant.getExchangeId();
            val uidOrResourceId = resourceParticipant.getUidOrResourceId();

            val logData = new EwsActionLogData(uidOrResourceId, new EventIdLogDataJson(updatedEvent), actionInfo);

            EwsModifyingItemId ewsUpdatingItemId = occurrenceIdO.isPresent()
                    ? EwsModifyingItemId.fromEmailAndOccurrenceId(email, occurrenceIdO.get())
                    : EwsModifyingItemId.fromExchangeId(exchangeId.get());

            Option<String> exchangeIdO = ewsProxyWrapper.updateItem(ewsUpdatingItemId, changeDescriptions, logData);
            if (exchangeIdO.isPresent()) {
                eventResourceDao.saveUpdateExchangeId(resourceId, updatedEvent.getId(), exchangeIdO.get(), actionInfo);
            }
        }
    }

    private List<ResourceParticipantBriefInfo> mapToBriefInfo(List<ResourceParticipantInfo> resourceParticipants) {
        return StreamEx.of(resourceParticipants).map(resourceParticipantInfo ->
                new ResourceParticipantBriefInfo(resourceParticipantInfo, resourceRoutines.getExchangeEmail(resourceParticipantInfo.getResource())))
                .toImmutableList();
    }

    private ListF<ExchangeIdLogData> doCreateMeetings(CalendarItemType meeting, List<ResourceParticipantBriefInfo> resourceParticipants,
                                                      EventWithRelations event, RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo) {
        ListF<ExchangeIdLogData> storedExchangeIds = Cf.arrayList();
        try {
            for (ResourceParticipantBriefInfo resource : resourceParticipants) {
                val email = resource.getExchangeEmail();
                val resourceEmail = resource.getResourceEmail();
                val uidOrResourceId = resource.getUidOrResourceId();
                val resourceId = resource.getResourceId();
                val eventId = resource.getEventId();

                val logId = new EventIdLogDataJson(event);
                val logData = new EwsActionLogData(uidOrResourceId, logId, actionInfo);

                val exchangeId = ewsProxyWrapper.createEvent(email, meeting, logData);
                storedExchangeIds.add(new ExchangeIdLogData(exchangeId, uidOrResourceId, logId));

                val newCreateEvent = getItemWithActualConflicts(
                        email, exchangeId, repetitionInfo,  actionInfo.getNow()).get();
                val conflicts = newCreateEvent.getConflictingItems();

                if (conflicts.size() == 1 && isSameEvent(event, conflicts.single())) {
                    ewsProxyWrapper.cancelMeetingsSafe(Cf.list(storedExchangeIds.last()), actionInfo);
                    storedExchangeIds.remove(storedExchangeIds.size() - 1);

                    ewsMonitoring.reportMeetingCreateOrUpdateAttempt();
                    saveUpdateExchangeId(resourceId, eventId, conflicts.single().getExchangeId(), actionInfo);
                } else if (conflicts.isEmpty()) {
                    ewsMonitoring.reportMeetingCreateOrUpdateAttempt();
                    saveUpdateExchangeId(resourceId, eventId, exchangeId, actionInfo);
                } else {
                    ewsMonitoring.reportMeetingCreateOrUpdateConflict();
                    throw conflictCommandRunException(event.getEvent(), resourceEmail, newCreateEvent);
                }
            }
            return storedExchangeIds;
        } catch (Throwable t) {
            ewsProxyWrapper.cancelMeetingsSafe(storedExchangeIds, actionInfo);
            throw ExceptionUtils.throwException(t);
        }
    }

    private void doCancelOrDeclineMeetings(List<ResourceParticipantBriefInfo> resourceParticipants,
                                           EventWithRelations updatedEvent, ActionInfo actionInfo) {
        val exchangeIdLogData = StreamEx.of(resourceParticipants)
                .flatMap(r -> r.getExchangeId()
                        .map(exchangeId -> new ExchangeIdLogData(exchangeId, r.getUidOrResourceId(), new EventIdLogDataJson(updatedEvent)))
                        .stream())
                .collect(CollectorsF.toList());
        ewsProxyWrapper.cancelOrDeclineMeetingSafe(exchangeIdLogData, actionInfo);
    }

    private void doCancelOrDeclineOccurrence(OccurrenceId occurrenceId, List<ResourceParticipantBriefInfo> resourceParticipantBriefInfos,
                                             EventWithRelations event, ActionInfo actionInfo) {
        for (ResourceParticipantBriefInfo resource : resourceParticipantBriefInfos) {
            val email = resource.getExchangeEmail();
            val uidOrResourceId = resource.getUidOrResourceId();
            ewsProxyWrapper.cancelOrDeclineMeetingOccurrenceSafe(
                    EwsModifyingItemId.fromEmailAndOccurrenceId(email, occurrenceId),
                    new EwsActionLogData(uidOrResourceId, new EventIdLogDataJson(event), actionInfo));
        }
    }

    private void doCancelOrDeclineOccurrences(
            ListF<Instant> occurrencesStarts,
            List<ResourceParticipantBriefInfo> resourceParticipants,
            EventWithRelations event, ActionInfo actionInfo) {
        for (ResourceParticipantBriefInfo resource : resourceParticipants) {
            val email = resource.getExchangeEmail();
            val uidOrResourceId = resource.getUidOrResourceId();
            ewsProxyWrapper.cancelOrDeclineMeetingOccurrencesSafe(email, event.getExternalId(), occurrencesStarts,
                    new EwsActionLogData(uidOrResourceId, new EventIdLogDataJson(event), actionInfo));
        }
    }

    private void saveUpdateExchangeId(long resourceId, long eventId, String exchangeId, ActionInfo actionInfo) {
        eventResourceDao.saveUpdateExchangeId(resourceId, eventId, exchangeId, actionInfo);
    }

    public Option<ItemWithConflicts> getItemWithActualConflicts(
            Email email, String exchangeId, RepetitionInstanceInfo repetitionInfo, Instant now)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list(
                UnindexedFieldURIType.CALENDAR_CONFLICTING_MEETING_COUNT,
                UnindexedFieldURIType.CALENDAR_CONFLICTING_MEETINGS);

        Option<CalendarItemType> item = ewsProxyWrapper.getEvents(Cf.list(exchangeId), fields).singleO();

        ListF<ConflictingItem> conflicts = item.isPresent() ? getConflictingItems(item.get(), now) : Cf.list();

        Option<InstantInterval> instanceInterval =
                RepetitionUtils.getInstanceIntervalEndingAfter(repetitionInfo, now);

        if (item.isPresent() && instanceInterval.isPresent()) { // CAL-6923
            conflicts = conflicts.plus(findActualConflictingItems(email, exchangeId, instanceInterval.get(), now));

            conflicts = conflicts.stableUniqueBy(ConflictingItem.getExchangeIdF());
        }
        return item.isPresent()
                ? Option.of(new ItemWithConflicts(item.get().getItemId().getId(), conflicts))
                : Option.empty();
    }

    private ListF<ConflictingItem> getConflictingItems(CalendarItemType calendarItem, Instant now) {
        if (Integer.valueOf(0).equals(calendarItem.getConflictingMeetingCount())) return Cf.list();

        if (calendarItem.getConflictingMeetings() == null) {
            throw new EwsConflictingMeetingsNotCalculated();
        }

        ListF<CalendarItemType> conflicts = Cf.x(calendarItem.getConflictingMeetings().getItemOrMessageOrCalendarItem())
                .filterByType(CalendarItemType.class);

        conflicts = conflicts.filter(c -> EwsUtils.xmlGregorianCalendarInstantToInstant(c.getEnd()).isAfter(now));

        ListF<UnindexedFieldURIType> additionFields = Cf.list(
                UnindexedFieldURIType.CALENDAR_UID,
                UnindexedFieldURIType.CALENDAR_CALENDAR_ITEM_TYPE,
                UnindexedFieldURIType.CALENDAR_IS_CANCELLED);

        ListF<String> additionExtendedFields = Cf.list(
                EwsUtils.EXTENDED_PROPERTY_SOURCE, EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID);

        ListF<CalendarItemType> addition = ewsProxyWrapper.getEvents(
                conflicts.map(EwsUtils.itemIdF()), additionFields, additionExtendedFields);

        addition = addition.filterNot(CalendarItemType::isIsCancelled);

        Tuple2List<CalendarItemType, CalendarItemType> joined = addition.zipWith(
                EwsUtils.itemIdF().andThen(Cf2.f(conflicts.toMapMappingToKey(EwsUtils.itemIdF())::getOrThrow)));

        return joined.map(consConflictingItemF());
    }

    public ListF<ConflictingItem> findActualConflictingItems(
            Email email, String exchangeId, InstantInterval instanceInterval, Instant now)
    {
        if (instanceInterval.getEnd().isBefore(now)) return Cf.list();

        ListF<UnindexedFieldURIType> fields = Cf.list(
                UnindexedFieldURIType.CALENDAR_UID,
                UnindexedFieldURIType.CALENDAR_CALENDAR_ITEM_TYPE,
                UnindexedFieldURIType.ITEM_SUBJECT,
                UnindexedFieldURIType.CALENDAR_START,
                UnindexedFieldURIType.CALENDAR_END,
                UnindexedFieldURIType.CALENDAR_RECURRENCE_ID);

        ListF<String> extendedFields = Cf.list(
                EwsUtils.EXTENDED_PROPERTY_SOURCE, EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID);

        ListF<CalendarItemType> items = ewsProxyWrapper.findInstanceEvents(email, instanceInterval, fields, extendedFields);

        Function<CalendarItemType, String> exchangeIdF = EwsUtils.calendarItemExchangeIdF();

        MapF<String, Boolean> isCancelledByExchangeId = ewsProxyWrapper
                .getEvents(items.map(exchangeIdF), Cf.list(UnindexedFieldURIType.CALENDAR_IS_CANCELLED))
                .toMap(exchangeIdF, CalendarItemType::isIsCancelled);

        MapF<String, Option<String>> recurringMasterIdByItemId =
                ewsProxyWrapper.findRecurringMasterIdsByInstanceIds(items.map(exchangeIdF)).toMap();

        ListF<ConflictingItem> conflicts = items.zipWith(Function.identityF()).map(consConflictingItemF());

        return conflicts.filter(i -> i.getInterval().overlaps(instanceInterval)
                && !i.getExchangeId().equals(exchangeId)
                && !recurringMasterIdByItemId.getOrThrow(i.getExchangeId()).isSome(exchangeId)
                && isCancelledByExchangeId.getO(i.getExchangeId()).isSome(false)
                && i.getEnd().isAfter(now));
    }

    private Function2<CalendarItemType, CalendarItemType, ConflictingItem> consConflictingItemF() {
        return (addition, conflict) -> new ConflictingItem(
                conflict.getItemId().getId(),
                conflict.getSubject(),
                EwsUtils.xmlGregorianCalendarInstantToInstant(conflict.getStart()),
                EwsUtils.xmlGregorianCalendarInstantToInstant(conflict.getEnd()),
                EwsUtils.convertExtendedProperties(addition).getRecurrenceId(),
                StringUtils.defaultIfEmpty(addition.getUID(), "?"),
                addition.getCalendarItemType(),
                EwsUtils.convertExtendedProperties(addition).getWasCreatedFromYaTeamCalendar());
    }

    private static CommandRunException conflictCommandRunException(
            Event event, Email resourceEmail, ItemWithConflicts itemWithConflicts)
    {
        String message = "conflicting items count: " + itemWithConflicts.getConflictingItems().size();
        ConflictingItem conflictingItem = itemWithConflicts.getConflictingItems().first();

        message += ", first item info: ";
        message += LogMarker.EXCHANGE_ID.format(conflictingItem.getExchangeId());
        message += LogMarker.EXT_ID.format(conflictingItem.getExternalId());
        message += ", event name: " + conflictingItem.getName();
        message += ", event start: " + conflictingItem.getStart();
/*
        // markers that can help to determine events that need to be 'fixed'
        if (ExchangeAdminManager.isAllDay(conflictingItem)) {
            message += " ALL_DAY";
        }
        if (ExchangeAdminManager.isUtcItemWithType(conflictingItem, CalendarItemTypeType.OCCURRENCE)) {
            message += " UTC_OCCURRENCE";
        }
*/
        String reason =
                "This meeting (" + event.getId() + ")" +
                " conflicts with existing meetings (" + message + ")" +
                ", email = " + resourceEmail;
        return CommandRunException.createSituation(reason, Situation.EXCHANGE_EVENT_CONFLICT);
    }

    private boolean isSameEvent(EventWithRelations event, ConflictingItem conflictingItem) {
        return event.getExternalId().equals(conflictingItem.getExternalId())
                && event.getRecurrenceId().equals(conflictingItem.getRecurrenceId());
    }

    // https://jira.yandex-team.ru/browse/CAL-3470
    public void deleteEventIfNeeded(long eventId) {
        deleteEventsIfNeeded(Cf.list(eventId), ActionInfo.adminManager());
    }

    public void deleteEventsIfNeeded(ListF<Long> eventIds, ActionInfo actionInfo) {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (ensureActionSource(actionInfo.getActionSource())) {
            MapF<Long, Event> eventById = eventDao.findEventsByIds(eventIds).toMapMappingToKey(Event::getId);

            MapF<Long, MainEvent> mainById = mainEventDao.findMainEventsByIds(
                    eventById.values().map(Event::getMainEventId)).toMapMappingToKey(MainEvent::getId);

            MapF<Long, MainEvent> mainByEventId = eventById.mapValues(e -> mainById.getOrThrow(e.getMainEventId()));

            ListF<Long> deletedOrganizerEventIds = deleteEventsIfNeededForOrganizer(eventIds.toTuple2List(
                    id -> Tuple2.tuple(eventById.getOrThrow(id), mainByEventId.getOrThrow(id))), actionInfo);

            // Remove events only from exchange calendars of resources
            // (we have no permissions to remove events from exchange calendars
            // of users)
            ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIds(
                    eventIds.filterNot(deletedOrganizerEventIds::containsTs));

            SetF<Long> asyncIds = resourceDao.findResourceIdsAsyncWithExchangeByIds(
                    eventResources.map(EventResource.getResourceIdF())).unique();

            Function1B<EventResource> isAsyncF = EventResource.getResourceIdF().andThen(asyncIds.containsF());
            if (asyncIds.isNotEmpty()) {
                submitAsyncExportByEventExternalIds(mainEventDao.findExternalIdsByEventIds(
                        eventResources.filter(isAsyncF).map(EventResource.getEventIdF())), actionInfo);
            }
            ewsProxyWrapper.cancelOrDeclineMeetingSafe(eventResources.filter(isAsyncF.notF())
                    .filterMap(er -> er.getExchangeId().map(exchId -> new ExchangeIdLogData(
                            exchId, UidOrResourceId.resource(er.getResourceId()), new EventIdLogDataJson(
                                    mainByEventId.getOrThrow(er.getEventId()), eventById.getOrThrow(er.getEventId()))))),
                    actionInfo);
        }
    }

    public void cancelOccurrenceIfNeeded(
            long eventId, OccurrenceId occurrenceId, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        cancelOccurrenceIfNeeded(eventDbManager.getEventWithRelationsById(eventId), occurrenceId, actionInfo);
    }

    public void cancelOccurrenceIfNeeded(
            EventWithRelations event, OccurrenceId occurrenceId, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (ensureActionSource(actionInfo.getActionSource())) {
            long eventId = event.getId();

            if (doCancelOrDeclineOccurrenceForOrganizerIfNeeded(event, occurrenceId, actionInfo)) {
                return;
            }

            ListF<ResourceParticipantInfo> resourceParticipants =
                    resourceDao.findYaTeamResourceLayersThatSyncWithExchangeWithEvents(Cf.list(eventId));

            if (resourceParticipants.exists(ResourceParticipantInfo.isAsyncWithExchangeF())) {
                submitAsyncExportByEventExternalIds(mainEventDao.findExternalIdsByEventIds(Cf.list(eventId)), actionInfo);
            }
            doCancelOrDeclineOccurrence(
                    occurrenceId, mapToBriefInfo(resourceParticipants.filter(ResourceParticipantInfo.isAsyncWithExchangeF().notF())),
                    event, actionInfo);
        }
    }

    private boolean doCancelOrDeclineOccurrenceForOrganizerIfNeeded(
            EventWithRelations event, OccurrenceId occurenceId, ActionInfo actionInfo)
    {
        if (!event.isExportedWithEws()) {
            return false;
        }

        EwsModifyingItemId ewsId = EwsModifyingItemId.fromEmailAndOccurrenceId(
                getOrganizerIfMeetingOrElseCreatorExchangeEmail(event), occurenceId);
        ewsProxyWrapper.cancelOrDeclineMeetingOccurrenceSafe(
                ewsId, MessageDispositionType.SEND_ONLY, CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL,
                new EwsActionLogData(getOrganizerIfMeetingOrElseCreator(event), new EventIdLogDataJson(event), actionInfo));
        return true;
    }

    public void cancelOccurrencesIfNeeded(
            EventWithRelations event, ListF<Instant> starts, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (ensureActionSource(actionInfo.getActionSource())) {
            if (doCancelOrDeclineOccurrencesForOrganizerIfNeeded(event, starts, actionInfo)) {
                return;
            }

            val resources = mapToBriefInfo(event.getResourceParticipantsYaTeamAndSyncWithExchange());
            val resourcesByAsyncWithExchange = partitioningByAsyncWithExchange(resources);

            submitAsyncExport(resourcesByAsyncWithExchange.get(true), event.getMainEvent(), actionInfo);
            doCancelOrDeclineOccurrences(starts, resourcesByAsyncWithExchange.get(false), event, actionInfo);
        }
    }

    private boolean doCancelOrDeclineOccurrencesForOrganizerIfNeeded(
            EventWithRelations event, ListF<Instant> starts, ActionInfo actionInfo)
    {
        if (!event.isExportedWithEws()) {
            return false;
        }

        ewsProxyWrapper.cancelOrDeclineMeetingOccurrencesSafe(getOrganizerIfMeetingOrElseCreatorExchangeEmail(event),
                event.getExternalId(), starts, MessageDispositionType.SEND_ONLY,
                CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL,
                new EwsActionLogData(getOrganizerIfMeetingOrElseCreator(event), new EventIdLogDataJson(event), actionInfo));
        return true;
    }

    public boolean ensureActionSource(ActionSource actionSource) {
        // Don't export to exchange changes from UNKNOWN source.
        // For example, EventMerger calls eventRoutines.deleteCommon() with UNKNOWN
        // action source, because EventMerger shouldn't remove events from exchange.
        return actionSource != ActionSource.EXCHANGE &&
               actionSource != ActionSource.EXCHANGE_SYNCH &&
               actionSource != ActionSource.EXCHANGE_PULL &&
               actionSource != ActionSource.EXCHANGE_ASYNCH &&
               actionSource != ActionSource.UNKNOWN &&
               actionSource != ActionSource.MAILHOOK &&
               actionSource != ActionSource.MAIL &&
               actionSource != ActionSource.DB_REPAIRER &&
               actionSource != ActionSource.WEB_ICS;
    }

    private boolean exportToExchangeIfNeededOnCreateForOrganizer(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Optional<ExchangeMails> mails, ActionInfo actionInfo)
    {
        if (!event.isExportedWithEws()) {
            return false;
        }

        if (event.getRecurrenceId().isPresent()) {
            Option<Event> masterEvent = eventDao.findMasterEventByMainId(event.getMainEventId()).singleO();

            Validate.some(masterEvent, "master event is required to modify occurrence in ews");

            EventChangesInfoForExchange eventChangesInfo = eventChangesFinder
                    .getEventChangesInfoForExchangeForNewRecurrence(masterEvent.get(), event.getEvent());

            Duration instanceDuration = EventRoutines.getInstantInterval(event.getEvent()).getDuration();
            InstantInterval instanceInterval = new InstantInterval(event.getRecurrenceId().get(), instanceDuration);
            OccurrenceId occurrenceId = new OccurrenceId(event.getExternalId(), instanceInterval, event.getRecurrenceId().get());

            updateRecurrenceForOrganizer(occurrenceId, eventChangesInfo, event, mails, actionInfo);
        } else {
            String exchangeId = createMeetingsForOrganizer(event, repetitionInfo, Option.empty(),
                    MessageDispositionType.SEND_ONLY,
                    CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL, actionInfo);
            eventUserDao.saveUpdateExchangeId(event.getOrganizerIfMeetingOrElseCreator().getUid(),
                    event.getId(), exchangeId, actionInfo);
        }

        return true;
    }

    private boolean exportToExchangeIfNeededOnUpdateForOrganizer(
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetition,
            Optional<OccurrenceId> occurrenceIdO, EventChangesInfoForExchange changes,
            Optional<ExchangeMails> mails, ActionInfo actionInfo) {
        if (!updatedEvent.isExportedWithEws()) {
            return false;
        }

        if (occurrenceIdO.isPresent()) {
            updateRecurrenceForOrganizer(occurrenceIdO.get(), changes, updatedEvent, mails, actionInfo);
        } else {
            updateMeetingsForOrganizer(changes, updatedEvent, updatedRepetition, mails, actionInfo);
        }

        return true;
    }

    private ListF<Long> deleteEventsIfNeededForOrganizer(Tuple2List<Event, MainEvent> events, ActionInfo actionInfo) {
        events = events.filter(e -> e.get2().getIsExportedWithEws().isSome(true));

        ListF<Long> exportedWithEwsEventIds = events.map(e -> e.get1().getId());

        SetF<Long> masterMainIds = events.filterMap(
                e -> Option.when(!e.get1().getRecurrenceId().isPresent(), e.get2().getId())).unique();

        events = events.filter((e, me) -> !e.getRecurrenceId().isPresent() || !masterMainIds.containsTs(me.getId()));

        ListF<EventUser> organizers = eventUserDao.findEventUsers(
                EventUserFields.EVENT_ID.column().inSet(events.map(e -> e.get1().getId()))
                        .and(EventUserFields.IS_ORGANIZER.column().eq(true)));

        ListF<EventUser> creators = eventUserDao.findEventUsersByEventIdsAndUids(
                events.map(e -> e.get1().getId()), events.map(e -> e.get1().getCreatorUid()));

        MapF<Long, EventUser> organizerByEventId = organizers.toMapMappingToKey(EventUser::getEventId);
        MapF<Long, EventUser> creatorByEventId = creators.toMapMappingToKey(EventUser::getEventId);

        ListF<ExchangeIdLogData> exchangeIds = events.filterMap(
                t -> organizerByEventId.getO(t.get1().getId()).orElse(creatorByEventId.getO(t.get1().getId()))
                        .filterMap(eu -> eu.getExchangeId().map(exchId -> new ExchangeIdLogData(
                                exchId, UidOrResourceId.user(eu.getUid()), new EventIdLogDataJson(t.get2(), t.get1())))));

        ewsProxyWrapper.deleteEvents(exchangeIds, CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL, actionInfo);

        return exportedWithEwsEventIds;
    }

    private void updateRecurrenceForOrganizer(
            OccurrenceId occurrenceId, EventChangesInfoForExchange eventChangesInfo,
            EventWithRelations event, Optional<ExchangeMails> mails, ActionInfo actionInfo)
    {
        Validate.some(event.getRecurrenceId());

        RepetitionInstanceInfo repetitionInfo = RepetitionInstanceInfo.noRepetition(
                new InstantInterval(event.getEvent().getStartTs(), event.getEvent().getEndTs()));

        ListF<ItemChangeDescriptionType> changesToUpdate =
                eventToCalendarItemConverter.convertToChangeDescriptions(event, repetitionInfo, eventChangesInfo);

        doUpdateMeetingsForOrganizer(Option.of(occurrenceId), changesToUpdate, event, mails, false, actionInfo);
    }

    private String createMeetingsForOrganizer(
            EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            Option<BodyType> overwriteBody, MessageDispositionType dispositionType,
            CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        Validate.none(event.getRecurrenceId());

        CalendarItemType calendarItem = eventToCalendarItemConverter.convertToCalendarItem(event, repetitionInfo);

        if (overwriteBody.isPresent()) {
            calendarItem.setBody(overwriteBody.get());
        }

        ListF<Instant> exdates = repetitionInfo.getExdates().map(Rdate.getStartTsF());

        Option<ExchangeIdLogData> createdExchangeIdO = Option.empty();
        try {
            UidOrResourceId subjectId = getOrganizerIfMeetingOrElseCreator(event);
            val logId = new EventIdLogDataJson(event);
            val logData = new EwsActionLogData(subjectId, logId, actionInfo);

            val createdExchangeId = ewsProxyWrapper.createEvent(
                    getOrganizerIfMeetingOrElseCreatorExchangeEmail(event), calendarItem,
                    dispositionType, operationType, logData);

            createdExchangeIdO = Option.of(new ExchangeIdLogData(createdExchangeId, subjectId, logId));
            ewsProxyWrapper.cancelOrDeclineMeetingOccurrencesSafe(
                    getOrganizerIfMeetingOrElseCreatorExchangeEmail(event), event.getExternalId(), exdates,
                    dispositionType, operationType, logData);
            return createdExchangeId;

        } catch (Throwable t) {
            ewsMonitoring.reportIfEwsException(t);
            ewsProxyWrapper.cancelMeetingsSafe(createdExchangeIdO, operationType, actionInfo);
            throw ExceptionUtils.throwException(t);
        }
    }

    private void updateMeetingsForOrganizer(EventChangesInfoForExchange eventChangesInfo,
            EventWithRelations updatedEvent, RepetitionInstanceInfo updatedRepetitionInfo,
            Optional<ExchangeMails> mails, ActionInfo actionInfo)
    {
        Validate.none(updatedEvent.getRecurrenceId());

        ListF<ItemChangeDescriptionType> changeDescriptions = eventToCalendarItemConverter
                .convertToChangeDescriptions(updatedEvent, updatedRepetitionInfo, eventChangesInfo);

        doUpdateMeetingsForOrganizer(
                Option.empty(), changeDescriptions, updatedEvent, mails, false, actionInfo);
    }

    private void doUpdateMeetingsForOrganizer(
            Option<OccurrenceId> occurrenceIdO,
            ListF<ItemChangeDescriptionType> changeDescriptions, EventWithRelations updatedEvent,
            Optional<ExchangeMails> mails, boolean findExchangeId, ActionInfo actionInfo)
    {
        EwsModifyingItemId ewsUpdatingItemId;

        if (occurrenceIdO.isPresent()) {
            ewsUpdatingItemId = EwsModifyingItemId.fromEmailAndOccurrenceId(
                    getOrganizerIfMeetingOrElseCreatorExchangeEmail(updatedEvent), occurrenceIdO.get());
        } else {
            Option<String> exchangeIdO;
            if (findExchangeId) {
                Option<Email> email = updatedEvent.getOrganizerIfMeetingOrElseCreatorEmail();

                if (!email.isPresent()) {
                    return;
                }

                Event event = updatedEvent.getEvent();
                InstantInterval interval = new InstantInterval(event.getStartTs(), event.getEndTs());

                exchangeIdO = ewsProxyWrapper.findMasterAndSingleEventIdsByExternalId(
                        email.get(), interval, updatedEvent.getExternalId()).singleO();

                if (!exchangeIdO.isPresent()) {
                    return;
                }

            } else {
                exchangeIdO = updatedEvent.getOrganizerIfMeetingOrElseCreatorEventExchangeId();

                if (!exchangeIdO.isPresent()) {
                    doUpdateMeetingsForOrganizer(
                            occurrenceIdO, changeDescriptions, updatedEvent, mails, true, actionInfo);
                    return;
                }
            }
            ewsUpdatingItemId = EwsModifyingItemId.fromExchangeId(exchangeIdO.get());
        }

        EwsActionLogData logData = new EwsActionLogData(
                getOrganizerIfMeetingOrElseCreator(updatedEvent), new EventIdLogDataJson(updatedEvent), actionInfo);

        Option<String> newExchangeIdO = ewsProxyWrapper.updateItem(ewsUpdatingItemId, changeDescriptions,
                mails.orElse(ExchangeMails.ALL).getForUpdate(), logData);

        if (newExchangeIdO.isPresent()) {
            eventUserDao.saveUpdateExchangeId(updatedEvent.getOrganizerIfMeetingOrElseCreator().getUid(),
                    updatedEvent.getId(), newExchangeIdO.get(), actionInfo);
        } else if (!findExchangeId) {
            doUpdateMeetingsForOrganizer(occurrenceIdO, changeDescriptions, updatedEvent, mails, true, actionInfo);
        }
    }

    public void detachEvents(
            ListF<EventAndRepetition> events, MainEvent mainEvent, PassportUid uid, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (!settingsRoutines.getIsEwser(uid)) {
            return;
        }
        ListF<Event> eventsToDetach = events.filterMap(
                e -> Option.when(e.goesOnAfter(actionInfo.getNow()), e.getEvent()));

        ListF<EventUser> eventUsers =
                eventUserDao.findEventUsersByEventIdsAndUid(events.map(EventAndRepetition::getEventId), uid);

        ListF<Long> isEventUser = eventUsers.map(EventUser::getEventId);
        ListF<Long> isAttendeeEvents = eventUsers.filter(EventUser::getIsAttendee).map(EventUser::getEventId);

        eventsToDetach = eventsToDetach.filter(e -> isEventUser.containsTs(e.getId()));

        updateAttendeeDecisionIfNeeded(eventsToDetach, mainEvent, UidOrResourceId.user(uid), Decision.NO,
                Option.empty(), isAttendeeEvents::containsTs, actionInfo);
    }

    public MapF<Long, Boolean> updateAttendeeDecisionIfNeeded(ListF<Event> events, MainEvent mainEvent,
            UidOrResourceId attendee, Decision decision, Option<String> reason, boolean sendMail, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return events.toMap(Event::getId, e -> true);
        return updateAttendeeDecisionIfNeeded(events, mainEvent, attendee, decision, reason, e -> sendMail, actionInfo);
    }

    private MapF<Long, Boolean> updateAttendeeDecisionIfNeeded(ListF<Event> events, MainEvent mainEvent,
            UidOrResourceId attendee, Decision decision, Option<String> reason,
            Function1B<Long> doSendMail, ActionInfo actionInfo)
    {
        ListF<Event> masterOrAllRecurrences = events.find(event -> !event.getRecurrenceId().isPresent());
        Function1B<Long> doSendMailInner;

        if (masterOrAllRecurrences.isNotEmpty()) {
            boolean masterSendMail = events.exists(e -> doSendMail.apply(e.getId()));
            doSendMailInner = e -> masterSendMail;
        } else {
            masterOrAllRecurrences = events;
            doSendMailInner = doSendMail;
        }

        final var result = masterOrAllRecurrences.toMap(Event::getId, event -> updateAttendeeDecisionIfNeeded(
                event, mainEvent, attendee, decision, reason, doSendMailInner.apply(event.getId()), actionInfo));
        if (result.size() != events.size()) {
            val missingEventsEntries = StreamEx.of(events)
                .map(Event::getId)
                .filter(not(result::containsKeyTs))
                .mapToEntry(id -> false)
                .toImmutableList();
            result.putAllEntries(missingEventsEntries);
        }
        return result;
    }

    public boolean updateAttendeeDecisionIfNeeded(Event event, MainEvent mainEvent,
            UidOrResourceId attendee, Decision decision, Option<String> reason, boolean sendMail, ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return true;
        if (!attendee.isResource() && !settingsRoutines.getIsEwser(attendee.getUid())) {
            return false;
        }

        if (attendee.isResource() && !mainEvent.getIsExportedWithEws().getOrElse(false)) {
            return false;
        }

        EwsActionLogData logData = new EwsActionLogData(
                attendee, new EventIdLogDataJson(mainEvent.getExternalId(), event), actionInfo);

        CalendarItemCreateOrDeleteOperationType operationType = sendMail
            ? CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL
            : CalendarItemCreateOrDeleteOperationType.SEND_TO_NONE;

        Function<String, Boolean> setDecision = exchangeId -> ewsProxyWrapper.setUserDecision(
                exchangeId, decision, reason, MessageDispositionType.SEND_ONLY, operationType, logData);

        Option<String> exchangeId = attendee.isResource()
                ? event.getFieldValueO(EventFields.ID).filterMap(
                        eventId -> eventResourceDao.findEventResourceByEventIdAndResourceId(
                                eventId, attendee.getResourceId())).filterMap(EventResource::getExchangeId)
                : eventUserDao.findEventUserByEventIdAndUid(event.getId(), attendee.getUid())
                        .filterMap(EventUser::getExchangeId);

        if (exchangeId.isPresent() && setDecision.apply(exchangeId.get())) {
            return true;
        }

        logger.info("Ews export for decision failed via exchange id, stored in DB or no id found. " +
                "Resolving id from Exchange...");

        exchangeId = findIdInExchange(attendee, event, mainEvent);

        if (!exchangeId.isPresent()) {
            // CAL-8540
            eventsLogger.log(new EwsCallLogEventJson(
                    EwsCallOperation.REPLY_MEETING, logData.logEventId, Optional.empty(),
                    Optional.of(MessageDispositionType.SEND_ONLY),
                    Either.left(CalendarItemCreateOrDeleteOperationType.SEND_ONLY_TO_ALL),
                    Optional.of("Not found")), actionInfo);

            if (attendee.isUser()) {
                if (decision == Decision.NO) {
                    // if no event were found in exchange on decline, there is no need to reinvite user
                    // but we'll need to send the decline email ourselves.
                    return false;
                }
                // if event is missing in exchange, reinvite user.
                // His decision wil be then set by EventRoutines.fixDecisionInExchangeIfNeeded
                EventWithRelations eventWR = eventDbManager.getEventWithRelationsByEvent(event);
                eventInvitationManager.restoreEventMissingInExchange(eventWR, attendee.getUid(), actionInfo);
            }

            return true;
        }

        if (setDecision.apply(exchangeId.get())) {
            return true;
        } else {
            throw CommandRunException.createSituation(
                    "Ews export for decision failed via exchange id, resolved from Exchange. " +
                    "Check inviteBot2013 permissions for user " + attendee,
                    Situation.EWS_SET_DECISION_FAILED);
        }
    }

    public Option<String> findIdInExchange(UidOrResourceId subject, Event event, MainEvent mainEvent) {
        Email email = subject.isResource()
                ? resourceRoutines.getExchangeEmailById(subject.getResourceId())
                : userManager.getLdEmailByUid(subject.getUid());

        InstantInterval interval = new InstantInterval(event.getStartTs(), event.getEndTs());
        String externalId = mainEvent.getExternalId();

        String intervalString = interval.getStart() + "-" + interval.getEnd();

        Option<String> exchangeId = ewsProxyWrapper.findMasterAndSingleOrInstanceEventId(
                email, interval, externalId, event.getRecurrenceId().isPresent());

        if (!exchangeId.isPresent()) {
            logger.warn("Failed to find event by interval " + intervalString + " and id " + externalId);
        }
        return exchangeId;
    }

    public void setResourceDecisionIfNeeded(
            long resourceId, EventWithRelations eventWithRelations, RepetitionInstanceInfo repetitionInfo,
            Option<Decision> currentSubjectDecision, Decision decision, Option<NameI18n> reasonI18n,
            ActionInfo actionInfo)
    {
        if (!ewsAliveHandler.isEwsAlive()) return;
        if (actionInfo.getActionSource() == ActionSource.DISPLAY || !repetitionInfo.goesOnAfter(actionInfo.getNow())) {
            return;
        }

        Event event = eventWithRelations.getEvent();
        MainEvent mainEvent = eventWithRelations.getMainEvent();

        Option<String> reason = reasonI18n.map(name ->
                name.getName(settingsRoutines.getLanguageOrGetDefault(eventWithRelations.getOrganizerUidIfMeeting())));

        if (currentSubjectDecision.getOrElse(Decision.UNDECIDED) != decision) {
            updateAttendeeDecisionIfNeeded(
                    event, mainEvent, UidOrResourceId.resource(resourceId), decision, reason, true, actionInfo);
        }

    }

    public void forceUpdateEventForOrganizer(EventWithRelations event, RepetitionInstanceInfo repetitionInfo,
            ActionInfo actionInfo)
    {
/*      do nothing. still leads to cyclic updates investigation required
        Option<OccurrenceId> occurrenceIdO = event.getOccurenceId();
        EventChangesInfoForExchange changes = new EventChangesInfoForExchange(
                event.getEvent(), repetitionInfo.getRepetitionOrNone(), true, Cf.list());
        Option<ExchangeMails> mails = Option.of(ExchangeMails.ALL);

        if (occurrenceIdO.isPresent()) {
            updateRecurrenceForOrganizer(occurrenceIdO.get(), changes, event, mails, actionInfo);
        } else {
            updateMeetingsForOrganizer(changes, event, repetitionInfo, mails, actionInfo);
        }
*/
    }

    private Email getOrganizerIfMeetingOrElseCreatorExchangeEmail(EventWithRelations event) {
        return getOrganizerIfMeetingOrElseCreatorExchangeEmailSafe(event).get();
    }

    private Option<Email> getOrganizerIfMeetingOrElseCreatorExchangeEmailSafe(EventWithRelations event) {
        UidOrResourceId organizerOrCreator = getOrganizerIfMeetingOrElseCreator(event);
        if (organizerOrCreator.isResource()) {
            return Option.empty();
        }
        return Option.of(userManager.getLdEmailByUid(organizerOrCreator.getUid()));
    }

    private UidOrResourceId getOrganizerIfMeetingOrElseCreator(EventWithRelations event) {
        return event.getOrganizerIfMeetingOrElseCreator();
    }

    public void setEwsProxyWrapperForTest(EwsProxyWrapper ewsProxyWrapper) {
        this.ewsProxyWrapper = ewsProxyWrapper;
    }
}
