package ru.yandex.calendar.frontend.ews.exp;

import java.util.List;

import lombok.Value;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.misc.bender.annotation.BenderBindAllFields;

@Value
@BenderBindAllFields
public class ResourceParticipantChangesInfo {
    List<ResourceParticipantBriefInfo> newResources;
    List<ResourceParticipantBriefInfo> removedResources;
    List<ResourceParticipantBriefInfo> notChangedResources;

    public ResourceParticipantChangesInfo(List<ResourceParticipantBriefInfo> newResources,
                                          List<ResourceParticipantBriefInfo> removedResources,
                                          List<ResourceParticipantBriefInfo> notChangedResources) {
        val resources = StreamEx.of(newResources)
                .append(removedResources)
                .append(notChangedResources)
                .toImmutableList();
        validateYaTeamAndSyncWithExchange(resources);
        checkForDuplicates(resources);

        this.newResources = newResources;
        this.removedResources = removedResources;
        this.notChangedResources = notChangedResources;
    }

    private static List<ResourceParticipantBriefInfo> subtractSets(List<ResourceParticipantBriefInfo> initialList, List<ResourceParticipantBriefInfo> resourcesToRemove) {
        val resourcesToRemoveSet = StreamEx.of(resourcesToRemove)
                .map(ResourceParticipantBriefInfo::getResourceId)
                .toImmutableSet();
        return StreamEx.of(initialList)
                .remove(r -> resourcesToRemoveSet.contains(r.getResourceId()))
                .toImmutableList();
    }

    public static ResourceParticipantChangesInfo byMasterAndNewRecurrence(
            List<ResourceParticipantBriefInfo> masterResources, List<ResourceParticipantBriefInfo> recurrenceResources) {
        val added = subtractSets(recurrenceResources, masterResources);
        val removed = subtractSets(masterResources, recurrenceResources);
        val notChanged = subtractSets(recurrenceResources, added);

        return new ResourceParticipantChangesInfo(added, removed, notChanged);
    }

    public static ResourceParticipantChangesInfo byParticipantChangesInfo(List<ResourceParticipantBriefInfo> actualResources,
                                                                          List<ResourceParticipantBriefInfo> removed, List<Long> newResources) {
        val actual = StreamEx.of(actualResources).groupingBy(ResourceParticipantBriefInfo::getResourceId);
        val addedIds = StreamEx.of(newResources)
                .filter(actual::containsKey)
                .toImmutableSet();
        val added = StreamEx.of(addedIds)
                .flatMap(r -> actual.get(r).stream())
                .toImmutableList();
        val notChanged = StreamEx.of(actualResources)
                .remove(r -> addedIds.contains(r.getResourceId()))
                .toImmutableList();

        return new ResourceParticipantChangesInfo(added, removed, notChanged);
    }

    private void checkForDuplicates(List<ResourceParticipantBriefInfo> resources) {
        val distinctAmount = StreamEx.of(resources)
                .map(ResourceParticipantBriefInfo::getResourceId)
                .distinct()
                .count();
        if (distinctAmount != resources.size()) {
            throw new IllegalArgumentException("Duplicate resources.");
        }
    }

    private void validateYaTeamAndSyncWithExchange(List<ResourceParticipantBriefInfo> resources) {
        if (resources.stream().anyMatch(r -> !r.isSyncWithExchange())) {
            throw new IllegalArgumentException("There is not synced resource.");
        }
        if (resources.stream().anyMatch(r -> !r.isYandexTeam())) {
            throw new IllegalArgumentException("There is not yandex-team resource.");

        }
    }

    public List<ResourceParticipantBriefInfo> getAllResources() {
        return StreamEx.of(newResources).append(removedResources).append(notChangedResources).toImmutableList();
    }

    public ResourceParticipantChangesInfo filterNotAsync() {
        val newFiltered = removeAsyncWithExchange(newResources);
        val removedFiltered = removeAsyncWithExchange(removedResources);
        val notChangedFiltered = removeAsyncWithExchange(notChangedResources);
        return new ResourceParticipantChangesInfo(newFiltered, removedFiltered, notChangedFiltered);
    }

    private List<ResourceParticipantBriefInfo> removeAsyncWithExchange(List<ResourceParticipantBriefInfo> resources) {
        return StreamEx.of(resources)
                .remove(ResourceParticipantBriefInfo::isAsyncWithExchange)
                .toImmutableList();
    }
}
