package ru.yandex.calendar.frontend.ews.exp.async;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallbackWithoutResult;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.frontend.ews.EwsConflictingMeetingsNotCalculated;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.exp.ConflictingItem;
import ru.yandex.calendar.frontend.ews.exp.EventToCalendarItemConverter;
import ru.yandex.calendar.frontend.ews.exp.EwsExportRoutines;
import ru.yandex.calendar.frontend.ews.exp.EwsModifyingItemId;
import ru.yandex.calendar.frontend.ews.exp.ItemWithConflicts;
import ru.yandex.calendar.frontend.ews.imp.EwsImporter;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventData;
import ru.yandex.calendar.frontend.ews.proxy.EwsActionLogData;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.proxy.ExchangeIdLogData;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.beans.generated.DeletedEventResource;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.YtEwsExportingEvent;
import ru.yandex.calendar.logic.beans.generated.YtEwsExportingEventFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventResourceId;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.logic.event.MainEventInfo;
import ru.yandex.calendar.logic.event.archive.ArchiveManager;
import ru.yandex.calendar.logic.event.archive.DeletedEventDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.util.Environment;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.time.InstantInterval;

public class EwsResourceMeetingExporter {
    private static final Logger logger = LoggerFactory.getLogger(EwsResourceMeetingExporter.class);

    private static final String CONFLICTS_NOT_CALCULATED_REASON = "ConflictingMeetingsNotCalculated";

    @Autowired
    private DeletedEventDao deletedEventDao;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventToCalendarItemConverter eventToCalendarItemConverter;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private YtEwsExportingEventDao ytEwsExportingEventDao;
    @Autowired
    private EwsExportRoutines ewsExportRoutines;
    @Autowired
    private EwsImporter ewsImporter;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ArchiveManager archiveManager;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EwsMonitoring ewsMonitoring;

    public final DynamicProperty<Boolean> ignoreConflicts = new DynamicProperty<>(
            "ewsAsyncIgnoreConflicts", Environment.isProductionOrPre());

    public void exportMeetings() {
        final AtomicInteger counter = new AtomicInteger();

        final Function1V<YtEwsExportingEvent> exportF = m -> {
            String rid = RequestIdStack.current().getOrElse(RequestIdStack.generateId());
            RequestIdStack.Handle ridHandle = RequestIdStack.pushReplace(rid + "_" + counter.incrementAndGet());

            CalendarRequestHandle requestHandle = CalendarRequest.push(
                    ActionSource.EXCHANGE_ASYNCH, "Exporting meeting " + m.getExternalId());

            ActionInfo actionInfo = requestHandle.getActionInfo();
            try {
                exportMeeting(m, actionInfo);

            } finally {
                requestHandle.popSafely();
                ridHandle.popSafely();
            }
        };
        final ExecutorService executor = Executors.newFixedThreadPool(10);
        try {
            ytEwsExportingEventDao.findEventsScheduledBefore(events -> {
                try {
                    Function<YtEwsExportingEvent, Void> callableF =
                            MasterSlaveContextHolder.withStandardThreadLocalsF(exportF.<Void>asFunctionReturnNull());

                    ThreadLocalTimeout.check();
                    executor.invokeAll(events.map(callableF.bindF()));

                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    executor.shutdown();
                    throw new RuntimeException("abort");
                }
            }, Instant.now(), 30);
        } finally {
            executor.shutdownNow();
        }

    }

    void exportMeeting(YtEwsExportingEvent m, ActionInfo actionInfo) {
        try {
            exportMeetingInner(m, actionInfo);
        } catch (Exception e) {
            ewsMonitoring.reportIfEwsException(e);
            logger.error("Failed to export meeting {}: {}", m.getExternalId(), e);
            ExceptionUtils.rethrowIfTlt(e);

            String reason = e instanceof EwsConflictingMeetingsNotCalculated
                    ? CONFLICTS_NOT_CALCULATED_REASON
                    : e.getMessage();

            scheduleNextAttemptSafe(m, reason, actionInfo);

            throw ExceptionUtils.translate(e);
        }
    }

    private void scheduleNextAttemptSafe(YtEwsExportingEvent m, String failureReason, ActionInfo actionInfo) {
        try {
            YtEwsExportingEvent update = m.copy();
            update.setLastAttemptReqId(actionInfo.getRequestIdWithHostId());
            update.setLastAttemptTs(actionInfo.getNow());

            Instant nextAttempt = Random2.R.nextInstant(
                    actionInfo.getNow().plus(Duration.standardMinutes(5)),
                    actionInfo.getNow().plus(Duration.standardMinutes(15)));

            update.setNextAttemptTs(nextAttempt);
            update.setFailureReason(StringUtils.take(failureReason, YtEwsExportingEventFields.FAILURE_REASON.getCut().get()));

            ytEwsExportingEventDao.updateByExternalIdAndLastSubmitTs(update, m.getExternalId(), m.getLastSubmitTs());
        } catch (Exception e) {
            ewsMonitoring.reportIfEwsException(e);
            logger.error("Failed to schedule next attempt: {}", e);
        }
    }

    private void exportMeetingInner(YtEwsExportingEvent meeting, final ActionInfo actionInfo) {
        logger.debug("Going to export meeting {} to exchange", LogMarker.EXT_ID.format(meeting.getExternalId()));

        String externalId = meeting.getExternalId();
        ListF<MainEventInfo> mainEvents = findMainEventsHasResourceAsyncWithExchange(externalId);
        ListF<Long> eventIds = mainEvents.map(MainEventInfo::getEventIds).flatten();

        Instant syncTo = mainEvents.map(MainEventInfo::getMainEventLastUpdateTs)
                .minO().getOrElse(meeting.getLastSubmitTs());

        ListF<DeletedEventResource> deleted = findEventResourcesAsyncWithExchangeDeletedNotAfter(externalId, syncTo)
                .plus(findEventResourcesByEventIdsDeletedNotAfter(eventIds, syncTo)).stableUnique();
        Option<EwsConflictingMeetingsNotCalculated> notCalculatedException = Option.empty();

        for (MainEventInfo mainEvent : mainEvents) {
            for (long resourceId: mainEvent.findResourcesIdsAsyncWithExchange()) {
                ListF<CreatedMeeting> created;

                if (meeting.getFailureReason().isSome(CONFLICTS_NOT_CALCULATED_REASON)) {
                    created = convertCurrentToCreatedMeetings(resourceId, mainEvent);
                } else {
                    created = createMeeting(resourceId, mainEvent, actionInfo);
                    cancelOrDeclineExchangeCreatedOrDeleteMeetings(resourceId, mainEvent, actionInfo);
                }
                try {
                    processConflictingMeetings(resourceId, created, actionInfo);
                } catch (EwsConflictingMeetingsNotCalculated e) {
                    notCalculatedException = Option.of(e);
                }
            }
        }
        ListF<EventInfo> events = mainEvents.flatMap(MainEventInfo::getEventInfos);

        MapF<Long, Long> mainIdByEventId = events.toMap(EventInfo::getEventId, EventInfo::getMainEventId);
        MapF<Long, Option<Instant>> recurrenceIdByEventId = events.toMap(EventInfo::getEventId, EventInfo::getRecurrenceId);

        cancelOrDeclineExchangeCreatedOrDeleteMeetings(deleted.filterMap(d -> d.getExchangeId()
                .map(exchId -> new ExchangeIdLogData(exchId, UidOrResourceId.resource(d.getResourceId()),
                        new EventIdLogDataJson(d.getEventId(), mainIdByEventId.getOrElse(d.getEventId(), 0L),
                                externalId,
                                recurrenceIdByEventId.getO(d.getEventId()).getOrElse(Option.empty()).toOptional())))),
                    actionInfo);

        ytEwsExportingEventDao.deleteEventByExternalIdSubmittedNotAfter(externalId, syncTo);

        if (notCalculatedException.isPresent()) {
            throw notCalculatedException.get();
        }
    }

    private ListF<MainEventInfo> findMainEventsHasResourceAsyncWithExchange(String externalId) {
        ListF<MainEvent> mes = mainEventDao.findMainEventsByExternalId(new ExternalId(externalId));

        ListF<MainEventInfo> mainEvents = eventInfoDbLoader.getMainEventInfos(
                Option.empty(), mes, ActionSource.UNKNOWN);

        return mainEvents.filter(me -> me.findResourcesIdsAsyncWithExchange().isNotEmpty());
    }

    private ListF<DeletedEventResource> filterDeletedEventResources(ListF<DeletedEventResource> found) {
        SetF<Long> asyncIds = resourceDao.findResourceIdsAsyncWithExchangeByIds(
                found.map(DeletedEventResource::getResourceId)).unique();

        return found.filter(DeletedEventResource.getResourceIdF().andThen(asyncIds.containsF()));
    }

    private ListF<DeletedEventResource> findEventResourcesAsyncWithExchangeDeletedNotAfter(
            String externalId, Instant maxDeletionTs)
    {
        return filterDeletedEventResources(
                deletedEventDao.findEventResourcesByEventExternalIdDeletedNotAfter(externalId, maxDeletionTs));
    }

    private ListF<DeletedEventResource> findEventResourcesByEventIdsDeletedNotAfter(
            ListF<Long> eventIds, Instant maxDeletionTs)
    {
        return filterDeletedEventResources(
                deletedEventDao.findEventResourcesByEventIdsDeletedNotAfter(eventIds, maxDeletionTs));
    }

    private ListF<CreatedMeeting> convertCurrentToCreatedMeetings(final long resourceId, MainEventInfo mainEvent) {
        return mainEvent.getEventInfos().filterMap(event -> {
            Option<ResourceParticipantInfo> resourceP = event.findResourceParticipant(resourceId);

            if (resourceP.isPresent()) {
                String exchangeId = resourceP.get().getExchangeId().getOrThrow(
                        "No exchange id for event " + event.getEventId() + " and resource " + resourceId);

                Email email = resourceRoutines.getExchangeEmail(resourceP.get().getResource());

                return Option.of(new CreatedMeeting(resourceP.get(), email, exchangeId, event));
            }
            return Option.empty();
        });
    }

    ListF<CreatedMeeting> createMeeting(
            final long resourceId, final MainEventInfo mainEventInfo, final ActionInfo actionInfo)
    {
        return deleteStoredIfException(stored -> {
            MainEventInfo mainEvent = filterPastOccurrences(mainEventInfo, actionInfo);

            Option<EventInfo> master = mainEvent.getMasterEventInfos().singleO();
            Option<ResourceParticipantInfo> masterParticipant = master
                    .flatMapO(ei -> ei.findResourceParticipant(resourceId));

            Option<CreatedMeeting> createdMaster = Option.empty();

            if (masterParticipant.isPresent()) {
                boolean hasInstances = master.get().getEventAndRepetition()
                        .getFirstInstanceIntervalStartingAfter(master.get().getEvent().getStartTs()).isPresent();
                boolean hasRecurrences = mainEvent.getRecurrenceEventInfos()
                        .exists(ei -> ei.findResourceParticipant(resourceId).isPresent());

                if (!hasInstances && !hasRecurrences) {
                    logger.info("Skipping export event with no occurrences for resource {}", resourceId);
                    return;
                }
                createdMaster = Option.of(createMasterOrSingleMeeting(masterParticipant.get(), master.get(), actionInfo));
            }
            stored.addAll(createdMaster);

            for (EventInfo recurrence : mainEvent.getRecurrenceEventInfos()) {
                Option<ResourceParticipantInfo> participant = recurrence.findResourceParticipant(resourceId);

                if (participant.isPresent() && createdMaster.isPresent()) {
                    if (recurrence.getRecurrenceId().isSome(recurrence.getEvent().getStartTs())
                        && recurrence.getDuration().isEqual(master.get().getDuration()))
                    {
                        Instant recurrenceId = recurrence.getRecurrenceId().get();
                        logger.info("Skipping export recurrence with not changed time {}", recurrenceId);
                    } else {
                        stored.add(modifyOccurrence(participant.get(), createdMaster.get(), recurrence, actionInfo));
                    }
                } else if (participant.isPresent()) {
                    stored.add(createMasterOrSingleMeeting(participant.get(), recurrence, actionInfo));
                } else if (createdMaster.isPresent()) {
                    deleteOccurrence(createdMaster.get(), recurrence.getRecurrenceId().get(), actionInfo);
                }
            }
            saveUpdateExchangeIdsInTransaction(stored, actionInfo);
        }, actionInfo);
    }

    private MainEventInfo filterPastOccurrences(MainEventInfo mainEvent, ActionInfo actionInfo) {
        ListF<EventInfo> masters = mainEvent.getMasterEventInfos().map(event -> {
            ListF<Instant> exdates = event.getRepetitionInstanceInfo().getExdateStarts();
            ListF<Instant> pastExdates = exdates.filter(actionInfo.getNow().minus(event.getDuration())::isAfter);

            if (pastExdates.isNotEmpty()) {
                logger.info("Skipping export {} past exdates: {}", pastExdates.size(), pastExdates);
                return event.withRepetitionInfo(event.getRepetitionInstanceInfo().excludeExdates(pastExdates));
            } else {
                return event;
            }
        });
        Tuple2<ListF<EventInfo>, ListF<EventInfo>> pastAndNotRecurrences = mainEvent.getRecurrenceEventInfos()
                .partition(event -> event.getEvent().getEndTs().isBefore(actionInfo.getNow()));

        if (pastAndNotRecurrences.get1().isNotEmpty()) {
            logger.info("Skipping export {} past recurrences: {}",
                    pastAndNotRecurrences.get1().size(),
                    pastAndNotRecurrences.get1().map(e -> e.getRecurrenceId().get()));
        }
        return new MainEventInfo(mainEvent.getMainEvent(), masters.plus(pastAndNotRecurrences.get2()));
    }

    private void saveUpdateExchangeIdsInTransaction(final ListF<CreatedMeeting> meetings, final ActionInfo actionInfo) {
        transactionTemplate.execute(new TransactionCallbackWithoutResult() {
            protected void doInTransactionWithoutResult(TransactionStatus s) {
                for (CreatedMeeting m : meetings) {
                    eventResourceDao.saveUpdateExchangeId(
                            m.getResource().getId(), m.getEventId(), m.getExchangeId(), actionInfo);
                }
            }
        });
    }

    private void processConflictingMeetings(long resourceId, ListF<CreatedMeeting> meetings, final ActionInfo actionInfo) {
        ListF<ConflictingItem> exchangeCreated = Cf.arrayList();
        ListF<String> calendarCreatedSameOccurrenceIds = Cf.arrayList();

        for (CreatedMeeting meeting : meetings) {
            Option<ItemWithConflicts> itemO = ewsExportRoutines.getItemWithActualConflicts(
                    meeting.getExchangeEmail(), meeting.getExchangeId(),
                    meeting.getEvent().getRepetitionInstanceInfo(), actionInfo.getNow());

            ListF<ConflictingItem> conflicts = itemO
                    .getOrThrow("Failed to fetch created meeting " + formatForLog(meeting, meeting.getRecurrenceId()))
                    .getConflictingItems();

            ListF<ConflictingItem> calendarCreated = conflicts.filter(ConflictingItem.isFromCalendarF());

            Tuple2<ListF<ConflictingItem>, ListF<ConflictingItem>> p = calendarCreated.partition(
                    ConflictingItem.getExternalIdF().andThenEquals(meeting.getExternalId()));
            if (p.get1().isNotEmpty()) {
                logger.debug("Found {} calendar created conflicting occurrences with same external-id", p.get1().size());
            }
            if (p.get2().isNotEmpty()) {
                logger.debug("Found {} calendar created conflicting occurrences with different external-id: {}",
                        p.get2().size(), p.get2().map(conflictForLogF()));
            }
            if (!ignoreConflicts.get()) {
                deleteMeetings(p.get2().map(c -> c.getExchangeIdLogData(resourceId)), actionInfo);
            }
            calendarCreatedSameOccurrenceIds.addAll(p.get1().map(ConflictingItem.getExchangeIdF()));

            exchangeCreated.addAll(conflicts.filter(ConflictingItem.isFromCalendarF().notF()));
        }
        deleteMeetings(ewsProxyWrapper.getMasterAndSingleEvents(calendarCreatedSameOccurrenceIds)
                .map(item -> new ExchangeIdLogData(item.getItemId().getId(), UidOrResourceId.resource(resourceId),
                        new EventIdLogDataJson(0L, 0L, item.getUID(), EwsUtils.getRecurrenceId(item).toOptional()))), actionInfo);

        ListF<CalendarItemType> conflictingItems = ewsProxyWrapper.getMasterAndSingleEvents(
                exchangeCreated.map(ConflictingItem.getExchangeIdF()));

        if (conflictingItems.isNotEmpty()) {
            logger.debug("Found {} exchange created conflicting meetings: {}",
                    conflictingItems.size(), conflictingItems.map(calendarItemForLogF()));
        }
        for (CalendarItemType conflict: conflictingItems) {
            if (!ignoreConflicts.get()) {
                processExchangeCreatedConflictingMasterOrSingleMeeting(resourceId, conflict, actionInfo);
            }
        }
    }

    private void processExchangeCreatedConflictingMasterOrSingleMeeting(
            final long resourceId, final CalendarItemType meeting, final ActionInfo actionInfo)
    {
        final UidOrResourceId resId = UidOrResourceId.resource(resourceId);
        final ListF<ExchangeEventData> events = ewsImporter.convertEventAndItsRecurrences(resId, meeting, false);

        lockTransactionManager.lockAndDoInTransaction(ewsImporter.consLocker(events), new Function0V() {
            public void apply() {
                logger.debug("Going to import and than decline exchange created conflicting meeting: {}",
                        calendarItemForLogF().apply(meeting));

                ewsImporter.createOrUpdateOrDeleteEventsWithoutLockAndTransaction(resId, events, actionInfo, false);

                String exchangeId = meeting.getItemId().getId();
                Option<Event> event = eventRoutines.findEventByExchangeId(exchangeId);

                if (event.isPresent()) {
                    ListF<Long> eventIds = eventRoutines.findMasterAndSingleEventIds(event.get().getId());
                    ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIdsAndResourceIds(
                            eventIds, resId.getResourceIdO());

                    archiveManager.storeDeletedEventResources(eventResources, actionInfo);
                    eventResourceDao.deleteEventResourcesByIds(eventResources.map(EventResourceId::of));
                    lastUpdateManager.updateTimestampsAsync(event.get().getMainEventId(), actionInfo);
                } else {
                    logger.warn("No event found in calendar after import exchange event with id {}", exchangeId);
                }
                Option<Email> organizerEmail = events.first().getEventData().getInvData().getOrganizerEmail();
                Option<PassportUid> organizerUid = Cf2.flatBy2(userManager.getUidsByEmails(organizerEmail)).get2().singleO();

                Language lang = settingsRoutines.getDefaultLanguage();
                if (organizerUid.isPresent()) {
                    lang = settingsRoutines.getLanguage(organizerUid.get());
                }
                cancelOrDeclineMeetingInExchange(resourceId, exchangeId, event.map(Event.getIdF()), lang,
                        new EwsActionLogData(UidOrResourceId.resource(resourceId),
                                new EventIdLogDataJson(0L, 0L, meeting.getUID(), EwsUtils.getRecurrenceId(meeting).toOptional()), actionInfo));
            }
        });
    }

    private static final NameI18n CANNOT_BE_BOOKED_I18N = new NameI18n(
            "Выбранная вами переговорка не может быть забронирована.",
            "Chosen meeting room can not be booked.");

    private static final NameI18n CHOOSE_ANOTHER_ONE_I18N = new NameI18n(
            "Пожалуйста, <a href=\"$\">выберите другую переговорку</a>.",
            "Please <a href=\"$\">choose another one</a>.");

    private void cancelOrDeclineMeetingInExchange(
            long resourceId, String exchangeId, Option<Long> eventId, Language lang, EwsActionLogData actionInfo)
    {
        Option<String> calendarLinkText = Option.empty();
        if (eventId.isPresent()) {
            long officeId = resourceRoutines.getResourcesByIds(Cf.list(resourceId)).single().getOfficeId();

            String url = svcRoutines.getCalendarUrlForDomain(PassportAuthDomain.YANDEX_TEAM_RU) + "/event";
            url = UrlUtils.addParameter(url, "event_id", eventId.get(), "suggest_office_id", officeId);
            calendarLinkText = Option.of(CHOOSE_ANOTHER_ONE_I18N.getName(lang).replace("$", url));
        }
        String replyHtml = "<html><body>"
                + CANNOT_BE_BOOKED_I18N.getName(lang) + " " + calendarLinkText.getOrElse("")
                + "</body></html>";

        ewsProxyWrapper.cancelOrDeclineMeetingWithReply(exchangeId, replyHtml, actionInfo);
    }

    private CreatedMeeting createMasterOrSingleMeeting(
            final ResourceParticipantInfo participant, final EventInfo meeting, ActionInfo actionInfo)
    {
        final Email email = resourceRoutines.getExchangeEmail(participant.getResource());
        final CalendarItemType data = convertToCalendarItem(meeting);

        return deleteStoredIfException(created -> {
            EwsActionLogData logData = new EwsActionLogData(
                    UidOrResourceId.resource(participant), new EventIdLogDataJson(meeting), actionInfo);

            created.add(new CreatedMeeting(participant, email, ewsProxyWrapper.createEvent(email, data, logData), meeting));
            deleteOccurrences(created.single(), meeting.getRepetitionInstanceInfo().getExdateStarts(), actionInfo);
        }, actionInfo).single();
    }

    private CreatedMeeting modifyOccurrence(
            ResourceParticipantInfo participant, CreatedMeeting master, EventInfo recurrence, ActionInfo actionInfo)
    {
        String occurrenceExchangeId = getOccurrenceId(master, recurrence.getRecurrenceId().get());
        EwsModifyingItemId id = EwsModifyingItemId.fromExchangeId(occurrenceExchangeId);

        ListF<ItemChangeDescriptionType> changes = convertToChangesDescription(master.getEvent(), recurrence);

        EwsActionLogData logData = new EwsActionLogData(
                UidOrResourceId.resource(participant), new EventIdLogDataJson(recurrence), actionInfo);

        String exchangeId = ewsProxyWrapper.updateItem(id, changes, logData)
                .getOrThrow("Failed to modify occurrence " + formatForLog(master, recurrence.getRecurrenceId()));

        return new CreatedMeeting(participant, master.getExchangeEmail(), exchangeId, recurrence);
    }

    private void deleteOccurrence(CreatedMeeting master, Instant occurrenceStart, ActionInfo actionInfo) {
        deleteOccurrences(master, Cf.list(occurrenceStart), actionInfo);
    }

    private void deleteOccurrences(CreatedMeeting masters, ListF<Instant> occurrenceStarts, ActionInfo actionInfo) {
        for (Instant start : occurrenceStarts) {
            ExchangeIdLogData occurrenceExchangeId = new ExchangeIdLogData(getOccurrenceId(masters, start),
                    UidOrResourceId.resource(masters.getResourceId()), new EventIdLogDataJson(masters.getEvent()));

            deleteMeetings(Cf.list(occurrenceExchangeId), actionInfo);
        }
    }

    private void deleteMeetings(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        ewsProxyWrapper.deleteEvents(exchangeIds, actionInfo);
    }

    private void cancelOrDeclineExchangeCreatedOrDeleteMeetings(
            long resourceId, MainEventInfo mainEvent, ActionInfo actionInfo)
    {
        cancelOrDeclineExchangeCreatedOrDeleteMeetings(mainEvent.getEventInfos()
                .filterMap(e -> e.findResourceParticipant(resourceId).flatMapO(ResourceParticipantInfo::getExchangeId)
                        .map(exchId -> new ExchangeIdLogData(exchId, UidOrResourceId.resource(resourceId), new EventIdLogDataJson(e)))),
                actionInfo);
    }

    private void cancelOrDeclineExchangeCreatedOrDeleteMeetings(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        ewsProxyWrapper.cancelOrDeclineExchangeCreatedOrDeleteMeetings(exchangeIds, actionInfo);
    }

    private String getOccurrenceId(CreatedMeeting master, Instant occurrenceStart) {
        InstantInterval interval = new InstantInterval(occurrenceStart, occurrenceStart);
        ListF<String> instanceIds = ewsProxyWrapper.findInstanceEventIds(master.getExchangeEmail(), interval);

        return ewsProxyWrapper.findRecurringMasterIdsByInstanceIds(instanceIds)
                .findBy2(Cf2.isSomeF(master.getExchangeId()))
                .getOrThrow("Occurrence not found " + formatForLog(master, Option.of(occurrenceStart))).get1();
    }

    private ListF<CreatedMeeting> deleteStoredIfException(
            Function1V<ArrayListF<CreatedMeeting>> callback, ActionInfo actionInfo)
    {
        val stored = new ArrayListF<CreatedMeeting>();
        try {
            callback.apply(stored);
            return stored;

        } catch (Exception e) {
            try {
                deleteMeetings(stored.map(s -> new ExchangeIdLogData(s.getExchangeId(), UidOrResourceId.resource(s.getResourceId()), new EventIdLogDataJson(s.getEvent()))), actionInfo);
            } catch (Exception ex) {
                logger.error("Failed to delete stored events: {}", ex);
            }
            throw ExceptionUtils.translate(e);
        }
    }

    private Function<ConflictingItem, String> conflictForLogF() {
        return conflictOrCalendarItemForLogF().compose(Either.<ConflictingItem, CalendarItemType>leftF());
    }

    private Function<CalendarItemType, String> calendarItemForLogF() {
        return conflictOrCalendarItemForLogF().compose(Either.<ConflictingItem, CalendarItemType>rightF());
    }

    private Function<Either<ConflictingItem, CalendarItemType>, String> conflictOrCalendarItemForLogF() {
        return e -> {
            Tuple2List<Object, Object> fields = Tuple2List.fromPairs(
                    "ext-id", e.fold(ConflictingItem.getExternalIdF(), EwsUtils.calendarItemUidF()),
                    "start-ts", e.fold(ConflictingItem.getStartF(), EwsUtils.calendarItemStartF()),
                    "type", e.fold(ConflictingItem.getTypeF(), EwsUtils.calenderItemTypeF()));

            return "{" + fields.mkString("=", ", ") + "}";
        };
    }

    private static String formatForLog(CreatedMeeting master, Option<Instant> recurrenceId) {
        return "{resource-id=" + master.getResourceId() + ", recurrence-id=" + recurrenceId + "}";
    }

    private CalendarItemType convertToCalendarItem(EventInfo e) {
        return eventToCalendarItemConverter.convertToCalendarItem(
                e.getEventWithRelations(), e.getRepetitionInstanceInfo());
    }

    private ListF<ItemChangeDescriptionType> convertToChangesDescription(EventInfo master, EventInfo recurrence) {
        return eventToCalendarItemConverter.convertToChangeDescriptions(
                recurrence.getEventWithRelations(),
                recurrence.getRepetitionInstanceInfo(),
                eventChangesFinder.getEventChangesInfoForExchangeForNewRecurrence(
                        master.getEvent(), recurrence.getEvent()));
    }
}
