package ru.yandex.calendar.frontend.ews.hook;

import java.net.InetAddress;
import java.util.Collections;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;

import ru.yandex.commune.a3.security.AuthenticationType;
import ru.yandex.commune.a3.security.UnauthenticatedException;
import ru.yandex.commune.dynproperties.DynamicProperty;

/**
 * Поскольку внедрение Tvm для Exchange довольно сложно,  придется немного попотеть и фильтровать по IP,
 * с которых приходит запрос на Exchange. При этом производится приведение адресов к ip6 формату, чтобы
 * не зависеть от смены формата (например, вдруг адреса начнут приходить в верхнем регистре, а не нижнем,
 *  и с компрессией нулевых октетов), чтобы не получить в рантайме внезапный отказ для запросов от Exchange
 *  в случае каких-либо сетевых обновлений.
 */
@Slf4j
public class EwsFirewallImpl implements EwsFirewall {
    private final DynamicProperty<EwsFirewallConfig> forbiddenSrc = new DynamicProperty<>("ews.firewall.config",
            new EwsFirewallConfig(false, Collections.emptyList()));

    public void checkSrc(Optional<String> ip) {
        val config = forbiddenSrc.get();
        ip.ifPresentOrElse(x -> checkSrc(x, config), () -> handleUnexpectedIp("UNKNOWN", config));
    }

    static void checkSrc(String ip, EwsFirewallConfig config) {
        val actualIp = parse(ip);

        actualIp.filter(address -> StreamEx.of(config.getAllowedSrc())
                        .map(EwsFirewallImpl::parse)
                        .flatMap(Optional::stream)
                        .anyMatch(address::equals))
                .ifPresentOrElse(address -> {}, () -> handleUnexpectedIp(ip, config));
    }

    private static void handleUnexpectedIp(String ip, EwsFirewallConfig config) {
        if (config.isRejectUnknown()) {
            throw new UnauthenticatedException(AuthenticationType.TVM, "A request without tvm header came");
        }
        log.warn("A request came from the unexpected ip: {}", ip);
    }

    private static Optional<InetAddress> parse(String ip) {
        try {
            return Optional.of(InetAddress.getByName(ip));
        } catch (Exception e) {
            return Optional.empty();
        }
    }
}
