package ru.yandex.calendar.frontend.ews.hook;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.NotificationType;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.frontend.ews.imp.EwsImporter;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.misc.db.masterSlave.MasterSlaveUnitUnavailableException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

/**
 * @author akirakozov
 */
public class EwsNotificationEventHandler {
    private static final Logger logger = LoggerFactory.getLogger(EwsNotificationEventHandler.class);

    @Autowired
    protected EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    protected EwsImporter ewsImporter;
    @Autowired
    private EwsMonitoring ewsMonitoring;
    @Autowired
    private EwsAliveHandler ewsAliveHandler;

    public void handle(UidOrResourceId subjectId, NotificationType notifications, ActionInfo actionInfo) {
        if (ewsAliveHandler.isEwsAlive()) {
            EwsNotificationResolver.resolveEvents(notifications)
                    .forEach(item -> handle(subjectId, item.itemId, item.eventType, actionInfo));
        }
    }

    public void handle(final UidOrResourceId subjectId, final String exchangeId, final EwsNotificationEventType eventType, ActionInfo actionInfo) {
        try {
            logger.info(LogMarker.EXCHANGE_ID.format(exchangeId));
            logger.debug("Event is being handled by " + subjectId + "'s subscription (" + eventType.getName() + ")");
            switch (eventType) {
            case CREATE:
                handleCreate(subjectId, exchangeId, actionInfo);
                break;
            case MOVED:
            case DELETE:
                handleDelete(subjectId, exchangeId, actionInfo);
                break;
            case MODIFIED:
                handleModified(subjectId, exchangeId, actionInfo);
                break;
            default:
                throw new IllegalStateException();
            }
        } catch (MasterSlaveUnitUnavailableException | ThreadLocalTimeoutException e) {
            throw e;
        } catch (Exception e) {
            logger.warn(e, e);
            ewsMonitoring.reportIfEwsException(e);
        }
    }

    private void handleCreate(UidOrResourceId subjectId, String exchangeId, ActionInfo actionInfo) {
        Option<CalendarItemType> calendarItemO = ewsProxyWrapper.getEvent(exchangeId);
        if (calendarItemO.isPresent()) {
            ewsImporter.createOrUpdateEventWithRecurrences(subjectId, calendarItemO.get(), actionInfo, false);
        } else {
            warnAboutNotFoundEvent(exchangeId);
        }
    }

    private void handleDelete(UidOrResourceId subjectId, String exchangeId, ActionInfo actionInfo) {
        ewsImporter.removeEvent(subjectId, exchangeId, actionInfo);
    }

    private void handleModified(UidOrResourceId subjectId, String exchangeId, ActionInfo actionInfo) {
        Option<CalendarItemType> calendarItemO = ewsProxyWrapper.getEvent(exchangeId);
        if (calendarItemO.isPresent()) {
            ewsImporter.createOrUpdateEventWithRecurrences(subjectId, calendarItemO.get(), actionInfo, false);
        } else {
            warnAboutNotFoundEvent(exchangeId);
        }
    }


    protected void warnAboutNotFoundEvent(String exchangeId) {
        logger.warn("Event not found by exchange id: " + exchangeId);
    }
}
