package ru.yandex.calendar.frontend.ews.hook;

import javax.xml.bind.JAXBElement;

import com.microsoft.schemas.exchange.services._2006.types.BaseNotificationEventType;
import com.microsoft.schemas.exchange.services._2006.types.BaseObjectChangedEventType;
import com.microsoft.schemas.exchange.services._2006.types.FolderIdType;
import com.microsoft.schemas.exchange.services._2006.types.ItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.MovedCopiedEventType;
import com.microsoft.schemas.exchange.services._2006.types.NotificationType;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class EwsNotificationResolver {
    private static final Logger logger = LoggerFactory.getLogger(EwsNotificationResolver.class);

    public static ListF<ItemEvent> resolveEvents(NotificationType notification) {
        MapF<String, ListF<String>> eventsByItemId = groupEventsByItem(notification);
        ListF<ItemEvent> itemEvents = Cf.arrayList();

        for (String itemId : eventsByItemId.keySet()) {
            // even though notification may contain multiple events for one item,
            // we can synchronize by performing at most one operation per item
            Option<EwsNotificationEventType> eventO = resolveEventSequence(eventsByItemId.getTs(itemId));
            if (eventO.isPresent()) {
                itemEvents.add(new ItemEvent(itemId, eventO.get()));
            }
        }

        return itemEvents.sortedBy(e -> e.eventType.getProcessPriority()); // CAL-6893
    }

    private static MapF<String, ListF<String>> groupEventsByItem(NotificationType notification) {
        MapF<String, ListF<String>> eventsByItemId = Cf.hashMap();

        ListF<JAXBElement<? extends BaseNotificationEventType>> ntfEvents =
                Cf.x(notification.getCopiedEventOrCreatedEventOrDeletedEvent());
        for (JAXBElement<? extends BaseNotificationEventType> ntfEvent : ntfEvents) {
            String ntfEventName = ntfEvent.getName().getLocalPart();
            BaseNotificationEventType bnet = ntfEvent.getValue();
            logger.debug("Ntf event type: " + ntfEventName);
            logger.debug("Message type: " + bnet.toString());
            if (bnet instanceof BaseObjectChangedEventType) {
                BaseObjectChangedEventType bocet = (BaseObjectChangedEventType) bnet;
                debugField("folder id", bocet.getFolderId());
                debugField("parent folder id", bocet.getParentFolderId());
                debugField("item id", bocet.getItemId());

                // We should take old item id for moved and modified events,
                // and item id for created, deleted events
                ItemIdType itemId;
                if (bnet instanceof MovedCopiedEventType) {
                    MovedCopiedEventType mcet = (MovedCopiedEventType) bnet;
                    debugField("old folder id", mcet.getOldFolderId());
                    debugField("old parent folder id", mcet.getOldParentFolderId());
                    debugField("old item id", mcet.getOldItemId());
                    itemId = mcet.getOldItemId();
                } else {
                    itemId = bocet.getItemId();
                }

                if (itemId != null) {
                    eventsByItemId.
                            getOrElseUpdate(itemId.getId(), Cf.<String>arrayList()).add(ntfEventName);
                } else {
                    logger.debug("Not item change: " + bnet.toString());
                }
            } else {
                logger.debug("Unsupported event notification type: " + bnet.toString());
            }
        } // for notification events
        return eventsByItemId;
    }

    private static Option<EwsNotificationEventType> resolveEventSequence(ListF<String> eventSequence) {
        boolean created = eventSequence.exists(EwsNotificationEventType.CREATE.nameEq);
        boolean deleted = eventSequence.exists(EwsNotificationEventType.DELETE.nameEq);
        boolean modified = eventSequence.exists(EwsNotificationEventType.MODIFIED.nameEq);
        boolean moved = eventSequence.exists(EwsNotificationEventType.MOVED.nameEq);

        // achtung! tricky logic!
        if (created && (deleted || moved)) {
            return Option.empty();
        } else if (created) {
            return Option.of(EwsNotificationEventType.CREATE);
        } else if (deleted) {
            return Option.of(EwsNotificationEventType.DELETE);
        } else if (moved) {
            return Option.of(EwsNotificationEventType.MOVED);
        } else if (modified) {
            return Option.of(EwsNotificationEventType.MODIFIED);
        } else {
            throw new IllegalArgumentException("All notification event types are unknown!");
        }
    }

    public static class ItemEvent {
        public final String itemId;
        public final EwsNotificationEventType eventType;

        public ItemEvent(String itemId, EwsNotificationEventType eventType) {
            this.itemId = itemId;
            this.eventType = eventType;
        }
    }

    private static void debugField(String name, FolderIdType value) {
        logger.debug(name + ": " + (value != null ? value.getId() : "{null}"));
    }

    private static void debugField(String name, ItemIdType value) {
        logger.debug(name + ": " + (value != null ? value.getId() : "{null}"));
    }

}
