package ru.yandex.calendar.frontend.ews.hook;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.AddrUtils;
import ru.yandex.calendar.frontend.HeaderUtils;
import ru.yandex.commune.a3.security.UnauthenticatedException;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.web.servlet.HttpServletResponseX;
import ru.yandex.misc.xml.dom.DomUtils;

@SuppressWarnings("serial")
@Slf4j
public class NotificationEwsServlet2 extends HttpServlet {
    private final static String EWS_HOOK_REQUESTS_COUNTER_PREFIX = "application.request.ews.hook";
    @Autowired
    private MeterRegistry registry;
    @Autowired
    private NotificationExchangeWebService notificationExchangeWebService;
    @Autowired
    private EwsFirewall ewsFirewall;

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) {
        try {
            checkSrc(req);
            val content = SoapXmlizer.parseSoapRequest(req.getInputStream());
            val sendNotificationResponseType = EwsXmlizer.unmarshalResponse(content);

            val sendNotificationResultType =
                    notificationExchangeWebService.sendNotification(sendNotificationResponseType);

            val document = EwsXmlizer.marshalResult(sendNotificationResultType);

            val respXml = SoapXmlizer.serializeSoapResponse(document);
            val respBytes = DomUtils.I.writeElementToBytes(respXml);

            HttpServletResponseX.wrap(resp).writeContent(respBytes, "text/xml; charset=UTF-8");
            incrementMetrics(req, "success");
        } catch (Exception e) {
            incrementMetrics(req, "error");
            throw ExceptionUtils.translate(e);
        }
    }

    private void checkSrc(HttpServletRequest req) {
        val ip = AddrUtils.getIp(HeaderUtils.getHeaderReader(req));
        try {
            ewsFirewall.checkSrc(ip);
        } catch (UnauthenticatedException | UnauthorizedException e) {
            log.error("Auth exception for IP {}", ip.orElse("UNKNOWN"), e);
            throw e;
        }
    }

    private void incrementMetrics(HttpServletRequest req, String status) {
        registry.counter(String.join(".", EWS_HOOK_REQUESTS_COUNTER_PREFIX, req.getRequestURI(), status)).increment();
    }
}
