package ru.yandex.calendar.frontend.ews.imp;

import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.datatype.XMLGregorianCalendar;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemTypeType;
import com.microsoft.schemas.exchange.services._2006.types.DeletedOccurrenceInfoType;
import com.microsoft.schemas.exchange.services._2006.types.EmailAddressType;
import com.microsoft.schemas.exchange.services._2006.types.ItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfDeletedOccurrencesType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseTypeType;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.ews.EwsBadEmailException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeEmailManager;
import ru.yandex.calendar.frontend.ews.ExtendedCalendarItemProperties;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.sync.IgnoreReason;
import ru.yandex.calendar.frontend.ews.sync.IgnoredEventManager;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.LastUpdateManager;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventAttachedUser;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceStatusChecker;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.ExchangeEventSynchData;
import ru.yandex.calendar.logic.event.IcsEventSynchData;
import ru.yandex.calendar.logic.event.SequenceAndDtStamp;
import ru.yandex.calendar.logic.event.UpdateInfo;
import ru.yandex.calendar.logic.event.archive.DeletedEventDao;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.EventUserUpdate;
import ru.yandex.calendar.logic.event.model.ParticipantsOrInvitationsData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.repetition.UnsupportedRepetitionException;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.ResourcesChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UsersChangesJson;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.notification.xiva.notify.XivaMobileEwsNotifier;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData.Meeting;
import ru.yandex.calendar.logic.update.LockHandle;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.commune.util.serialize.ToMultilineSerializer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

/**
 * @see ru.yandex.calendar.logic.ics.imp.IcsEventImporter
 */
@Slf4j
public class EwsImporter {
    @Value("${ews.domain}")
    private String ewsDomain;

    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventInstanceStatusChecker eventInstanceStatusChecker;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private IgnoredEventManager ignoredEventManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ToMultilineSerializer toMultilineSerializer;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventUserRoutines eventUserRoutines;
    @Autowired
    private LastUpdateManager lastUpdateManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;
    @Autowired
    private XivaMobileEwsNotifier xivaMobileNotifier;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private ExchangeEmailManager exchangeEmailManager;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private DeletedEventDao deletedEventDao;
    @Autowired
    private EventsLogger eventsLogger;

    private final Tuple2List<Pattern, Function<String, Email>> patternsAndActions =
            new Tuple2List<String, Function<String, Email>>(Cf.list())
                    .plus1("^'(.*?)'$", Email::new)
                    .plus1("^/o=yandex.*/cn=([^/]*?)$", login -> userManager.getYtUserEmailByLogin(login).get())
                    .plus1("\\[name=([^;]*?);", Email::new)
                    .map1(Pattern::compile);


    public ListF<EwsImportStatus> createOrUpdateEventWithRecurrences(
            final UidOrResourceId subjectId, CalendarItemType calendarItem,
            final ActionInfo actionInfo, final boolean dryRun)
    {

        ListF<ParticipantId> subjects = ExchangeEventDataConverter.getOrganizerEmailSafe(calendarItem)
                .filterMap(userManager::getUserByEmail).map(yu -> ParticipantId.yandexUid(yu.getUid()))
                .plus(subjectId.toParticipantId());

        boolean isExportedWithEws = eventRoutines.getMainEventBySubjectsAndExternalId(subjects, calendarItem.getUID())
                .filterMap(MainEvent::getIsExportedWithEws).getOrElse(false);

        if (subjectId.isResource() && !isExportedWithEws) {  // CAL-8188
            return Cf.list();
        }

        final ListF<ExchangeEventData> events = convertEventAndItsRecurrences(subjectId, calendarItem, dryRun);

        Function0<ListF<EwsImportResult>> importF =
                () -> createOrUpdateOrDeleteEventsInner(subjectId, events, actionInfo, dryRun);

        return importWithLockInTransactionInner(consLocker(events), importF, actionInfo);
    }

    public EwsImportStatus createOrUpdateEventForTest(
            final UidOrResourceId subjectId, CalendarItemType calendarItem,
            final ActionInfo actionInfo, final boolean dryRun)
    {
        fixEmailsIfPossible(calendarItem);
        final ExchangeEventData event = convertExchangeData(subjectId, calendarItem, dryRun);

        Function0<ListF<EwsImportResult>> importF =
                () -> Cf.list(createOrUpdateEventInner(subjectId, event, actionInfo, dryRun));

        return importWithLockInTransactionInner(consLocker(Cf.list(event)), importF, actionInfo).single();
    }

    public EwsImportStatus removeEvent(
            final UidOrResourceId subjectId, final String exchangeId, final ActionInfo actionInfo)
    {
        final Option<Event> event = findEventByExchangeIdForDelete(exchangeId, actionInfo);

        if (!event.isPresent()) return EwsImportStatus.SKIPPED;

        boolean isExportedWithEws =
                mainEventDao.findMainEventById(event.get().getMainEventId()).getIsExportedWithEws().getOrElse(false);

        if (subjectId.isResource() && !isExportedWithEws) {  // CAL-8212
            return EwsImportStatus.SKIPPED;
        }

        Function0<ListF<EwsImportResult>> importF = () -> Cf.list(removeEventInner(subjectId, event.get(), actionInfo));

        return importWithLockInTransactionInner(
                () -> updateLock2.lockEvent(event.get().getId()), importF, actionInfo).single();
    }

    public ListF<EwsImportStatus> createOrUpdateOrDeleteEventsWithoutLockAndTransaction(
            UidOrResourceId subjectId, ListF<ExchangeEventData> events, ActionInfo actionInfo, boolean dryRun)
    {
        return createOrUpdateOrDeleteEventsInner(subjectId, events, actionInfo, dryRun)
                .map(EwsImportResult.getStatusF());
    }

    public ListF<ExchangeEventData> convertEventAndItsRecurrences(
            UidOrResourceId subjectId, CalendarItemType calendarItemType, boolean dryRun)
    {
        ListF<CalendarItemType> items = ewsProxyWrapper.getEventAndItsModifiedOccurrences(calendarItemType);

        ListF<XMLGregorianCalendar> cancelledRecurrenceIds = items.drop(1).filterMap(item ->
                EwsUtils.isCancelled(item) ? Option.ofNullable(item.getRecurrenceId()) : Option.empty());

        if (cancelledRecurrenceIds.isNotEmpty()) {
            if (items.first().getDeletedOccurrences() == null) {
                items.first().setDeletedOccurrences(new NonEmptyArrayOfDeletedOccurrencesType());
            }
            items.first().getDeletedOccurrences().getDeletedOccurrence()
                    .addAll(cancelledRecurrenceIds.map(recurrenceId -> {
                        DeletedOccurrenceInfoType occurrence = new DeletedOccurrenceInfoType();
                        occurrence.setStart(recurrenceId);

                        return occurrence;
                    }));
        }
        return items.map(convertExchangeDataF(subjectId, dryRun));
    }

    public Function0<LockHandle> consLocker(ListF<ExchangeEventData> datas) {
        return () -> updateLock2.lockForUpdate(
                datas.filterMap(d -> d.getEventData().getExternalId()).map(LockResource::event).stableUnique());
    }

    private ListF<EwsImportStatus> importWithLockInTransactionInner(
            Function0<LockHandle> locker, Function0<ListF<EwsImportResult>> importF, ActionInfo actionInfo)
    {
        ListF<EwsImportResult> results = lockTransactionManager.lockAndDoInTransaction(locker, importF);

        xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(
                results.flatMap(EwsImportResult.getAffectedLayerIdsF()), actionInfo);

        return results.map(EwsImportResult.getStatusF());
    }

    private ListF<EwsImportResult> createOrUpdateOrDeleteEventsInner(
            UidOrResourceId subjectId, ListF<ExchangeEventData> events, ActionInfo actionInfo, boolean dryRun)
    {
        Option<String> externalId = events.foldLeft(Option.empty(),
                (acc, e) -> acc.orElse(e.getEventData().getExternalId()));

        if (externalId.isPresent() && eventRoutines.findDeletedMasterExistsByExternalIdAndParticipantEmails(
                externalId.get(), events.flatMap(e -> e.getEventData().getParticipantEmails())))
        {
            log.info("Skipping group {} that looks like already deleted", externalId.get());
            return events.map(Function.constF(new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId)));
        }

        val master = events.findNot(e -> e.getRecurrenceId().isPresent());
        Option<EventInstanceStatusInfo> masterStatus = master
                .map(m -> getEventSynchronizationData(subjectId, m.getEventData(), actionInfo));

        if (masterStatus.exists(EventInstanceStatusInfo::isFound)) { // CAL-7436
            val emailF = Cf2.f0(
                    () -> exchangeEmailManager.getExchangeEmailFromSubscriptionOrBySubjectId(subjectId)).memoize();

            val recurrenceEvents = eventDao.findRecurrenceEventsByMainId(
                    eventDao.findMainEventIdByEventId(masterStatus.get().getEventId()).get());

            val occurrenceIds = recurrenceEvents
                    .filterMap(Event.getRecurrenceIdF())
                    .filterNot(events.filterMap(ExchangeEventData::getRecurrenceId).unique().containsF())
                    .filterMap(i -> ewsProxyWrapper.getOccurrenceIdByTimeInterval(
                            emailF.apply(), new InstantInterval(i, i), master.get().getCalendarItem().getUID()));

            events = events.plus(ewsProxyWrapper
                    .getEvents(occurrenceIds.map(ItemIdType::getId))
                    .map(convertExchangeDataF(subjectId, dryRun)));
        }
        val indexed = events.zipWithIndex()
                .sortedBy1(Comparator.<ExchangeEventData>constEqualComparator()
                        .thenComparing(e -> e.getRecurrenceId().isPresent()));

        val results = Cf.toArrayList(events.map(
                Function.constF(new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId))));

        val email = Lazy.withSupplier(() -> subjectId.isResource()
                ? resourceRoutines.getResourceEmailByResourceId(subjectId.getResourceId())
                : settingsRoutines.getSettingsByUid(subjectId.getUid()).getEmail());

        for (Tuple2<ExchangeEventData, Integer> t : indexed) {
            val data = t.get1();
            if (EwsUtils.isCancelled(data.getCalendarItem())) {
                if (dryRun) {
                    log.warn("Dry-run mode is on, not trying to delete event");
                } else {
                    val event = findEventByExchangeIdForDelete(
                            data.getEventData().getExchangeData().get().getExchangeId(), actionInfo);
                    if (event.isPresent()) {
                        if (eventUserDao.findOrganizerByEventId(event.get().getId())
                                .map(id -> subjectId.getUidO().containsTs(id)).getOrElse(true))
                        {
                            results.set(t.get2(), removeEventInner(subjectId, event.get(), actionInfo));
                        } else {
                            UsersChangesJson usersChanges = UsersChangesJson.empty();
                            ResourcesChangesJson resourcesChanges = ResourcesChangesJson.empty();

                            if (subjectId.isResource()) {
                                val resources =
                                        eventResourceDao.findEventResourceByEventIdAndResourceId(
                                                event.get().getId(), subjectId.getResourceId());
                                eventRoutines.storeAndDeleteEventResources(resources, actionInfo);

                                resourcesChanges = new ResourcesChangesJson(resources.map(er ->
                                        new ResourcesChangesJson.ResourceIdentJson(er.getResourceId(), email.get())), Cf.list());

                            } else {
                                val users = eventUserDao.findEventUserByEventIdAndUid(
                                        event.get().getId(), subjectId.getUid());
                                eventRoutines.storeAndDeleteEventUsers(users, actionInfo);

                                usersChanges = UserRelatedChangesJson.find(
                                        email.get(), users.toOptional(), Optional.empty(), Optional.empty()).userChanges;
                            }

                            eventsLogger.log(EventChangeLogEvents.updated(
                                    ActorId.userOrResource(subjectId), consLogId(event.get(), data.getEventData()),
                                    EventChangesJson.empty().withUsers(usersChanges).withResources(resourcesChanges)), actionInfo);
                        }
                    }
                }
            } else {
                results.set(t.get2(), createOrUpdateEventInner(subjectId, data, actionInfo, dryRun));
            }
        }
        return results;
    }

    private void logCalendarItemData(CalendarItemType calItem) {
        log.debug("Importing exchange event\n{}", toMultilineSerializer.serialize(calItem));

        log.debug(LogMarker.EXCHANGE_ID.format(calItem.getItemId().getId()));
        log.debug(LogMarker.EXT_ID.format(StringUtils.defaultIfEmpty(calItem.getUID(), "?")));
        CalendarItemTypeType itemType = calItem.getCalendarItemType();
        log.debug("Item type: {}", itemType);
        log.debug("Event name: {}", calItem.getSubject());
        final Option<Instant> startTs = EwsUtils.toInstantO(calItem.getStart());
        log.debug("Event start timestamp: {}", startTs);
        try {
            final Option<Email> organizerEmail = ExchangeEventDataConverter.getOrganizerEmailSafe(calItem);
            log.debug("Organizer: {}", organizerEmail);
        } catch (Exception e) {
            log.warn("Could not get organizer email:", e);
        }
        try {
            final ListF<Email> attendeeEmails = ExchangeEventDataConverter.getAttendeeEmails(calItem);
            log.debug("Attendees: {}", toMultilineSerializer.serialize(attendeeEmails));
        } catch (Exception e) {
            log.warn("Could not get attendee emails:", e);
        }
    }

    private Option<Event> findEventByExchangeIdForDelete(String exchangeId, ActionInfo actionInfo) {
        Option<Event> eventO = eventRoutines.findEventByExchangeId(exchangeId);
        if (!eventO.isPresent()) {
            log.warn("No event was found with exchange id: {}", exchangeId);
            if (actionInfo.getActionSource() == ActionSource.EXCHANGE_SYNCH) { // not for EXCHANGE
                ignoredEventManager.storeIgnoredEventSafe(
                        exchangeId, IgnoreReason.CANCELED_EVENT_NOT_FOUND, Option.empty());
            }
        }
        return eventO;
    }

    private ExchangeEventData convertExchangeData(
            UidOrResourceId subjectId, CalendarItemType calendarItem, boolean dryRun)
    {
        logCalendarItemData(calendarItem);

        EventData exchangeEventData;
        ExtendedCalendarItemProperties extendedProperties = EwsUtils.convertExtendedProperties(calendarItem);
        try {
            exchangeEventData = ExchangeEventDataConverter.convert(
                    calendarItem, subjectId, extendedProperties.getOrganizerEmail(),
                    resourceRoutines::selectResourceEmails);
        } catch (UnsupportedRepetitionException e) {
            if (dryRun) {
                log.warn("Dry-run mode is on, do not mark event as ignored");
            } else {
                final String exchangeId = calendarItem.getItemId().getId();
                ignoredEventManager.storeIgnoredEventSafe(
                        exchangeId, IgnoreReason.UNSUPPORTED_REPETITION, Option.of(e.getMessage()));
            }
            throw e;
        }
        if (subjectId.isResource()) {
            ParticipantsData participantsData = exchangeEventData.getInvData().getParticipantsData();
            ListF<Email> partcipantEmails = participantsData.getParticipantsSafe().map(ParticipantData.getEmailF());
            if (!eventInvitationManager.getSubjectsFromEmails(partcipantEmails).containsTs(subjectId)) {
                log.warn("No subject resource among participants, "
                        + "subject id: {}, participants: {}"
                        + " when processing {}"
                        + ", adding it now", subjectId, partcipantEmails, LogMarker.EXCHANGE_ID.format(calendarItem.getItemId().getId()));

                Resource resource = resourceDao.findResourceById(subjectId.getResourceId());
                Email resourceEmail = ResourceRoutines.getResourceEmail(resource);
                ParticipantData resourceData = new ParticipantData(
                        resourceEmail, resource.getName().getOrElse(""), Decision.YES, true, false, false);
                log.info("Adding participant: {}", resourceData);
                participantsData = new Meeting(participantsData.getParticipantsSafe().plus(resourceData));
                exchangeEventData.setInvData(participantsData);

            }
        }
        if (calendarItem.getMyResponseType() != ResponseTypeType.ORGANIZER) {
            exchangeEventData.getEvent().unsetField(EventFields.DESCRIPTION); // CAL-5647
        }
        return new ExchangeEventData(calendarItem, exchangeEventData, extendedProperties);
    }

    private Function<CalendarItemType, ExchangeEventData> convertExchangeDataF(
            final UidOrResourceId subjectId, final boolean dryRun)
    {
        return i -> {
            fixEmailsIfPossible(i);
            return convertExchangeData(subjectId, i, dryRun);
        };
    }

    private static boolean isEventOlderThanAWeek(ExchangeEventData data) {
        val threshold = Instant.now().minus(Duration.standardDays(7));
        val repetition = data.getEventData().getRepetition();
        return repetition.getType() != RegularRepetitionRule.NONE
                ? repetition.getDueTs().map(t -> t.isBefore(threshold)).getOrElse(false)
                : data.getEventData().getEvent().getEndTs().isBefore(threshold);
    }

    private boolean isEventAlreadyDeleted(ExchangeEventData data) {
        return data
                .getEventData()
                .getExternalId()
                .map(extId -> deletedEventDao.existsDeletedEvent(extId, data.getRecurrenceId().toOptional()))
                .getOrElse(false);
    }

    private EwsImportResult createOrUpdateEventInner(
            UidOrResourceId subjectId, ExchangeEventData data, ActionInfo actionInfo, boolean dryRun)
    {
        if (isEventOlderThanAWeek(data)) {
            log.debug("Ignore event that is older than a week from now. id - {}. Recurrence - {}. Start - {}",
                    data.getEventData().getExternalId(), data.getRecurrenceId(), data.getEventData().getEvent().getStartTs());
            return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);
        }

        if (isEventAlreadyDeleted(data)) {
                log.debug("Ignore event that is already marked as deleted in deleted_events. id - {}. Recurrence - {}. Start - {}",
                    data.getEventData().getExternalId(), data.getRecurrenceId(), data.getEventData().getEvent().getStartTs());
            return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);
        }

        val exchangeEventData = data.getEventData();
        val extendedProperties = data.getExtendedProperties();
        val instInfo = getEventSynchronizationData(subjectId, exchangeEventData, actionInfo);

        log.info("Event synchronization status: {}", instInfo);

        if (instInfo.isAlreadyUpdated()) {
            return new EwsImportResult(EwsImportStatus.ALREADY_UPDATED, Cf.list(), subjectId);
        }

        if (dryRun) {
            log.warn("Dry-run mode is on, do nothing, status: {}", instInfo);
            return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);
        }

        if (extendedProperties.getWasCreatedFromYaTeamCalendar() && subjectId.isResource()
                && data.getCalendarItem().getMyResponseType() == ResponseTypeType.ORGANIZER)
        {
            log.debug("Ignore creation or update event came to exchange from ya-team calendar.");
            return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);
        }

        if (instInfo.isFound()) {
            excludeExdatesThatAreRecurrencesWithoutSubjectParticipation(subjectId, instInfo.getEventId(), exchangeEventData);
        }

        if (instInfo.isNeedToUpdateWithNewExternalId()) {
            exchangeEventData.getEvent().setMainEventId(eventRoutines.createMainEvent(
                    instInfo.getChangedExternalId(), exchangeEventData.getTimeZone(),
                    instInfo.getIsExportedWithEws(), actionInfo));
            return new EwsImportResult(
                    EwsImportStatus.UPDATED,
                    updateEventInner(subjectId, instInfo.getEventId(), exchangeEventData, actionInfo, instInfo)
                            .getAffectedLayerIds(),
                    subjectId
            );
        }

        if (instInfo.isNeedToUpdate()) {
            return new EwsImportResult(
                    EwsImportStatus.UPDATED,
                    updateEventInner(subjectId, instInfo.getEventId(), exchangeEventData, actionInfo, instInfo)
                            .getAffectedLayerIds(),
                    subjectId
            );

        } else if (instInfo.isNotFound()) {
            if (!data.getEventData().getEvent().getRecurrenceId().isPresent() // CAL-7950
                    && extendedProperties.getWasCreatedFromYaTeamCalendar()
                    && actionInfo.getActionSource().isByExchangeNotification())
            {
                // https://jira.yandex-team.ru/browse/CAL-2410
                log.debug(
                        "Ignore event creation, because event came to " +
                        "exchange from ya-team calendar.");
                return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);
            } else if (EwsUtils.isCancelled(data.getCalendarItem())) {
                log.info("Ignore cancelled event creation");
                return new EwsImportResult(EwsImportStatus.SKIPPED, Cf.list(), subjectId);

            } else {
                return new EwsImportResult(
                        EwsImportStatus.CREATED,
                        createEventInner(subjectId, exchangeEventData, actionInfo),
                        subjectId);
            }
        } else {
            throw new IllegalStateException("Cannot be here, status = " + instInfo);
        }
    }

    private void excludeExdatesThatAreRecurrencesWithoutSubjectParticipation(
            UidOrResourceId subjectId, long eventId, EventData eventData)
    {
        Function1B<Rdate> isRdateF = Function1B.wrap(Rdate.getIsRdateF());
        ListF<Instant> exdateStarts = eventData.getRdates().filter(isRdateF.notF()).map(Rdate.getStartTsF());

        if (exdateStarts.isEmpty()) return;

        long mainEventId = eventDao.findMainEventIdByEventId(eventId).get();
        ListF<Event> recurrenceEvents = eventDao.findRecurrenceEventsByMainId(mainEventId, exdateStarts, false);
        ListF<Long> recurrenceEventIds = recurrenceEvents.map(Event.getIdF());

        if (recurrenceEventIds.isEmpty()) return;

        final ListF<Long> recurrenceEventIdsWithSubject;
        if (subjectId.isResource()) {
            recurrenceEventIdsWithSubject = eventResourceDao
                    .findEventResourcesByEventIdsAndResourceIds(recurrenceEventIds, Cf.list(subjectId.getResourceId()))
                    .map(EventResource.getEventIdF());
        } else {
            recurrenceEventIdsWithSubject = eventUserDao
                    .findEventUsersByEventIdsAndUid(recurrenceEventIds, subjectId.getUid())
                    .filter(EventUser.getDecisionF().andThenEquals(Decision.NO).notF())
                    .map(EventUser.getEventIdF());
        }
        ListF<Instant> recurrenceIdsWithoutSubject = recurrenceEvents
                .filter(Event.getIdF().andThen(recurrenceEventIdsWithSubject.unique().containsF().notF()))
                .map(e -> e.getRecurrenceId().get());

        eventData.setRdates(eventData.getRdates().filter(
                isRdateF.orF(Rdate.getStartTsF().andThen(recurrenceIdsWithoutSubject.unique().containsF().notF()))));
    }

    private EventInstanceStatusInfo getEventSynchronizationData(
            UidOrResourceId subjectId, EventData exchangeEventData, ActionInfo actionInfo)
    {
        { // search by exchange-id first (because UID can absent)
            ExchangeEventSynchData exchangeSynchData = EwsUtils.createExchangeSynchData(exchangeEventData);
            EventInstanceStatusInfo statusInfo = eventInstanceStatusChecker.getStatusByExchangeSynchData(
                    subjectId, exchangeSynchData, exchangeEventData.getExternalId(), actionInfo);

            if (statusInfo.isFound() || !exchangeEventData.getExternalId().isPresent()) {
                if (statusInfo.isAlreadyUpdated() && actionInfo.getActionSource().isByExchangeNotification()) {
                    return EventInstanceStatusInfo.needToUpdate(statusInfo.getEventId());
                }
                return statusInfo;
            }
        }

        { // event could be added through other sources - search by ics data (requires UID)
            IcsEventSynchData icsSynchData = IcsUtils.createIcsSynchData(exchangeEventData);

            ParticipantsOrInvitationsData participantsOrInvitationsData = exchangeEventData.getInvData();
            Validate.V.isTrue(participantsOrInvitationsData.isParticipantsData(), "we must know all participants to find event");

            // XXX: should check domain here?
            ListF<UidOrResourceId> subjects = eventInvitationManager.getSubjectsFromEmails(
                    participantsOrInvitationsData.getParticipantsData().getParticipantsSafe().map(ParticipantData.getEmailF()));

            EventInstanceStatusInfo statusInfo =
                    eventInstanceStatusChecker.getStatusByParticipantsAndIcsSyncData(
                            subjectId, icsSynchData, subjects, actionInfo.getActionSource());

            if (statusInfo.isAlreadyUpdated()) {
                // XXX ssytnik@ TEMP rolling back to stone age to hopefully fix https://jira.yandex-team.ru/browse/CAL-3207
                log.info("Status by external id is UPDATED. Temporarily, doing workaround for CAL-3207 - NEED_TO_UPDATE");
                //if (!updateExchangeIdOrIgnoreCopyEvent(subjectId, statusInfo.getEventId(), exchangeEventData, dryRun)) {
                    statusInfo = EventInstanceStatusInfo.needToUpdate(statusInfo.getEventId());
                //}
            }
            return statusInfo;
        }
    }

    private ListF<Long> createEventInner(
            UidOrResourceId subjectId, EventData eventData, ActionInfo actionInfo)
    {
        long mainEventId;
        if (eventData.getExternalId().isPresent()) {
            ListF<Email> participantEmails =
                    eventData.getInvData().getParticipantsData()
                            .getParticipantsSafe().map(ParticipantData.getEmailF());
            mainEventId = eventRoutines.getMainEventIdBySubjectsIdAndExternalIdOrCreateNew(
                            subjectId, participantEmails, eventData, actionInfo);
        } else {
            mainEventId = eventRoutines.createMainEvent(subjectId, eventData, actionInfo);
        }

        Option<Long> masterEventId = eventRoutines.findMasterEventByMainId(mainEventId).map(Event.getIdF());

        if (subjectId.isUser()) {
            Option<ParticipantData> participant = findUserParticipant(subjectId, eventData);
            if (!participant.isPresent() || participant.get().isOrganizer()) {
                setYesOrNoDecision(eventData);

            } else if (masterEventId.isPresent()
                && eventData.getEventUserData().getDecision().isSome(Decision.UNDECIDED)
                && eventUserRoutines.findEventUserDecision(
                        subjectId.getUid(), masterEventId.get()).exists(Decision::goes))
            {
                unsetDecisionAndAvailability(eventData);
            }
        }


        if (eventData.getEvent().getRecurrenceId().isPresent() && masterEventId.isPresent()) {
            PassportUid creatorUid = resourceRoutines.getUidOrMasterOfResource(subjectId, eventData);

            long recurrenceEventId = eventRoutines.createNotChangedRecurrence(
                    creatorUid, masterEventId.get(), eventData.getEvent().getRecurrenceId().get(), actionInfo);

            eventData.getEvent().setId(recurrenceEventId);

            return eventRoutines.updateEventFromIcsOrExchange(
                    subjectId, eventData, NotificationsData.notChanged(),
                    EventInstanceStatusInfo.needToUpdate(recurrenceEventId),
                    EwsUtils.createExchangeSynchData(eventData).getSequenceAndDtstamp(), actionInfo)
                    .getAffectedLayerIds();

        } else {
            CreateInfo createInfo = eventRoutines.createUserOrFeedEvent(
                    subjectId, EventType.USER, mainEventId, eventData, NotificationsData.useLayerDefaultIfCreate(),
                    InvitationProcessingMode.SAVE_ATTACH, actionInfo);

            if (eventData.getEvent().getRecurrenceId().isPresent()) {
                lastUpdateManager.updateTimestampsAsync(createInfo.getEvent().getMainEventId(), actionInfo);
                updateSequenceAndDtstampForUser(subjectId, createInfo.getEvent().getId(), eventData, actionInfo);
            }
            return createInfo.getAllLayerIds();
        }
    }

    private UpdateInfo updateEventInner(
            UidOrResourceId subjectId, long eventId, EventData eventData,
            ActionInfo actionInfo, EventInstanceStatusInfo eventInstanceStatusInfo)
    {
        log.info(LogMarker.EVENT_ID.format(eventId));

        eventData.getEvent().setId(eventId);
        SequenceAndDtStamp userVersion = EwsUtils.createExchangeSynchData(eventData).getSequenceAndDtstamp();
        Participants participants = eventInvitationManager.getParticipantsByEventId(eventId);
        ParticipantsData participantsData = eventData.getInvData().getParticipantsData();
        if (subjectId.isUser() && actionInfo.getActionSource() != ActionSource.EXCHANGE_SYNCH &&
                (isNotOrganizerInCalendar(subjectId.getUid(), participants) ||
                 isNotOrganizerInExchange(subjectId.getUid(), participantsData))
        ) {
            log.info("Updating attendee only");
            EventAndRepetition event = eventDbManager.getEventAndRepetitionByIdForUpdate(eventId);

            return new UpdateInfo(
                    processOnlyUserRelatedStuff(subjectId, event, eventData, actionInfo),
                    Cf.list(), Cf.list());

        } else if (subjectId.isResource()) {
            log.info("Set room decision...got push");
            log.info("Updating whole event");

            return eventRoutines.updateEventFromIcsOrExchange(
                    subjectId, eventData, NotificationsData.notChanged(),
                    eventInstanceStatusInfo, userVersion, actionInfo);
        } else {
            log.info("Updating whole event");

            setYesOrNoDecision(eventData);

            UpdateInfo info = eventRoutines.updateEventFromIcsOrExchange(
                    subjectId, eventData, NotificationsData.notChanged(),
                    eventInstanceStatusInfo, userVersion, actionInfo);

            EventAndRepetition event = eventDbManager.getEventAndRepetitionByIdForUpdate(eventId);

            // sometimes Exchange sends Delete/Create instead of Modified notification
            // and this restores possibly deleted event layer (CAL-3207)
            EventAttachedLayerId attachedIds = eventDbManager.saveEventLayerIfAbsentForUser(
                    subjectId.getUid(), event, actionInfo);

            eventsLogger.log(EventChangeLogEvents.updated(
                    ActorId.userOrResource(subjectId), consLogId(event.getEvent(), eventData), attachedIds), actionInfo);

            return info.plusAffectedLayerIds(attachedIds.getAllLayerIds());
        }
    }

    // see IcsEventImporter#processOnlyUserRelatedStuff
    private ListF<Long> processOnlyUserRelatedStuff(
            UidOrResourceId subjectId, EventAndRepetition event, EventData eventData, ActionInfo actionInfo)
    {
        PassportUid actorUid = subjectId.getUid();
        Email userEmail = userManager.getEmailByUid(actorUid).get();

        if (event.getRepetitionInfo().getRepetition().isPresent()
                && eventData.getRepetition().getType() == RegularRepetitionRule.NONE
                && !eventData.getEvent().getRecurrenceId().isPresent())
        {
            Option<Instant> recurrenceId = event.getRepetitionInfo()
                    .findRecurrenceIdByStart(eventData.getEvent().getStartTs());

            if (recurrenceId.isPresent()) {
                event = eventDbManager.getEventAndRepetitionByIdForUpdate(
                        eventDao.findRecurrenceEventByMainId(event.getMainEventId(), recurrenceId.get()).single().getId());
            } else {
                log.debug("Ignore single event looking like recurrence that does not match");
                return Cf.list();
            }
        }
        if (event.getRepetitionInfo().getRepetition().isPresent()) {
            ListF<Instant> exdates = event.getRepetitionInfo().getExdateStarts();

            ListF<Instant> newExdates = eventData.getRdates()
                    .filterMap(r -> Option.when(!r.getIsRdate() && !exdates.containsTs(r.getStartTs()), r.getStartTs()));

            Event m = event.getEvent();
            Lazy<EventInfo> master = Lazy.withSupplier(() -> eventInfoDbLoader.getEventInfoByEvent(
                    subjectId.getUidO(), m, actionInfo.getActionSource()));

            MapF<Instant, Event> recurrences = eventDao.findRecurrenceEventsByMainId(
                    event.getMainEventId(), newExdates, false).toMapMappingToKey(e -> e.getRecurrenceId().get());

            MapF<Long, Event> exdated = eventDao.findEventsByIds(newExdates.filterMap(ex -> {
                Option<Event> found = recurrences.getO(ex);

                if (found.isPresent()) {
                    return found.filterMap(e -> Option.when(e.getEndTs().isAfter(actionInfo.getNow()), e.getId()));

                } else if (ex.isAfter(actionInfo.getNow())
                        && RepetitionUtils.isValidStart(master.get().getRepetitionInstanceInfo(), ex)
                        && master.get().getEventWithRelations().findUserEventUser(actorUid).isPresent())
                {
                    return Option.of(eventRoutines.createNotChangedRecurrence(actorUid, master.get(), ex, actionInfo));

                } else {
                    return Option.empty();
                }
            })).toMapMappingToKey(Event::getId);

            EventUser euData = new EventUser();
            euData.setDecision(Decision.NO);

            ListF<EventUserUpdate> updates = eventUserDao.findEventUsersByEventIdsAndUid(exdated.keys(), actorUid)
                    .map(eu -> EventUserUpdate.update(eu, eventUserRoutines
                            .eventUserUpdateData(eu.getId(), euData, actionInfo).createOrUpdate.getRight()));

            eventUserDao.updateEventUsersBatch(updates.map(u -> u.cur.get()), actionInfo);

            updates.forEach(up -> eventsLogger.log(EventChangeLogEvents.updated(
                    ActorId.userOrResource(subjectId),
                    new EventIdLogDataJson(
                            master.get().getMainEvent().getExternalId(),
                            exdated.getOrThrow(up.cur.get().getEventId())),
                    userEmail, up.old, up.cur, Option.empty()), actionInfo));
        }

        Option<ParticipantData> participantDataO = findUserParticipant(subjectId, eventData);

        EventUser newEventUserData = eventData.getEventUserData().getEventUser().copy();
        Option<Decision> decision = eventData.getEventUserData().getDecision();

        boolean meetingIsRejected = decision.isSome(Decision.NO);

        Option<EventUser> oldEventUser = eventUserDao.findEventUserByEventIdAndUid(event.getEventId(), actorUid);

        EventUserUpdate eventUserUpdate;

        if (!oldEventUser.isPresent()) {
            newEventUserData.setIsAttendee(participantDataO.isPresent());
            newEventUserData.setIsOptional(false);

            if (decision.isPresent() && (!participantDataO.isPresent() || participantDataO.get().isOrganizer())) {
                newEventUserData.setDecision(toYesOrNoDecision(decision.get()));
            }
            eventUserUpdate = eventUserRoutines.saveEventUserAndNotification(
                    actorUid, event, newEventUserData, NotificationsData.useLayerDefaultIfCreate(),
                    layerRoutines.getOrCreateDefaultLayer(actorUid), actionInfo);
        } else {
            if (decision.isPresent() && (!oldEventUser.get().getIsAttendee() || oldEventUser.get().getIsOrganizer())) {
                newEventUserData.setDecision(toYesOrNoDecision(decision.get()));

            } else if (decision.isSome(Decision.UNDECIDED) && oldEventUser.get().getDecision().goes()) {
                unsetDecisionAndAvailability(newEventUserData);
            }
            eventUserUpdate = eventUserRoutines.updateEventUser(oldEventUser.get(), newEventUserData, actionInfo);
        }

        Option<EventAttachedLayerId> layerId = Option.when(!meetingIsRejected, event)
                .map(e -> eventDbManager.saveEventLayerIfAbsentForUser(actorUid, e, actionInfo));

        lastUpdateManager.updateTimestampsAsync(event.getMainEventId(), actionInfo);

        eventRoutines.saveUpdateExchangeId(
                subjectId, event.getEventId(),
                eventData.getExchangeData().get().getExchangeId(), actionInfo);

        eventsLogger.log(EventChangeLogEvents.updated(
                ActorId.userOrResource(subjectId), consLogId(event.getEvent(), eventData),
                userEmail, eventUserUpdate.old, eventUserUpdate.cur, layerId), actionInfo);

        if (settingsRoutines.getIsEwser(actorUid)) {
            // the check above will be repeated in fixDecisionInExchangeIfNeeded
            // it is here to get rid of getEventWithRelations in most cases
            EventWithRelations eventWR = eventDbManager.getEventWithRelationsByEvent(event.getEvent());
            eventRoutines.fixDecisionInExchangeIfNeeded(
                    actorUid, eventWR, event.getRepetitionInfo(),
                    eventData.getEventUserData().getDecision(), actionInfo);
        }

        return layerId.flatMap(EventAttachedLayerId::getAllLayerIds);
    }

    /**
     * for details about getting sequence and dtstamp for synch data, see
     * {@link #getEventSynchronizationData(UidOrResourceId, EventData, ActionInfo)}
     */
    private void updateSequenceAndDtstampForUser(
            UidOrResourceId subjectId, long eventId, EventData eventData, ActionInfo actionInfo)
    {
        if (subjectId.isUser()) {
            SequenceAndDtStamp version = EwsUtils.createExchangeSynchData(eventData).getSequenceAndDtstamp();
            eventInvitationManager.updateEventUserSequenceAndDtstamp(subjectId.getUid(), eventId, version, actionInfo);
        }
    }

    private boolean isOrganizer(PassportUid uid, ParticipantsData participantsData) {
        Option<ParticipantId> organizerId = participantsData.getOrganizerEmailSafe()
                .map(eventInvitationManager.getParticipantIdByEmailF());
        return organizerId.isPresent() && organizerId.get().isYandexUserWithUid(uid);
    }

    private boolean isNotOrganizerInExchange(PassportUid uid, ParticipantsData participantsData) {
        boolean res;
        if (!participantsData.isMeeting()) {
            res = false;
        } else {
            ParticipantId organizerId = eventInvitationManager.getParticipantIdByEmail(
                    participantsData.asMeeting().getOrganizer().getEmail());
            res = !organizerId.isYandexUserWithUid(uid);
        }
        log.debug("IsNotOrganizerInExchange: {}", res); // XXX temporary
        return res;
    }

    private boolean isNotOrganizerInCalendar(PassportUid uid, Participants participants) {
        boolean res = participants.isMeeting() && !participants.getOrganizer().getId().isYandexUserWithUid(uid);
        log.debug("IsNotOrganizerInCalendar: {}", res); // XXX temporary
        return res;
    }

    private EwsImportResult removeEventInner(
            final UidOrResourceId subjectId, Event event, final ActionInfo actionInfo)
    {
        long eventId = event.getId();

        Function0<EventParticipants> getParticipants =
                () -> eventDbManager.getParticipantsByEventIds(Cf.list(eventId)).single();

        EventParticipants participants = getParticipants.apply();

        ListF<Long> layerIds;

        if (shouldRemoveEventWithBadHack(subjectId, event, participants)) {
            log.info("Removing whole event");

            ListF<Long> eventIds = !event.getRecurrenceId().isPresent()
                    ? eventDao.findEventIdsByMainEventId(event.getMainEventId())
                    : Cf.list(eventId);

            layerIds = eventLayerDao.findLayersIdsByEventIds(eventIds);
            eventRoutines.deleteEvents(Option.empty(), eventIds, InvitationProcessingMode.SAVE_ATTACH, actionInfo);

        } else if (subjectId.isResource()) {
            log.info("Removing resource only");
            layerIds = Cf.list();

            Option<EventAttachedLayerId> attachedLayerIds = eventInvitationManager.removeAttendeeByParticipantId(
                    eventId, subjectId.toParticipantId(), actionInfo);

            eventsLogger.log(EventChangeLogEvents.updated(ActorId.userOrResource(subjectId),
                    new EventIdLogDataJson(mainEventDao.findExternalIdByMainEventId(event.getMainEventId()), event),
                    participants, getParticipants.apply(), attachedLayerIds), actionInfo);

        } else {
            log.info("Declining meeting");

            EventAttachedUser detach = eventInvitationManager.rejectMeetingsByYandexUser(
                    Option.of(event), subjectId.getUid(), actionInfo).events.single();

            layerIds = detach.layerId.flatMap(EventAttachedLayerId::getAllLayerIds);

            // CAL-3057: for deleted exchange event, we should not find old "phantom" exchange id
            eventRoutines.saveUpdateExchangeId(subjectId, eventId, null, actionInfo);

            String externalId = mainEventDao.findExternalIdByMainEventId(detach.event.getMainEventId());
            Email email = settingsRoutines.getSettingsByUid(subjectId.getUid()).getEmail();

            eventsLogger.log(EventChangeLogEvents.updated(
                    ActorId.userOrResource(subjectId), new EventIdLogDataJson(externalId, detach.event),
                    EventChangesJson.empty().withUsersAndLayers(UserRelatedChangesJson.find(email, detach))), actionInfo);
        }

        return new EwsImportResult(EwsImportStatus.REMOVED, layerIds, subjectId);
    }

    private boolean shouldRemoveEventWithBadHack(UidOrResourceId subjectId, Event event, EventParticipants ep) {
        Participants participants = ep.getParticipants();

        boolean isExportedWithEws =
                mainEventDao.findMainEventByEventId(event.getId()).getIsExportedWithEws().getOrElse(false);

        if (subjectId.isUser()) {
            if (participants.isNotMeetingStrict()) {
                return false;
            }
            if (participants.getOrganizerIdSafe().containsTs(subjectId.toParticipantId())) {
                return event.getCreationSource().isFromExchangeOrMail() || isExportedWithEws;
            }
            return false;
        } else if (isExportedWithEws) {
            return false;
        } else {
            ListF<Long> resourceParticipants = participants.getResourceIdsSafeWithInconsistent();
            if (resourceParticipants.size() > 1) {
                return false;
            }
            // XXX also check that event organizer is not subscribed
            return resourceParticipants.containsTs(subjectId.getResourceId());
        }
    }

    private Option<ParticipantData> findUserParticipant(UidOrResourceId subjectId, EventData eventData) {
        if (!subjectId.isUser()) return Option.empty();

        Option<Email> email = eventInvitationManager.getParticipantIdsByEmails(eventData.getParticipantEmails())
                .findBy2(ParticipantId.isYandexUserWithUidF(subjectId.getUid()))
                .map(Tuple2::get1);

        return eventData.getInvData().getParticipantsData().getParticipantsSafe()
                .find(ParticipantData.getEmailF().andThen(email.containsF()));
    }

    private void setYesOrNoDecision(EventData data) {
        EventUser newEventUserData = data.getEventUserData().getEventUser().copy();
        Option<Decision> decision = newEventUserData.getFieldValueO(EventUserFields.DECISION);

        newEventUserData.setDecision(decision.map(EwsImporter::toYesOrNoDecision).getOrElse(Decision.YES));
        data.setEventUserData(data.getEventUserData().withEventUser(newEventUserData));
    }

    private void unsetDecisionAndAvailability(EventData data) {
        EventUser newEventUserData = data.getEventUserData().getEventUser().copy();
        unsetDecisionAndAvailability(newEventUserData);

        data.setEventUserData(data.getEventUserData().withEventUser(newEventUserData));
    }

    private void unsetDecisionAndAvailability(EventUser eventUser) {
        eventUser.unsetField(EventUserFields.DECISION);
        eventUser.unsetField(EventUserFields.AVAILABILITY);
    }

    private static Decision toYesOrNoDecision(Decision decision) {
        return decision == Decision.NO ? Decision.NO : Decision.YES;
    }

    public void fixEmailsIfPossible(CalendarItemType calendarItem) {

        Function<EmailAddressType, EmailAddressType> fixEmailIfPossible = sourceAddress -> {
            try {
                EwsUtils.getEmailO(sourceAddress);
            } catch (EwsBadEmailException e) {
                Option<Email> email = getEmailOSmart(sourceAddress);
                if (email.isPresent()) {
                    EmailAddressType address = new EmailAddressType();
                    address.setEmailAddress(email.get().toString());
                    return address;
                }
            }
            return sourceAddress;
        };

        if (calendarItem.getOrganizer() != null) {
            calendarItem.getOrganizer().setMailbox(fixEmailIfPossible.apply(calendarItem.getOrganizer().getMailbox()));
        }

        ExchangeEventDataConverter.getAllAttendees(calendarItem).forEach(attendee -> {
            attendee.setMailbox(fixEmailIfPossible.apply(attendee.getMailbox()));
        });
    }

    public Option<Email> getEmailOSmart(EmailAddressType emailAddress) {
        try {
            return EwsUtils.getEmailO(emailAddress);
        } catch (EwsBadEmailException e) {
            String normalizedEmail = e.getMessage().toLowerCase();

            for (Tuple2<Pattern, Function<String, Email>> entry : patternsAndActions) {
                Matcher m = entry._1.matcher(normalizedEmail);
                if (m.find()) {
                    try {
                        return Option.of(entry._2.apply(m.group(1)));
                    } catch (NoSuchElementException | IllegalArgumentException ignore) {
                    }
                }
            }
            return Option.empty();
        }
    }

    private EventIdLogDataJson consLogId(Event event, EventData data) {
        return new EventIdLogDataJson(data.getExternalId()
                .getOrElse(() -> mainEventDao.findExternalIdByMainEventId(event.getMainEventId())), event);
    }
}
