package ru.yandex.calendar.frontend.ews.imp;

import javax.xml.datatype.XMLGregorianCalendar;

import com.microsoft.schemas.exchange.services._2006.types.AttendeeType;
import com.microsoft.schemas.exchange.services._2006.types.BodyType;
import com.microsoft.schemas.exchange.services._2006.types.BodyTypeType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekIndexType;
import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekType;
import com.microsoft.schemas.exchange.services._2006.types.DeletedOccurrenceInfoType;
import com.microsoft.schemas.exchange.services._2006.types.EmailAddressType;
import com.microsoft.schemas.exchange.services._2006.types.ImportanceChoicesType;
import com.microsoft.schemas.exchange.services._2006.types.LegacyFreeBusyType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfAttendeesType;
import com.microsoft.schemas.exchange.services._2006.types.OccurrenceInfoType;
import com.microsoft.schemas.exchange.services._2006.types.RecurrenceType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseTypeType;
import com.microsoft.schemas.exchange.services._2006.types.SensitivityChoicesType;
import com.microsoft.schemas.exchange.services._2006.types.SingleRecipientType;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeData;
import ru.yandex.calendar.frontend.ews.ExtendedCalendarItemProperties;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.AbstractEventDataConverter;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.Completion;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.repetition.UnsupportedRepetitionException;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.html.HtmlUtils;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author akirakozov
 */
public class ExchangeEventDataConverter extends AbstractEventDataConverter {
    private static final Logger logger = LoggerFactory.getLogger(ExchangeEventDataConverter.class);
    private static final MapF<LegacyFreeBusyType, Availability> availabilityMap = Cf.hashMap();
    private static final MapF<ImportanceChoicesType, Priority> priorityMap = Cf.hashMap();
    private static final MapF<ResponseTypeType, Decision> decisionMap = Cf.hashMap();

    private final CalendarItemType calItem;
    /** required to fill exchangeData */
    private final Option<UidOrResourceId> subjectIdO;
    private final Option<Email> realOrganizerO;

    private final Function<ListF<Email>, ListF<Email>> selectResourceEmailsF;

    private final ExtendedCalendarItemProperties extendedProperties;

    static {
        availabilityMap.put(LegacyFreeBusyType.BUSY, Availability.BUSY);
        availabilityMap.put(LegacyFreeBusyType.FREE, Availability.AVAILABLE);
        availabilityMap.put(LegacyFreeBusyType.TENTATIVE, Availability.MAYBE);
        availabilityMap.put(LegacyFreeBusyType.OOF, Availability.AVAILABLE);
        availabilityMap.put(LegacyFreeBusyType.NO_DATA, Availability.AVAILABLE);
        priorityMap.put(ImportanceChoicesType.LOW, Priority.LOW);
        priorityMap.put(ImportanceChoicesType.NORMAL, Priority.NORMAL);
        priorityMap.put(ImportanceChoicesType.HIGH, Priority.HIGH);
        decisionMap.put(ResponseTypeType.NO_RESPONSE_RECEIVED, Decision.UNDECIDED);
        decisionMap.put(ResponseTypeType.UNKNOWN, Decision.UNDECIDED);
        decisionMap.put(ResponseTypeType.ORGANIZER, Decision.YES);
        decisionMap.put(ResponseTypeType.ACCEPT, Decision.YES);
        decisionMap.put(ResponseTypeType.TENTATIVE, Decision.MAYBE);
        decisionMap.put(ResponseTypeType.DECLINE, Decision.NO);
    }

    private ExchangeEventDataConverter(
            CalendarItemType calItem, Option<UidOrResourceId> subjectIdO, Option<Email> realOrganizerO,
            Function<ListF<Email>, ListF<Email>> selectResourceEmailsF)
    {
        // XXX we still cannot understand timezones like: "+0300 (Standard) / GMT +0400 (Daylight)",
        // can they only be found in old events, and in development's exchange? // ssytnik
        super(EwsUtils.getOrDefaultZone(calItem));
        logger.debug("converting exchange event using tz = " + eventTz.getID()); // CAL-4025

        this.calItem = calItem;
        this.subjectIdO = subjectIdO;
        this.realOrganizerO = realOrganizerO;
        this.extendedProperties = EwsUtils.convertExtendedProperties(calItem.getExtendedProperty());
        this.selectResourceEmailsF = selectResourceEmailsF;
    }

    private boolean toBool(Boolean b) {
        if (b == null) { return false; }
        return b.booleanValue();
    }

    protected EventData doConvert() {
        EventData eventData = convertEventData();
        eventData.setExchangeData(convertExchangeData());
        eventData.setEventUserData(convertEuData());
        eventData.setInvData(convertParticipantsData());
        eventData.getRdates().addAll(convertExdates()); // XXX and rdates? // ssytnik@
        eventData.setRepetition(convertRepetition(eventData));
        logInfoAboutRecurrences();
        return eventData;
    }

    private Option<ExchangeData> convertExchangeData() {
        if (!subjectIdO.isPresent()) {
            return Option.empty();
        }
        final String exchangeId = calItem.getItemId().getId();
        final ExchangeData exchangeData = new ExchangeData(subjectIdO.get(), exchangeId);
        return Option.of(exchangeData);
    }

    private void logInfoAboutRecurrences() {
        logger.debug("Recurrence id: " + calItem.getRecurrenceId());
        logger.debug("Recurrence id from extended property: " + extendedProperties.getRecurrenceId());
        if (calItem.getModifiedOccurrences() != null) {
            ListF<OccurrenceInfoType> arr = Cf.x(calItem.getModifiedOccurrences().getOccurrence());
            for (OccurrenceInfoType occurrence : arr) {
                logger.debug("Modified item id: " + occurrence.getItemId().getId());
            }
        }
    }

    private int getSequence() {
        final Integer sequence = calItem.getAppointmentSequenceNumber();
        return sequence != null ? sequence.intValue() : 0; // zero is default sequence number
    }

    private EventData convertEventData() {
        EventData eventData = new EventData();
        // synchronization data
        if (StringUtils.isNotEmpty(calItem.getUID())) {
            eventData.setExternalId(Option.of(calItem.getUID()));
        }

        eventData.getEvent().setRecurrenceId(EwsUtils.getRecurrenceId(calItem));

        eventData.getEvent().setDtstamp(EwsUtils.toInstantO(calItem.getDateTimeStamp()));
        eventData.getEvent().setSequence(getSequence());
        eventData.getEvent().setLastUpdateTs(EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getLastModifiedTime()));
        // date time data
        boolean isAllDay = toBool(calItem.isIsAllDayEvent());

        InstantInterval interval = new InstantInterval(
                EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getStart()),
                EwsUtils.xmlGregorianCalendarInstantToInstant(calItem.getEnd()));

        if (isAllDay) {
            Function<Instant, ReadableInstant> stick = ts -> {
                DateTime next = new DateTime(ts, eventTz).plusDays(1).withTimeAtStartOfDay();

                return new Duration(ts, next).isLongerThan(Duration.standardHours(3)) ? ts : next;
            };
            interval = new InstantInterval(stick.apply(interval.getStart()), stick.apply(interval.getEnd()));
        }

        eventData.getEvent().setStartTs(interval.getStart());
        eventData.getEvent().setEndTs(interval.getEnd());
        eventData.getEvent().setIsAllDay(isAllDay);
        eventData.setTimeZone(eventTz);
        // info data
        // XXX 'eventName': not good, temporary, solution // ssytnik@
        String eventName = StringUtils.defaultIfEmpty(calItem.getSubject(), UStringLiteral.NO_NAME);
        eventData.getEvent().setName(eventName);
        eventData.getEvent().setLocation(StringUtils.notBlankO(calItem.getLocation()).getOrElse(""));
        eventData.getEvent().setConferenceUrl(StringUtils.defaultIfEmpty(calItem.getMeetingWorkspaceUrl(), ""));

        SensitivityChoicesType sensitivity = calItem.getSensitivity();
        if (sensitivity != null && sensitivity != SensitivityChoicesType.NORMAL) {
            if (sensitivity != SensitivityChoicesType.PRIVATE) {
                logger.info("Sensitivity = " + sensitivity + ", forcing PRIVATE");
            }
            eventData.getEvent().setPermAll(EventActionClass.NONE);
        }

        BodyType body = calItem.getBody();
        if (body != null && body.getValue() != null) {
            String description;
            if (body.getBodyType().equals(BodyTypeType.HTML)) {
                //logger.debug("HTML content: " + body.getValue());
                description = HtmlUtils.html2text(body.getValue());
                //logger.debug("Converted to text: " + description);
            } else {
                description = body.getValue();
            }
            eventData.getEvent().setDescription(StringUtils.notBlankO(description).getOrElse(""));
        }
        return eventData;
    }

    public static Option<Availability> getAvailability(LegacyFreeBusyType legacyFreeBusyStatus) {
        return availabilityMap.getO(legacyFreeBusyStatus);
    }

    private EventUserData convertEuData() {
        Option<Availability> availability = getAvailability(calItem.getLegacyFreeBusyStatus());
        Option<Priority> priority = priorityMap.getO(calItem.getImportance());
        Option<Decision> decision = decisionMap.getO(calItem.getMyResponseType());

        EventUser eventUser = new EventUser();
        eventUser.setAvailability(availability.getOrElse(Availability.AVAILABLE));
        eventUser.setPriority(priority.getOrElse(Priority.NORMAL));
        eventUser.setCompletion(Completion.NOT_APPLICABLE);

        decision.forEach(eventUser::setDecision);

        return new EventUserData(eventUser, Cf.list());
    }

    private Option<ParticipantData> convertParticipantData(
            Decision decision, EmailAddressType mailbox, boolean organizer)
    {
        Option<Email> email = EwsUtils.getEmailOSafe(mailbox);
        if (email.isPresent()) {
            return email.map(e -> new ParticipantData(e, mailbox.getName(), decision, !organizer, organizer, false));
        } else {
            logger.warn("Attendee email not found by " + YandexToStringBuilder.reflectionToString(mailbox));
            return Option.empty();
        }
    }

    private ParticipantsData convertParticipantsData() {

        ListF<AttendeeType> allAttendees = getAllAttendees(calItem);
        ListF<Email> resourcesEmails =
                selectResourceEmailsF.apply(allAttendees.map(a -> new Email(a.getMailbox().getEmailAddress())));

        Function1B<ParticipantData> isNotDeclinedResourceF =
                p -> !(p.getDecision() == Decision.NO && resourcesEmails.containsTs(p.getEmail()));

        Function<ListF<AttendeeType>, AttendeeType> resolveAttendee = attendeesWithSameEmail -> {
            if (attendeesWithSameEmail.size() == 1) {
                return attendeesWithSameEmail.single();
            }
            MapF<Decision, ListF<AttendeeType>> attendeesByDecision = attendeesWithSameEmail.groupBy(
                    a -> EwsUtils.getDecisionSafe(a.getResponseType()));

            ListF<Decision> priority = Cf.list(Decision.YES, Decision.MAYBE, Decision.UNDECIDED, Decision.NO);

            for (Decision decision : priority) {
                Option<AttendeeType> attendee = attendeesByDecision.getO(decision).filterMap(ListF::firstO);
                if (attendee.isPresent()) {
                    return attendee.get();
                }
            }

            return attendeesWithSameEmail.first();
        };

        allAttendees = allAttendees.groupBy(attendee -> attendee.getMailbox().getEmailAddress())
                .mapValues(resolveAttendee).values().toList();

        ListF<ParticipantData> attendees = allAttendees.map(
                attendee -> {
                    final Option<Decision> decision = Decision.findByExchangeReponseType(attendee.getResponseType());
                    return convertParticipantData(decision.getOrElse(Decision.UNDECIDED), attendee.getMailbox(), false);
                }).filterMap(a -> a).filter(isNotDeclinedResourceF);

        final SingleRecipientType organizer = calItem.getOrganizer();
        if (organizer != null || realOrganizerO.isPresent()) {
            Option<ParticipantData> organizerData = realOrganizerO.isPresent()
                    ? Option.of(new ParticipantData(realOrganizerO.get(), null, Decision.YES, false, true, false))
                    : convertParticipantData(Decision.YES, organizer.getMailbox(), true);
            if (organizerData.isPresent()) {
                return ParticipantsData.merge(organizerData.get(), attendees);
            }
        }
        return ParticipantsData.notMeeting();
    }

    private Repetition convertRepetition(EventData eventData) {
        RecurrenceType recurrence = calItem.getRecurrence();
        if (recurrence != null) {
            Repetition repetition = new Repetition();
            if (recurrence.getDailyRecurrence() != null) {
                repetition.setType(RegularRepetitionRule.DAILY);
                repetition.setREach(recurrence.getDailyRecurrence().getInterval());
            } else if (recurrence.getWeeklyRecurrence() != null) {
                repetition.setType(RegularRepetitionRule.WEEKLY);
                repetition.setREach(recurrence.getWeeklyRecurrence().getInterval());
                ListF<DayOfWeekType> daysOfWeek = Cf.x(recurrence.getWeeklyRecurrence().getDaysOfWeek());

                ListF<DayOfWeek> ds = daysOfWeek.map(DayOfWeek.byDayOfWeekTypeF()).unique().sorted();
                repetition.setRWeeklyDays(ds.map(DayOfWeek.getDbValueF()).mkString(","));

            } else if (recurrence.getAbsoluteMonthlyRecurrence() != null) {
                // XXX ensure that 'dayOfMonth' conforms with event, otherwise UnsupportedRepetitionException
                repetition.setType(RegularRepetitionRule.MONTHLY_NUMBER);
                repetition.setREach(recurrence.getAbsoluteMonthlyRecurrence().getInterval());
            } else if (recurrence.getRelativeMonthlyRecurrence() != null) {
                // XXX ensure that 'dayOfWeekIndex' conforms with event's start day index, otherwise UnsupportedRepetitionException
                // XXX ensure that 'daysOfWeek' conforms with event's start day, otherwise UnsupportedRepetitionException
                repetition.setType(RegularRepetitionRule.MONTHLY_DAY_WEEKNO);
                repetition.setREach(recurrence.getRelativeMonthlyRecurrence().getInterval());
                boolean isLast = recurrence.getRelativeMonthlyRecurrence().getDayOfWeekIndex() == DayOfWeekIndexType.LAST;
                repetition.setRMonthlyLastweek(isLast);
            } else if (recurrence.getAbsoluteYearlyRecurrence() != null) {
                // XXX ensure that 'month' and 'dayOfMonth' conform with event, otherwise UnsupportedRepetitionException
                repetition.setType(RegularRepetitionRule.YEARLY);
                repetition.setREach(1);
            } else {
                throw new UnsupportedRepetitionException(
                        "Unsupported recurrence pattern type found: "
                        + YandexToStringBuilder.reflectionToString(recurrence));
            }

            Instant eventStartTs = eventData.getEvent().getStartTs();
            Instant eventEndTs = eventData.getEvent().getEndTs();
            if (recurrence.getNoEndRecurrence() != null) {
                repetition.setDueTs(Option.empty());
            } else if (recurrence.getEndDateRecurrence() != null) {
                Instant dueTs = getRecurrenceDueTs(recurrence, eventStartTs, eventTz);
                repetition.setDueTs(dueTs);
            } else if (recurrence.getNumberedRecurrence() != null) {
                int occurrences = recurrence.getNumberedRecurrence().getNumberOfOccurrences();

                // IMPORTANT exchange ignores exdates, rdates and recurrence-ids when calculating last instance // ssytnik@
                // TODO getting info from calItem.getLastOccurrence() would be much more simple;
                // however, it gives less result if exdates are in the end. Check if it's a problem.
                InstantInterval eventInterval = new InstantInterval(eventStartTs, eventEndTs);
                RepetitionInstanceInfo rii = RepetitionInstanceInfo.create(eventInterval, eventTz, Option.of(repetition));

                // XXX why date? is it guaranteed to be start of day in tz?
                Instant lastInstanceDate = RepetitionUtils.instanceStart(rii, occurrences);

                Instant dueTs = EventRoutines.calculateDueTsFromIcsUntilTs(eventStartTs, lastInstanceDate, eventTz);
                repetition.setDueTs(dueTs);
            } else {
                throw new UnsupportedRepetitionException(
                        "Unsupported recurrence range type found: "
                        + YandexToStringBuilder.reflectionToString(recurrence));
            }

            return repetition;
        }
        return RepetitionRoutines.createNoneRepetition();
    }

    public static Instant getRecurrenceDueTs(RecurrenceType recurrence, Instant eventStartMs, DateTimeZone tz) {
        XMLGregorianCalendar endDate = recurrence.getEndDateRecurrence().getEndDate();
        LocalDate dueDate = LocalDate.fromCalendarFields(endDate.toGregorianCalendar());
        return EventRoutines.calculateDueTsFromUntilDate(eventStartMs, dueDate, tz);
    }

    private ListF<Rdate> convertExdates() {
        ListF<Rdate> exdates = Cf.arrayList();
        if (calItem.getDeletedOccurrences() != null) {
            ListF<DeletedOccurrenceInfoType> occurrences =
                Cf.x(calItem.getDeletedOccurrences().getDeletedOccurrence());
            for (DeletedOccurrenceInfoType occurrence : occurrences) {
                exdates.add(RepetitionUtils.consExdate(
                        EwsUtils.xmlGregorianCalendarInstantToInstant(occurrence.getStart())));
            }
        }
        return exdates;
    }

    public static EventData convert(
            CalendarItemType calItem, UidOrResourceId subjectId, Option<Email> realOrganzierO,
            Function<ListF<Email>, ListF<Email>> selectResourceEmailsF)
    {
        ExchangeEventDataConverter conv =
                new ExchangeEventDataConverter(calItem, Option.of(subjectId), realOrganzierO, selectResourceEmailsF);
        return conv.convert();
    }

    public static EventData convertWithoutExchangeDataAndRealOrganizer(
            CalendarItemType calItem, Function<ListF<Email>, ListF<Email>> selectResourceEmailsF)
    {
        ExchangeEventDataConverter conv =
                new ExchangeEventDataConverter(calItem, Option.empty(), Option.empty(), selectResourceEmailsF);
        return conv.convert();
    }

    public static ListF<String> getRecurrenceItemsId(CalendarItemType calItem) {
        ListF<String> recurrencesIds = Cf.arrayList();
        if (calItem.getModifiedOccurrences() != null) {
            ListF<OccurrenceInfoType> occurrences = Cf.x(calItem.getModifiedOccurrences().getOccurrence());
            for (OccurrenceInfoType occurrence : occurrences) {
                recurrencesIds.add(occurrence.getItemId().getId());
            }
        }
        return recurrencesIds;
    }

    public static ListF<AttendeeType> getAllAttendees(CalendarItemType calItem) {
        ListF<AttendeeType> attendees = Cf.arrayList();
        attendees.addAll(getAttendeesList(calItem.getOptionalAttendees()));
        attendees.addAll(getAttendeesList(calItem.getRequiredAttendees()));
        attendees.addAll(getAttendeesList(calItem.getResources()));
        return attendees;
    }

    private static ListF<AttendeeType> getAttendeesList(NonEmptyArrayOfAttendeesType arrayOfAttendees) {
        if (arrayOfAttendees != null) {
            return Cf.toList(arrayOfAttendees.getAttendee());
        } else {
            return Cf.list();
        }
    }

    public static ListF<Email> getAttendeeEmails(CalendarItemType calItem) {
        Function<AttendeeType, Option<Email>> getEmailOF = attendee -> {
            EmailAddressType mailbox = attendee.getMailbox();
            try {
                return Option.of(EwsUtils.getEmail(mailbox));
            } catch (Exception ex) {
                logger.warn("skipping attendee with invalid or empty email: "
                        + YandexToStringBuilder.reflectionToString(mailbox), ex.getMessage());
                return Option.empty();
            }
        };
        return getAllAttendees(calItem).filterMap(getEmailOF);
    }

    public static Option<Email> getOrganizerEmailSafe(CalendarItemType calItem) {
        final SingleRecipientType organizer = calItem.getOrganizer();
        if (organizer != null) {
            final EmailAddressType mailbox = organizer.getMailbox();
            if (mailbox != null) {
                return EwsUtils.getEmailOSafe(mailbox);
            }
        }
        return Option.empty();
    }
}
