package ru.yandex.calendar.frontend.ews.proxy;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.SocketTimeoutException;

import javax.xml.ws.WebServiceException;

import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.util.ClassUtils;

import ru.yandex.bolts.collection.Try;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * Sends time consumed by a particular ews proxy method, or its timeout, to ews monitoring
 * @see EwsProxyWrapperContextConfiguration
 */
public class EwsMonitoringProxy implements java.lang.reflect.InvocationHandler {
    private static final Logger logger = LoggerFactory.getLogger(EwsMonitoringProxy.class);

    private final EwsMonitoring ewsMonitoring;
    private final EwsProxy target;
    private final int requestAttempts;

    private EwsMonitoringProxy(EwsMonitoring ewsMonitoring, EwsProxy target, int requestAttempts) {
        this.ewsMonitoring = ewsMonitoring;
        this.target = target;
        this.requestAttempts = requestAttempts;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        try {
            Instant start = new Instant();
            boolean isRead = isRead(method.getName());

            Try<Object> r = null;

            for (int attempt = 1; attempt <= requestAttempts; ++attempt) {
                r = Try.tryCatchException(() -> method.invoke(target, args));

                if (r.isSuccess()) {
                    val executionTimeMillis = new Duration(start, new Instant()).getMillis();
                    ewsMonitoring.reportMethodSuccess(method.getName(), executionTimeMillis);
                    return r.get();
                }
                boolean isAdmissible = ExceptionUtils.unwrap(r.getThrowable(), SocketTimeoutException.class)
                        .exists(e -> e.getMessage().endsWith("connect timed out"));

                if (!isRead && !isAdmissible || ThreadLocalTimeout.expiresWithin(Duration.ZERO)) {
                    throw r.getThrowable();
                }
                if (attempt < requestAttempts) {
                    logger.warn("Failed to invoke method {} #{}: {}",
                            method.getName(), attempt, ExceptionUtils.getAllMessages(r.getThrowable()));
                }
            }
            throw r == null ? new IllegalStateException("Not executed") : r.getThrowable();

        } catch (InvocationTargetException e) {
            if (e.getCause() instanceof WebServiceException) { // TODO verify, maybe add more // ssytnik@
                ewsMonitoring.reportTimeoutException();
            } else {
                ewsMonitoring.reportOtherException();
            }
            throw e.getCause();
        }
    }

    private boolean isRead(String methodName) {
        return methodName.startsWith("get")
                || methodName.startsWith("find")
                || methodName.startsWith("pull")
                || methodName.startsWith("ping");
    }


    public static EwsProxy newEwsMonitoringProxy(EwsMonitoring ewsMonitoring, EwsProxy object, int requestAttempts) {
        return (EwsProxy) java.lang.reflect.Proxy.newProxyInstance(EwsProxy.class.getClassLoader(),
                ClassUtils.getAllInterfacesForClass(EwsProxy.class),
                new EwsMonitoringProxy(ewsMonitoring, object, requestAttempts));
    }
}
