package ru.yandex.calendar.frontend.ews.proxy;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.boot.EwsAliveHandler;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.misc.db.PingerThread;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Daniel Brylev <dbrylev@yandex-team.ru>
 */
public class EwsProxyChecker {
    private static final Logger logger = LoggerFactory.getLogger(EwsProxyChecker.class);

    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EwsProxy ewsProxy;
    @Autowired
    private EwsAliveHandler ewsAliveHandler;

    private final EwsProxyCheckWorker checkWorker;

    private final long delayMs;

    public EwsProxyChecker(Duration delayBetweenChecks) {
        this.delayMs = delayBetweenChecks.getMillis();
        checkWorker = new EwsProxyCheckWorker();
    }

    public boolean isCheckingEnabled() {
        return passportAuthDomainsHolder.containsYandexTeamRu();
    }

    public boolean isEwsAvailable() {
        return checkWorker.isAlive() && checkWorker.isAvailable();
    }

    @PostConstruct
    public void init() {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) {
            logger.info("Checking will not be performed because this is not yandex-team application");
            return;
        }
        if (!ewsAliveHandler.isEwsAlive()) return;
        checkWorker.start();
    }

    @PreDestroy
    public void destroy() {
        if (checkWorker.isAlive()) {
            checkWorker.stopGracefully();
        }
    }

    private class EwsProxyCheckWorker extends PingerThread {
        private EwsProxyCheckWorker() {
            super("ews-proxy-checker", "Checks if exchange service is available");
        }

        @Override
        protected void ping() throws Exception {
            if (!ewsAliveHandler.isEwsAlive()) return;
            if (!ewsProxy.ping()) {
                throw new Exception("Exchange server request failed");
            }
        }

        @Override
        protected long delayBetweenExecutionsMillis() {
            return delayMs;
        }

        @Override
        protected void stateChanged(boolean available, boolean first) {

        }
    }
}
