package ru.yandex.calendar.frontend.ews.proxy;

import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.xml.bind.JAXBElement;

import com.microsoft.schemas.exchange.services._2006.messages.BaseResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.CreateItemType;
import com.microsoft.schemas.exchange.services._2006.messages.DeleteItemType;
import com.microsoft.schemas.exchange.services._2006.messages.ExchangeServicePortType;
import com.microsoft.schemas.exchange.services._2006.messages.FindItemResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.FindItemType;
import com.microsoft.schemas.exchange.services._2006.messages.GetEventsResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.GetEventsType;
import com.microsoft.schemas.exchange.services._2006.messages.GetItemType;
import com.microsoft.schemas.exchange.services._2006.messages.GetUserOofSettingsRequest;
import com.microsoft.schemas.exchange.services._2006.messages.ItemInfoResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.ResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.ResponseMessageType.MessageXml;
import com.microsoft.schemas.exchange.services._2006.messages.SubscribeResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.SubscribeType;
import com.microsoft.schemas.exchange.services._2006.messages.UnsubscribeType;
import com.microsoft.schemas.exchange.services._2006.messages.UpdateItemResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.UpdateItemType;
import com.microsoft.schemas.exchange.services._2006.types.BaseItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.BaseSubscriptionRequestType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemCreateOrDeleteOperationType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemUpdateOperationType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarViewType;
import com.microsoft.schemas.exchange.services._2006.types.ConflictResolutionType;
import com.microsoft.schemas.exchange.services._2006.types.DefaultShapeNamesType;
import com.microsoft.schemas.exchange.services._2006.types.DisposalType;
import com.microsoft.schemas.exchange.services._2006.types.DistinguishedFolderIdNameType;
import com.microsoft.schemas.exchange.services._2006.types.DistinguishedFolderIdType;
import com.microsoft.schemas.exchange.services._2006.types.EmailAddress;
import com.microsoft.schemas.exchange.services._2006.types.FieldOrderType;
import com.microsoft.schemas.exchange.services._2006.types.IndexBasePointType;
import com.microsoft.schemas.exchange.services._2006.types.IndexedPageViewType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeType;
import com.microsoft.schemas.exchange.services._2006.types.ItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.ItemQueryTraversalType;
import com.microsoft.schemas.exchange.services._2006.types.ItemResponseShapeType;
import com.microsoft.schemas.exchange.services._2006.types.ItemType;
import com.microsoft.schemas.exchange.services._2006.types.MessageDispositionType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfAllItemsType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfBaseFolderIdsType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfBaseItemIdsType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfFieldOrdersType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfItemChangeDescriptionsType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfItemChangesType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfNotificationEventTypesType;
import com.microsoft.schemas.exchange.services._2006.types.NonEmptyArrayOfPathsToElementType;
import com.microsoft.schemas.exchange.services._2006.types.NotificationEventTypeType;
import com.microsoft.schemas.exchange.services._2006.types.PullSubscriptionRequestType;
import com.microsoft.schemas.exchange.services._2006.types.PushSubscriptionRequestType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseClassType;
import com.microsoft.schemas.exchange.services._2006.types.RestrictionType;
import com.microsoft.schemas.exchange.services._2006.types.SortDirectionType;
import com.microsoft.schemas.exchange.services._2006.types.TargetFolderIdType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import com.microsoft.schemas.exchange.services._2006.types.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Minutes;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.boot.SpecialHosts;
import ru.yandex.calendar.frontend.ews.EwsConditions;
import ru.yandex.calendar.frontend.ews.EwsErrorCodes;
import ru.yandex.calendar.frontend.ews.EwsErrorResponseException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.subscriber.EwsSubscribeResult;
import ru.yandex.calendar.util.Environment;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.InstantInterval;

/**
 * Details of exchange calendar system: http://msdn.microsoft.com/en-us/library/cc500377.aspx
 */
@Slf4j
public class EwsProxyImpl implements EwsProxy {
    private static final String PRIMARY_ELEMENT_NAME = "Primary";
    private static final int MAX_SUBSCRIBE_ATTEMPTS_COUNT = 2;

    private final ExchangeService proxy;
    private final Email pingingEmail;

    private final EnvironmentType environmentType;
    private final int ewsJettyHttpPort;

    private final ConflictResolutionType conflictResolution = ConflictResolutionType.AUTO_RESOLVE;

    public EwsProxyImpl(
            ExchangeServicePortType proxy,
            Email pingingEmail,
            EnvironmentType environmentType, int ewsJettyHttpPort)
    {
        this.proxy = new ExchangeService(proxy);
        this.pingingEmail = pingingEmail;
        this.environmentType = environmentType;
        this.ewsJettyHttpPort = ewsJettyHttpPort;
    }

    private static String errorResponseToMessage(ResponseMessageType response) {
        String message =
                "Error " + response.getResponseCode() +
                ": " + response.getMessageText();

        val values = EwsErrorResponseException.getValues(response.getMessageXml());

        if (!values.isEmpty()) {
            if (!message.endsWith(".")) {
                message += ".";
            }
            message += " Xml: " + String.join(", ", values);
        }
        return message;
    }

    private void logResponse(ResponseMessageType response) {
        log.debug("Response class: {}", response.getResponseClass());
        log.debug("Response code: {}", response.getResponseCode());
        log.debug("Response message: {}", response.getMessageText());
    }

    private String getNtfServiceEndpointUrl() {
        val hostName = Environment.isProductionOrPre(environmentType) ?
                SpecialHosts.EXCHANGE_BALANCER : HostnameUtils.localHostname();
        val portSuffix = environmentType == EnvironmentType.DEVELOPMENT ?
                ":" + ewsJettyHttpPort : ""; // no ews proxying at dev
        return "http://" + hostName + portSuffix + "/soap/NtfExchange";
    }

    private static DistinguishedFolderIdType createCalendarFolderIdType() {
        val calendarFolder = new DistinguishedFolderIdType();
        calendarFolder.setId(DistinguishedFolderIdNameType.CALENDAR);
        return calendarFolder;
    }

    @SuppressWarnings("unchecked")
    private Option<Email> getPrimarySmtpAddress(MessageXml messageXml) {
        val elements = Cf.x(messageXml.getAny());
        for (Object obj : elements) {
            val value = ((JAXBElement<Value>) obj).getValue();
            if (PRIMARY_ELEMENT_NAME.equals(value.getName())) {
                return Option.of(new Email(value.getValue()));
            }
        }
        log.warn("Primary email not found");
        return Option.empty();
    }

    @Override
    public EwsSubscribeResult subscribeToPush(Email email) {
        // Create a new subscription
        val pushSubcrReq = new PushSubscriptionRequestType();
        // Subscribe to the calendar folder
        val folderIds = new NonEmptyArrayOfBaseFolderIdsType();
        folderIds.getFolderIdOrDistinguishedFolderId().add(createCalendarFolderIdType());
        pushSubcrReq.setFolderIds(folderIds);
        // Subscribe to create, delete, modify event actions
        pushSubcrReq.setEventTypes(notificationEventTypes());
        // SetF push notification timeout
        pushSubcrReq.setStatusFrequency(EwsUtils.STATUS_FREQUENCY_IN_MINUTES);
        // Identify the location of the client Web service
        val ntfServiceEndpointUrl = getNtfServiceEndpointUrl();
        log.debug("Notification endpoint url: {}", ntfServiceEndpointUrl);
        pushSubcrReq.setURL(ntfServiceEndpointUrl);
        // Create subscribe request
        val request = new SubscribeType();
        request.setPushSubscriptionRequest(pushSubcrReq);
        // Sends subscribe request and get response
        return subscribeInner(email, request);
    }

    @Override
    public EwsSubscribeResult subscribeToPull(Email email, Minutes timeout, Option<String> watermark) {
        val pullRequest = new PullSubscriptionRequestType();
        pullRequest.setTimeout(timeout.getMinutes());
        pullRequest.setEventTypes(notificationEventTypes());
        watermark.forEach(pullRequest::setWatermark);

        val folderIds = new NonEmptyArrayOfBaseFolderIdsType();
        folderIds.getFolderIdOrDistinguishedFolderId().add(createCalendarFolderIdType());
        pullRequest.setFolderIds(folderIds);

        val request = new SubscribeType();
        request.setPullSubscriptionRequest(pullRequest);

        return subscribeInner(email, request);
    }

    @Override
    public ResponseMessageType unsubscribe(String subscriptionId) {
        val request = new UnsubscribeType();
        request.setSubscriptionId(subscriptionId);

        return singleResponse(proxy.unsubscribe(request));
    }

    private NonEmptyArrayOfNotificationEventTypesType notificationEventTypes() {
        val eventTypes = new NonEmptyArrayOfNotificationEventTypesType();

        eventTypes.getEventType().add(NotificationEventTypeType.CREATED_EVENT);
        eventTypes.getEventType().add(NotificationEventTypeType.DELETED_EVENT);
        eventTypes.getEventType().add(NotificationEventTypeType.MODIFIED_EVENT);
        eventTypes.getEventType().add(NotificationEventTypeType.MOVED_EVENT);

        return eventTypes;
    }

    private EwsSubscribeResult subscribeInner(Email email, SubscribeType request) {
        int attemptNumber = 0;
        SubscribeResponseMessageType response;
        Option<String> subscripitonId = Option.empty();
        Option<String> watermark = Option.empty();
        do {
            // SetF calendar user
            val calendarUser = EwsUtils.createEmailAddressType(email);
            val subscribeRequests = Option.<BaseSubscriptionRequestType>empty()
                    .plus(Option.ofNullable(request.getPushSubscriptionRequest()))
                    .plus(Option.ofNullable(request.getPullSubscriptionRequest()));

            subscribeRequests.forEach(req -> req.getFolderIds().getFolderIdOrDistinguishedFolderId()
                    .forEach(folder -> ((DistinguishedFolderIdType) folder).setMailbox(calendarUser)));

            response = singleResponse(proxy.subscribe(request));
            logResponse(response);

            if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
                subscripitonId = Option.of(response.getSubscriptionId());
                watermark = Option.ofNullable(response.getWatermark());
                break;
            } else if (response.getResponseCode().equals(EwsErrorCodes.NON_PRIMARY_SMTP_ADDRESS)) {
                email = getPrimarySmtpAddress(response.getMessageXml()).get();
                log.debug("Email for subscription was changed to {}", email);
            } else {
                break;
            }
            attemptNumber++;
        } while (attemptNumber < MAX_SUBSCRIBE_ATTEMPTS_COUNT);
        return new EwsSubscribeResult(subscripitonId, watermark, response, email);
    }

    @Override
    public ListF<CalendarItemType> getEvents(ListF<? extends  BaseItemIdType> itemIds, final boolean idOnly) {
        return getEvents(itemIds, createResponseShapeType(idOnly));
    }

    @Override
    public ListF<CalendarItemType> getEvents(
            ListF<? extends BaseItemIdType> itemIds, ListF<UnindexedFieldURIType> fields, ListF<String> extendedFields)
    {
        return getEvents(itemIds, createResponseShapeType(fields, extendedFields));
    }

    private ListF<CalendarItemType> getEvents(
            ListF<? extends BaseItemIdType> itemIds, ItemResponseShapeType itemShape)
    {
        Function<ListF<BaseItemIdType>, GetItemType> requestF = (ids) -> {
            GetItemType request = new GetItemType();
            request.setItemShape(itemShape);
            request.setItemIds(createCalendarItemIdsArray(ids));

            return request;
        };
        // http://msdn.microsoft.com/en-us/library/aa565934.aspx
        // Error response is not documented there
        // If someone knows better link, please paste it here
        // stepancheg@
        List<ItemInfoResponseMessageType> responses = responses(proxy.getItem(requestF.apply(itemIds.cast())));

        // Parse server response

        ListF<CalendarItemType> resCalItems = Cf.arrayList();

        for (int i = 0; i < responses.size(); ++i) {
            val response = responses.get(i);
            if (response.getResponseClass() == ResponseClassType.SUCCESS) {
                for (ItemType item : response.getItems().getItemOrMessageOrCalendarItem()) {
                    if (item instanceof CalendarItemType) {
                        resCalItems.add((CalendarItemType) item);
                    }
                }
            } else if (EwsErrorCodes.BATCH_PROCESSING_STOPPED.equals(response.getResponseCode())) {
                resCalItems.addAll(getEvents(itemIds.subList(i, itemIds.size()), itemShape));
                break;

            } else {
                log.warn("Failed to fetch event: {}", errorResponseToMessage(response));
            }
        }

        return resCalItems;
    }

    @Override
    public ListF<CalendarItemType> findInstanceEvents(
            Email email, InstantInterval interval, boolean idOnly)
    {
        return findEventsInner(email, Option.of(interval), createResponseShapeType(idOnly), false);
    }

    @Override
    public ListF<CalendarItemType> findInstanceEvents(
            Email email, InstantInterval interval, ListF<UnindexedFieldURIType> fields, ListF<String> extendedFields)
    {
        return findEventsInner(email, Option.of(interval), createResponseShapeType(fields, extendedFields), false);
    }

    @Override
    public ListF<CalendarItemType> findMasterAndSingleEvents(
            Email email, Option<InstantInterval> intervalO, boolean idOnly)
    {
        return findEventsInner(email, intervalO, createResponseShapeType(idOnly), true);
    }

    @Override
    public ListF<CalendarItemType> findMasterAndSingleEvents(
            Email email, Option<InstantInterval> interval,
            List<UnindexedFieldURIType> fields, ListF<String> extendedFields)
    {
        return findEventsInner(email, interval, createResponseShapeType(fields, extendedFields), true);
    }

    private ListF<CalendarItemType> findEventsInner(
            Email email, Option<InstantInterval> intervalO,
            ItemResponseShapeType itemShape, boolean findMasterAndSingleEvents)
    {
        Supplier<FindItemType> requestF = () -> {
            val findItem = new FindItemType();
            findItem.setTraversal(ItemQueryTraversalType.SHALLOW);
            findItem.setItemShape(itemShape);
            findItem.setParentFolderIds(createFolderId(email));

            if (findMasterAndSingleEvents) {
                fillFindItemToFindMasterAndSingleEvents(findItem, intervalO.toOptional(), Optional.empty());
            } else {
                fillFindItemToFindInstanceEvents(findItem, intervalO.get());
            }
            return findItem;
        };

        List<FindItemResponseMessageType> responses = responses(proxy.findItem(requestF.get()));

        // Parse server response
        List<CalendarItemType> resCalItems = responses.stream()
                .flatMap(EwsProxyImpl::getCalendarItems)
                .collect(Collectors.toUnmodifiableList());
        return Cf.toArrayList(resCalItems);
    }

    @Override
    public GetEventsResponseMessageType pull(String subscriptionId, String watermark) {
        val request = new GetEventsType();
        request.setSubscriptionId(subscriptionId);
        request.setWatermark(watermark);

        return singleResponse(proxy.getEvents(request));
    }

    private void fillFindItemToFindInstanceEvents(
            FindItemType findItem, InstantInterval interval)
    {
        val calendarView = new CalendarViewType();
        val zone = DateTimeZone.UTC; // CalendarViewType doesn't support timezone
        calendarView.setStartDate(EwsUtils.instantToXMLGregorianCalendar(interval.getStart(), zone));
        calendarView.setEndDate(EwsUtils.instantToXMLGregorianCalendar(interval.getEnd(), zone));

        findItem.setCalendarView(calendarView);
    }

    private static void fillFindItemToFindMasterAndSingleEvents(
            FindItemType findItem, Optional<InstantInterval> intervalO, Optional<Integer> limit)
    {
        val view = new IndexedPageViewType();
        view.setBasePoint(IndexBasePointType.BEGINNING);
        view.setOffset(0);
        findItem.setIndexedPageItemView(view);
        limit.ifPresent(view::setMaxEntriesReturned);

        setSortingByStart(findItem);

        if (intervalO.isPresent()) {
            val restriction = new RestrictionType();
            restriction.setSearchExpression(
                EwsConditions.createEventIntervalOverlapCondition(intervalO.get())
            );
            findItem.setRestriction(restriction);
        }
    }

    private static void setSortingByStart(FindItemType findItem) {
        val sortOrder = new NonEmptyArrayOfFieldOrdersType();
        val startFieldOrder = new FieldOrderType();
        startFieldOrder.setPath(EwsConditions.createUnindexedFieldPath(UnindexedFieldURIType.CALENDAR_START));
        startFieldOrder.setOrder(SortDirectionType.ASCENDING);
        sortOrder.getFieldOrder().add(startFieldOrder);
        findItem.setSortOrder(sortOrder);
    }

    @Override
    public Tuple2List<String, ResponseMessageType> deleteEvents(
            ListF<String> exchangeIds, CalendarItemCreateOrDeleteOperationType operationType)
    {
        Tuple2List<String, ResponseMessageType> responseList = Tuple2List.arrayList();

        if (!exchangeIds.isEmpty()) {
            Function<ListF<BaseItemIdType>, DeleteItemType> requestF = itemIds -> {
                val request = new DeleteItemType();
                request.setDeleteType(DisposalType.HARD_DELETE);
                request.setItemIds(createCalendarItemIdsArray(itemIds));
                request.setSendMeetingCancellations(operationType);

                return request;
            };

            List<ResponseMessageType> responses = responses(proxy.deleteItem(requestF.apply(exchangeIds.map(EwsUtils::createItemId))));

            for (int i = 0; i < responses.size(); i++) {
                responseList.add(exchangeIds.get(i), responses.get(i));
            }
        } else {
            log.debug("Nothing to delete");
        }
        return responseList;
    }

    @Override
    public Tuple2List<ItemType, ResponseMessageType> createItems(
            final Option<Email> emailO, ListF<? extends ItemType> items, final MessageDispositionType disposition,
            CalendarItemCreateOrDeleteOperationType operationType)
    {
        Tuple2List<ItemType, ResponseMessageType> responseList = Tuple2List.arrayList();
        if (items.isNotEmpty()) {
            Function<ListF<ItemType>, CreateItemType> requestF = (requestedItems) -> {
                val request = new CreateItemType();
                // Set folder
                if (emailO.isPresent()) {
                    val targetFolderIdType = new TargetFolderIdType();
                    targetFolderIdType.setDistinguishedFolderId(createCalendarFolderForEmail(emailO.get()));
                    request.setSavedItemFolderId(targetFolderIdType);
                }
                val itemsArray = new NonEmptyArrayOfAllItemsType();
                itemsArray.getItemOrMessageOrCalendarItem().addAll(requestedItems);
                request.setItems(itemsArray);
                request.setMessageDisposition(disposition);
                request.setSendMeetingInvitations(operationType);

                return request;
            };

            List<ItemInfoResponseMessageType> responses = responses(proxy.createItem(requestF.apply(items.cast())));

            // akirakozov: response.length should be = itemIds.length
            for (int i = 0; i < responses.size(); i++) {
                val response = responses.get(i);
                val item = items.get(i);
                responseList.add(Tuple2.tuple(item, response));
                if (response.getResponseClass() != ResponseClassType.SUCCESS) {
                    val itemDescription =
                        item.getItemId() != null ? item.getItemId().getId() : Integer.toString(i);
                    log.warn("Send response failed for event {} : {}", itemDescription, errorResponseToMessage(response));
                }
            }
        }
        return responseList;
    }

    @Override
    public ResponseMessageType updateItem(
            ItemIdType itemId, ListF<? extends ItemChangeDescriptionType> changeDescriptions)
    {
        return updateItem(itemId, changeDescriptions, CalendarItemUpdateOperationType.SEND_TO_NONE);
    }

    @Override
    public ResponseMessageType updateItem(
            ItemIdType itemId, ListF<? extends ItemChangeDescriptionType> changeDescriptions,
            CalendarItemUpdateOperationType operationType)
    {
        val request = new UpdateItemType();

        val updates = new NonEmptyArrayOfItemChangeDescriptionsType();
        updates.getAppendToItemFieldOrSetItemFieldOrDeleteItemField().addAll(changeDescriptions);

        val itemChangeType = new ItemChangeType();
        itemChangeType.setUpdates(updates);
        itemChangeType.setItemId(itemId);

        val changes = new NonEmptyArrayOfItemChangesType();
        changes.getItemChange().add(itemChangeType);

        request.setItemChanges(changes);
        request.setConflictResolution(conflictResolution);
        request.setSendMeetingInvitationsOrCancellations(operationType);

        UpdateItemResponseMessageType response = singleResponse(proxy.updateItem(request));

        if (response.getResponseClass() != ResponseClassType.SUCCESS) {
            log.warn("Update failed for event {} : {}", itemId.getId(), errorResponseToMessage(response));
        }
        return response;
    }

    @Override
    public boolean ping() {
        return ping(proxy, pingingEmail);
    }

    private static boolean ping(ExchangeService proxy, Email email) {
        try {
            log.debug("Sending request to exchange server to check its availability using {} ...", email);
            val request = new GetUserOofSettingsRequest();
            val pingingEmailAddress = new EmailAddress();
            pingingEmailAddress.setAddress(email.getEmail());
            request.setMailbox(pingingEmailAddress);

            proxy.getUserOofSettings(request);

            log.debug("Exchange server responded normally for {}", email);
            return true;

        } catch (Exception e) {
            log.warn("Exchange server request failed for " + email, e);
            return false;
        }
    }

    private NonEmptyArrayOfBaseItemIdsType createCalendarItemIdsArray(ListF<? extends BaseItemIdType> ids) {
        val itemsArray = new NonEmptyArrayOfBaseItemIdsType();
        itemsArray.getItemIdOrOccurrenceItemIdOrRecurringMasterItemId().addAll(ids);
        return itemsArray;
    }

    private ItemResponseShapeType createResponseShapeType(boolean idOnly) {
        val itemShape = new ItemResponseShapeType();
        val baseShape = idOnly ?
                DefaultShapeNamesType.ID_ONLY :
                DefaultShapeNamesType.ALL_PROPERTIES;

        itemShape.setBaseShape(baseShape);

        if (!idOnly) {
            val paths = new NonEmptyArrayOfPathsToElementType();
            paths.getPath().add(EwsUtils.createExtendedFieldPath(EwsUtils.EXTENDED_PROPERTY_SOURCE));
            paths.getPath().add(EwsUtils.createExtendedFieldPath(EwsUtils.EXTENDED_PROPERTY_ORGANIZER));
            paths.getPath().add(EwsUtils.createExtendedFieldPath(EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID));
            itemShape.setAdditionalProperties(paths);

            itemShape.getAdditionalProperties().getPath()
                    .add(EwsUtils.createUnindexedFieldPath(UnindexedFieldURIType.CALENDAR_START_TIME_ZONE));

            return itemShape;

        } else {
            return itemShape;
        }
    }

    private ItemResponseShapeType createResponseShapeType(List<UnindexedFieldURIType> fields, ListF<String> extendedFields) {
        val itemShape = new ItemResponseShapeType();
        itemShape.setBaseShape(DefaultShapeNamesType.ID_ONLY);

        val paths = new NonEmptyArrayOfPathsToElementType();
        for (UnindexedFieldURIType field : fields) {
            paths.getPath().add(EwsUtils.createUnindexedFieldPath(field));
        }
        for (String extendedField : extendedFields) {
            paths.getPath().add(EwsUtils.createExtendedFieldPath(extendedField));
        }
        itemShape.setAdditionalProperties(paths);

        itemShape.getAdditionalProperties().getPath()
                .add(EwsUtils.createUnindexedFieldPath(UnindexedFieldURIType.CALENDAR_START_TIME_ZONE));

        return itemShape;
    }

    private static <T extends ResponseMessageType> T singleResponse(BaseResponseMessageType response) {
        List<T> responses = responses(response);
        Assert.hasSize(1, responses);
        return responses.get(0);
    }

    @SuppressWarnings("unchecked")
    private static <T extends ResponseMessageType> List<T> responses(BaseResponseMessageType message) {
        return message.getResponseMessages()
                .getCreateItemResponseMessageOrDeleteItemResponseMessageOrGetItemResponseMessage().stream()
                .map(el -> ((JAXBElement<T>) el).getValue())
                .collect(Collectors.toList());
    }

    private static NonEmptyArrayOfBaseFolderIdsType createFolderId(Email email) {
        val distinguishedFolderIdType = createCalendarFolderForEmail(email);
        val ids = new NonEmptyArrayOfBaseFolderIdsType();
        ids.getFolderIdOrDistinguishedFolderId().add(distinguishedFolderIdType);
        return ids;
    }

    private static DistinguishedFolderIdType createCalendarFolderForEmail(Email email) {
        val distinguishedFolderIdType = createCalendarFolderIdType();
        distinguishedFolderIdType.setMailbox(EwsUtils.createEmailAddressType(email));
        return distinguishedFolderIdType;
    }

    private static FindItemType createFindItem(Email email, Optional<Integer> limit, ItemResponseShapeType itemShape) {
        val findItem = new FindItemType();
        findItem.setTraversal(ItemQueryTraversalType.SHALLOW);
        findItem.setItemShape(itemShape);
        findItem.setParentFolderIds(createFolderId(email));

        fillFindItemToFindMasterAndSingleEvents(findItem, Optional.empty(), limit);

        return findItem;
    }

    private static Stream<CalendarItemType> getCalendarItems(FindItemResponseMessageType response) {
        if (response.getResponseClass() == ResponseClassType.SUCCESS) {
            return response.getRootFolder().getItems().getItemOrMessageOrCalendarItem().stream()
                    .filter(itemId -> {
                        final boolean isCalendarItem = itemId instanceof CalendarItemType;
                        if (!isCalendarItem) {
                            String itemDescription =
                                    itemId.getItemId() != null ? itemId.getItemId().getId() : "UnknownId";
                            log.debug("missing: " + itemDescription + ", " + itemId.getClass());
                        }
                        return isCalendarItem;
                    })
                    .map(t -> (CalendarItemType) t);
        } else {
            log.warn("Failed to fetch event: {}", errorResponseToMessage(response));
            return Stream.empty();
        }
    }

    @Override
    public List<CalendarItemType> findInstanceEventsForPurging(Email email, int limit) {
        val itemShape = createResponseShapeType(true);

        List<FindItemResponseMessageType> responses = responses(proxy.findItem(createFindItem(email, Optional.of(limit), itemShape)));

        return responses.stream()
                .flatMap(EwsProxyImpl::getCalendarItems)
                .collect(Collectors.toUnmodifiableList());
    }
}
