package ru.yandex.calendar.frontend.ews.proxy;

import java.util.Optional;

import com.microsoft.schemas.exchange.services._2006.messages.GetEventsResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.ItemInfoResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.messages.ResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.types.AcceptItemType;
import com.microsoft.schemas.exchange.services._2006.types.BaseItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.BodyType;
import com.microsoft.schemas.exchange.services._2006.types.BodyTypeType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemCreateOrDeleteOperationType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemTypeType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemUpdateOperationType;
import com.microsoft.schemas.exchange.services._2006.types.CancelCalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.DeclineItemType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import com.microsoft.schemas.exchange.services._2006.types.ItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.ItemType;
import com.microsoft.schemas.exchange.services._2006.types.MessageDispositionType;
import com.microsoft.schemas.exchange.services._2006.types.OccurrenceItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseClassType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseObjectCoreType;
import com.microsoft.schemas.exchange.services._2006.types.TentativelyAcceptItemType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import com.microsoft.schemas.exchange.services._2006.types.WellKnownResponseObjectType;
import org.joda.time.Instant;
import org.joda.time.Minutes;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Try;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.ews.EwsErrorCodes;
import ru.yandex.calendar.frontend.ews.EwsErrorResponseException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExtendedCalendarItemProperties;
import ru.yandex.calendar.frontend.ews.exp.EwsModifyingItemId;
import ru.yandex.calendar.frontend.ews.exp.OccurrenceId;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.frontend.ews.subscriber.EwsSubscribeResult;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

public class EwsProxyWrapper {
    private static final Logger logger = LoggerFactory.getLogger(EwsProxyWrapper.class);

    public static final CalendarItemCreateOrDeleteOperationType DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE =
            CalendarItemCreateOrDeleteOperationType.SEND_TO_NONE;
    public static final MessageDispositionType DEFAULT_DISPOSITION_TYPE = MessageDispositionType.SAVE_ONLY;

    private final EwsProxy ewsProxy;
    private final EnvironmentType environmentType;
    private final EventsLogger eventsLogger;

    public EwsProxyWrapper(EwsProxy ewsProxy, EnvironmentType environmentType, EventsLogger eventsLogger) {
        this.ewsProxy = ewsProxy;
        this.environmentType = environmentType;
        this.eventsLogger = eventsLogger;
    }

    public EwsSubscribeResult subscribeToPush(Email email){
        return ewsProxy.subscribeToPush(email);
    }

    public EwsSubscribeResult subscribeToPull(Email email, Option<String> watermark) {
        return subscribeToPull(email, Minutes.minutes(EwsUtils.PULL_SUBSCRIPTION_TIMEOUT_IN_MINUTES), watermark);
    }

    public EwsSubscribeResult subscribeToPull(Email email, Minutes timeout, Option<String> watermark) {
        return ewsProxy.subscribeToPull(email, timeout, watermark);
    }

    public void unsubscribeSafe(String subscriptionId) {
        Option<String> error;
        try {
            error = Option.of(ewsProxy.unsubscribe(subscriptionId))
                    .filterNot(r -> r.getResponseClass() == ResponseClassType.SUCCESS)
                    .map(EwsErrorResponseException::message);

        } catch (Exception e) {
            error = Option.of(ExceptionUtils.getAllMessages(e));
        }
        error.forEach(e -> logger.warn("Failed to unsubscribe " + subscriptionId + ": " + e));
    }

    public GetEventsResponseMessageType pull(String subscriptionId, String watermark) {
        return ewsProxy.pull(subscriptionId, watermark);
    }

    /** @return filled event: recurring master, single, occurrence or exception */
    public Option<CalendarItemType> getEvent(String exchangeId) {
        Validate.notEmpty(exchangeId);
        return getEvents(Cf.list(exchangeId)).singleO();
    }

    /** @return filled events: recurring masters, single, occurrences and exceptions */
    public ListF<CalendarItemType> getEvents(ListF<String> exchangeIds) {
        return getEvents(exchangeIds.map(EwsUtils.createItemIdF()), false);
    }

    public ListF<CalendarItemType> getEvents(ListF<String> exchangeIds, ListF<UnindexedFieldURIType> fields) {
        return getEvents(exchangeIds, fields, Cf.<String>list());
    }

    public ListF<CalendarItemType> getEvents(
            ListF<String> exchangeIds, final ListF<UnindexedFieldURIType> fields, final ListF<String> extendedFields)
    {
        return exchangeIds.paginate(200)
                .flatMap(exchangeIds1 -> ewsProxy.getEvents(exchangeIds1.map(EwsUtils.createItemIdF()), fields, extendedFields));
    }

    private ListF<CalendarItemType> getEvents(
            ListF<? extends  BaseItemIdType> itemIds, final boolean idOnly)
    {
        final int maxPageSize = 100;
        return itemIds.paginate(maxPageSize).flatMap(a -> ewsProxy.getEvents(a, idOnly));
    }

    public Option<CalendarItemType> getMasterOrSingleEvent(String instanceExchangeId) {
        return getMasterAndSingleEvents(Cf.list(instanceExchangeId)).singleO();
    }

    public ListF<CalendarItemType> getMasterAndSingleEvents(ListF<String> instanceExchangeIds) {
        return getMasterAndSingleEvents(instanceExchangeIds, false);
    }

    public ListF<String> getMasterAndSingleEventExchangeIds(ListF<String> instanceExchangeIds) {
        return getMasterAndSingleEvents(instanceExchangeIds, true).map(EwsUtils.calendarItemExchangeIdF());
    }

    /** @return filled recurring masters and single events, by instances: single, occurrences and exceptions  */
    private ListF<CalendarItemType> getMasterAndSingleEvents(ListF<String> instanceExchangeIds, boolean idOnly) {
        ListF<CalendarItemType> events = getEvents(
                instanceExchangeIds.stableUnique().map(EwsUtils.createItemIdF()), idOnly);

        Tuple2<ListF<CalendarItemType>, ListF<CalendarItemType>> partitionByType = events
                .partition(a -> a.getCalendarItemType() == CalendarItemTypeType.SINGLE);

        ListF<CalendarItemType> singleEvents = partitionByType.get1();

        final ListF<BaseItemIdType> recurItemIds =
                partitionByType.get2()
                .map(EwsUtils.calendarItemExchangeIdF())
                .map(EwsUtils.createRecurringMasterItemIdTypeF());
        ListF<CalendarItemType> masterEventsByRecurrenceInstances =
                getEvents(recurItemIds, idOnly);

        return singleEvents.plus(masterEventsByRecurrenceInstances).stableUniqueBy(EwsUtils.calendarItemExchangeIdF());
    }

    /** @return filled recurring masters and single events */
    public ListF<CalendarItemType> getMasterAndSingleEvents(Email email, Option<InstantInterval> intervalO) {
        return getEvents(findMasterAndSingleEventIds(email, intervalO));
    }

    public ListF<CalendarItemType> getModifiedOccurrences(CalendarItemType filledItem) {
        ListF<String> recurrencesExchangeIds = ExchangeEventDataConverter.getRecurrenceItemsId(filledItem);
        return recurrencesExchangeIds.isEmpty() ? Cf.list() : getEvents(recurrencesExchangeIds);
    }

    public ListF<CalendarItemType> getEventAndItsModifiedOccurrences(CalendarItemType filledItem) {
        return Cf.list(filledItem).plus(getModifiedOccurrences(filledItem));
    }

    /**
     * @return filled occurrence or exception, by master event id and index (ignoring rdates, exdates, exceptions).
     * Note: exchange (unlike calendar) does not allow to change the initial order of recurring events instances.
     */
    public Option<CalendarItemType> getOccurrenceByIndex(String recurringMasterExchangeId, int index) {
        Validate.notEmpty(recurringMasterExchangeId);
        final OccurrenceItemIdType itemId = EwsUtils.createOccurrenceItemIdType(recurringMasterExchangeId, index);
        return getEvents(Cf.list(itemId), false).singleO();
    }

    public ListF<CalendarItemType> getOccurrencesByTimeInterval(
            Email email, InstantInterval instanceInterval, final String externalId)
    {
        return ewsProxy.findInstanceEvents(email, instanceInterval, false).
                filter(calendarItem -> externalId.equals(calendarItem.getUID()));
    }

    public Option<ItemIdType> getOccurrenceIdByTimeInterval(
            Email email, InstantInterval interval, String externalId)
    {
        ListF<ItemIdType> occurrences = getOccurrenceIdsByTimeInterval(email, interval, externalId);

        if (occurrences.size() > 1) {
            throw new IllegalArgumentException(
                    "More than one occurrence of " + externalId + " found within " + interval + " for " + email);
        }
        return occurrences.singleO();
    }

    public ListF<ItemIdType> getOccurrenceIdsByTimeInterval(
            Email email, InstantInterval instantInterval, final String externalId)
    {
        ListF<CalendarItemType> items = ewsProxy.findInstanceEvents(email, instantInterval,
                Cf.list(UnindexedFieldURIType.CALENDAR_UID,
                        UnindexedFieldURIType.CALENDAR_START,
                        UnindexedFieldURIType.CALENDAR_END),
                Cf.list());

        return items.filterMap(c -> Option.of(c.getItemId()).filter(id -> externalId.equals(c.getUID())
                && (EwsUtils.xmlGregorianCalendarInstantToInstant(c.getEnd()).isAfter(instantInterval.getStart())
                || EwsUtils.xmlGregorianCalendarInstantToInstant(c.getStart()).isEqual(instantInterval.getStart()))));
    }

    public Tuple2List<String, Option<String>> findRecurringMasterIdsByInstanceIds(ListF<String> instanceExchangeIds) {
        return instanceExchangeIds.zipWith(id -> getEvents(Cf.list(EwsUtils.createRecurringMasterItemIdType(id)), true)
                .singleO().map(EwsUtils.calendarItemExchangeIdF()));
    }

    public ListF<String> findInstanceEventIdsByExternalId(
            Email email, InstantInterval interval, String externalId)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list(UnindexedFieldURIType.CALENDAR_UID);
        return ewsProxy.findInstanceEvents(email, interval, fields, Cf.<String>list())
                .filter(EwsUtils.calendarItemUidF().andThenEquals(externalId))
                .map(EwsUtils.calendarItemExchangeIdF());
    }

    public ListF<CalendarItemType> findInstanceEvents(
            Email email, InstantInterval interval, ListF<UnindexedFieldURIType> fields, ListF<String> extendedFields)
    {
        return ewsProxy.findInstanceEvents(email, interval, fields, extendedFields);
    }

    public ListF<String> findInstanceEventIds(Email email, InstantInterval interval) {
        return ewsProxy.findInstanceEvents(email, interval, true)
            .map(EwsUtils.calendarItemExchangeIdF());
    }

    public ListF<String> findMasterAndSingleEventIdsByExternalId(
            Email email, InstantInterval interval, String externalId)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list(UnindexedFieldURIType.CALENDAR_UID);
        return ewsProxy.findMasterAndSingleEvents(email, Option.of(interval), fields, Cf.<String>list())
                .filter(EwsUtils.calendarItemUidF().andThenEquals(externalId))
                .map(EwsUtils.calendarItemExchangeIdF());
    }

    /** like {@link #getMasterAndSingleEvents(Email, Option)}, but provides less filled fields and is much quicker */
    public ListF<CalendarItemType> findMasterAndSingleEvents(Email email, Option<InstantInterval> intervalO) {
        return ewsProxy.findMasterAndSingleEvents(email, intervalO, false);
    }

    public ListF<String> findMasterAndSingleEventIds(Email email, Option<InstantInterval> intervalO) {
        return ewsProxy.findMasterAndSingleEvents(email, intervalO, true)
                .map(EwsUtils.calendarItemExchangeIdF());
    }

    public Option<String> findMasterAndSingleOrInstanceEventId(
            Email email, InstantInterval interval, String externalId, boolean isRecurrence) {

        // recurrence can be single event for some participants
        Option<String> id = findMasterAndSingleEventIdsByExternalId(email, interval, externalId).singleO();

        if (!id.isPresent() && isRecurrence) {
            id = findInstanceEventIdsByExternalId(email, interval, externalId).singleO();
        }

        return id;
    }

    public Option<String> createEventSafe(Email email, ItemType item, EwsActionLogData actionInfo) {
        try {
            return Option.of(createEvent(email, item, actionInfo));
        } catch (Exception e) {
            logger.warn(e, e);
            return Option.empty();
        }
    }

    private String getFirstItemId(ResponseMessageType response) {
        if (response.getResponseClass().equals(ResponseClassType.SUCCESS)) {
            ItemInfoResponseMessageType infoResponse = (ItemInfoResponseMessageType) response;
            final ListF<ItemType> calendarItemInfos = Cf.x(infoResponse.getItems().getItemOrMessageOrCalendarItem());
            return calendarItemInfos.get(0).getItemId().getId();
        } else {
            throw new EwsErrorResponseException(response);
        }
    }

    public String createEvent(Email email, ItemType item, EwsActionLogData actionInfo) {
        return createEvent(email, item, MessageDispositionType.SAVE_ONLY,
                CalendarItemCreateOrDeleteOperationType.SEND_TO_NONE, actionInfo);
    }

    public String createEvent(
            Email email, ItemType item, MessageDispositionType disposition,
            CalendarItemCreateOrDeleteOperationType operationType, EwsActionLogData actionInfo)
    {
        Try<ResponseMessageType> result = Try.tryCatchThrowable(() -> ewsProxy.createItems(
                Option.of(email), Cf.list(item), disposition, operationType).get2().single());

        eventsLogger.log(new EwsCallLogEventJson(
                EwsCallOperation.CREATE_EVENT, actionInfo.logEventId, Optional.empty(),
                Optional.of(disposition), Either.left(operationType), getError(result).toOptional()), actionInfo.actionInfo);

        return getFirstItemId(getOrThrow(result));
    }

    public Option<String> updateItem(
            EwsModifyingItemId ewsUpdatingItemId,
            ListF<? extends ItemChangeDescriptionType> changeDescriptions, EwsActionLogData actionInfo)
    {
        return updateItem(ewsUpdatingItemId, changeDescriptions, CalendarItemUpdateOperationType.SEND_TO_NONE, actionInfo);
    }

    public Option<String> updateItem(
            EwsModifyingItemId ewsUpdatingItemId, ListF<? extends ItemChangeDescriptionType> changeDescriptions,
            CalendarItemUpdateOperationType operationType, EwsActionLogData actionInfo)
    {
        return updateItem(ewsUpdatingItemId, changeDescriptions, operationType, Option.empty(), actionInfo);
    }

    private Option<String> updateItem(
            EwsModifyingItemId ewsUpdatingItemId, ListF<? extends ItemChangeDescriptionType> changeDescriptions,
            CalendarItemUpdateOperationType operationType,
            Option<Tuple2<String, Integer>> prevAttempted, EwsActionLogData actionInfo)
    {
        if (changeDescriptions.isEmpty()) {
            logger.info("No exchange changes for event: " + ewsUpdatingItemId);
            return Option.empty();
        }
        final Option<ItemIdType> itemIdO;
        if (ewsUpdatingItemId.isOccurrence()) {
            itemIdO = getOccurrenceIdByTimeInterval(
                    ewsUpdatingItemId.getEmail(),
                    ewsUpdatingItemId.getInstanceInterval(),
                    ewsUpdatingItemId.getExternalId());
        } else {
            itemIdO = getEvents(Cf.list(EwsUtils.createItemId(ewsUpdatingItemId.getExchangeId())), true)
                    .singleO().map(EwsUtils.calendarItemItemIdF());
        }
        Option<Try<ResponseMessageType>> result = Option.empty();

        if (itemIdO.isPresent()) {
            result = Option.of(Try.tryCatchThrowable(() ->
                    ewsProxy.updateItem(itemIdO.get(), changeDescriptions, operationType)));
        } else {
            logger.info("Event wasn't found: " + ewsUpdatingItemId);
        }
        eventsLogger.log(new EwsCallLogEventJson(
                EwsCallOperation.UPDATE_ITEM, actionInfo.forItem(ewsUpdatingItemId).logEventId,
                Optional.empty(), Optional.empty(), Either.right(operationType),
                result.map(EwsProxyWrapper::getError).map(Option::toOptional).getOrElse(Optional.of("Not found"))), actionInfo.actionInfo);

        if (result.isPresent()) {
            ResponseMessageType response = getOrThrow(result.get());

            if (EwsErrorCodes.IRRESOLVABLE_CONFLICT.equals(response.getResponseCode())
                    && !prevAttempted.isMatch(t -> t.get1().equals(itemIdO.get().getChangeKey()))
                    && !prevAttempted.isMatch(t -> t.get2() > 2))
            {
                Tuple2<String, Integer> attempt = Tuple2.tuple(
                        itemIdO.get().getChangeKey(), prevAttempted.map(Tuple2::get2).getOrElse(0) + 1);

                return updateItem(ewsUpdatingItemId, changeDescriptions, operationType, Option.of(attempt), actionInfo);
            }
        }
        return result.map(t -> getFirstItemId(getOrThrow(t)));
    }

    private static DeclineItemType toDeclineMeetingType(ItemType item, Option<String> replyBodyHtml) {
        DeclineItemType declineItem = new DeclineItemType();
        declineItem.setReferenceItemId(EwsUtils.findItemId(item));

        if (replyBodyHtml.isPresent()) {
            BodyType body = new BodyType();
            body.setBodyType(BodyTypeType.HTML);
            body.setValue(replyBodyHtml.get());

            declineItem.setBody(body);
        }
        return declineItem;
    }

    private static Function<ItemIdType, CancelCalendarItemType> toCancelMeetingF() {
        return itemId -> {
            CancelCalendarItemType cancelItem = new CancelCalendarItemType();
            cancelItem.setReferenceItemId(itemId);
            return cancelItem;
        };
    }

    public void cancelMeetingsSafe(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        cancelMeetingsSafe(exchangeIds, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE, actionInfo);
    }

    public void cancelMeetingsSafe(
            ListF<ExchangeIdLogData> exchangeIds, CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        cancelMeetingsInner(exchangeIds, toCancelMeetingF(), operationType, actionInfo);
    }

    // ssytnik: workaround bug: any update of existing item changes 'changeKey' and further update of other occurrences fails.
    // thus, we create items with different change keys, in the first stage (bulk), and the others - one by one in the 2nd one.
    private void cancelMeetingsInner(
            ListF<ExchangeIdLogData> existingExchangeIds, Function<ItemIdType, ? extends ItemType> eventToItemConverter,
            CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        final ListF<CalendarItemType> events = getEvents(existingExchangeIds.map(id -> EwsUtils.createItemId(id.exchangeId)), true);

        final SetF<String> changeKeys = Cf.hashSet();
        Tuple2<ListF<CalendarItemType>, ListF<CalendarItemType>> stages =
                events.partition(a -> changeKeys.add(a.getItemId().getChangeKey()));

        Function<CalendarItemType, ItemType> convertF = EwsUtils.calendarItemItemIdF().andThen(eventToItemConverter);

        // stage 1
        createAndLogItems(stages.get1().map(convertF), Option.empty(), Option.of(operationType),
                Function1B.falseF(), existingExchangeIds, actionInfo);
        // stage 2 - recurring instances with repeating (found at stage 1) change keys
        for (CalendarItemType item : stages.get2()) {
            final ItemIdType newItemId = EwsUtils.createItemId(item.getItemId().getId());
            CalendarItemType updatedItem = getEvents(Cf.list(newItemId), true).single();

            createAndLogItems(Cf.list(updatedItem).map(convertF), Option.empty(), Option.of(operationType),
                    Function1B.falseF(), existingExchangeIds, actionInfo);
        }
    }

    public void cancelOrDeclineMeetingOccurrencesSafe(
            Email email, String externalId, ListF<Instant> starts, EwsActionLogData actionInfo)
    {
        cancelOrDeclineMeetingOccurrencesSafe(
                email, externalId, starts, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE, actionInfo);
    }

    public void cancelOrDeclineMeetingOccurrencesSafe(
            Email email, String externalId, ListF<Instant> starts,
            MessageDispositionType dispositionType, CalendarItemCreateOrDeleteOperationType operationType, EwsActionLogData actionInfo)
    {
        // batch cancellation occurrences of one event fails
        for (Instant start : starts) {
            OccurrenceId occurrenceId = new OccurrenceId(externalId, new InstantInterval(start, start), start);
            cancelOrDeclineMeetingOccurrenceSafe(
                    EwsModifyingItemId.fromEmailAndOccurrenceId(email, occurrenceId),
                    dispositionType, operationType, actionInfo);
        }
    }

    public void cancelOrDeclineMeetingOccurrenceSafe(EwsModifyingItemId ewsUpdatingItemId, EwsActionLogData actionInfo) {
        cancelOrDeclineMeetingOccurrenceSafe(
                ewsUpdatingItemId, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE, actionInfo);
    }

    public void cancelOrDeclineMeetingOccurrenceSafe(
            EwsModifyingItemId ewsUpdatingItemId, MessageDispositionType dispositionType,
            CalendarItemCreateOrDeleteOperationType operationType, EwsActionLogData actionInfo)
    {
        Validate.isTrue(ewsUpdatingItemId.isOccurrence());
        Option<ItemIdType> itemIdO = getOccurrenceIdByTimeInterval(
                ewsUpdatingItemId.getEmail(),
                ewsUpdatingItemId.getInstanceInterval(),
                ewsUpdatingItemId.getExternalId());

        if (itemIdO.isPresent()) {
            cancelOrDeclineItemsSafe(Cf.list(itemIdO.get()), dispositionType, operationType,
                    Cf.list(actionInfo.forItem(ewsUpdatingItemId).consExchangeId(itemIdO.get().getId())), actionInfo.actionInfo);
        } else {
            logger.info("Event wasn't found: " + ewsUpdatingItemId);

            eventsLogger.log(new EwsCallLogEventJson(
                    EwsCallOperation.CANCEL_OR_DECLINE_OCCURRENCE, actionInfo.forItem(ewsUpdatingItemId).logEventId,
                    Optional.of(ewsUpdatingItemId.getInstanceInterval().getStart()),
                    Optional.of(dispositionType), Either.left(operationType), Optional.of("Not found")),
                    actionInfo.actionInfo);
        }
    }

    public void cancelOrDeclineExchangeCreatedOrDeleteMeetings(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        cancelOrDeclineExchangeCreatedOrDeleteMeetings(exchangeIds, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE, actionInfo);
    }

    public void cancelOrDeclineExchangeCreatedOrDeleteMeetings(
            ListF<ExchangeIdLogData> exchangeIds, CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list();
        ListF<String> extendedFields = Cf.list(EwsUtils.EXTENDED_PROPERTY_SOURCE);
        ListF<CalendarItemType> events = getEvents(exchangeIds.map(id -> id.exchangeId), fields, extendedFields);

        Tuple2<ListF<CalendarItemType>, ListF<CalendarItemType>> p = events.partition(EwsUtils
                .convertExtendedPropertiesF().andThen(ExtendedCalendarItemProperties.wasCreatedFromYaTeamCalendarF()));

        logger.debug("Found events to cancel: " + p.get2().map(getItemIdAndChangeKeyF()));
        cancelOrDeclineItemsSafe(p.get2().map(EwsUtils.calendarItemItemIdF()), DEFAULT_DISPOSITION_TYPE, operationType, exchangeIds, actionInfo);

        logger.debug("Found events to delete: " + p.get1().map(getItemIdAndChangeKeyF()));
        deleteAndLogItems(exchangeIds, Option.empty(), actionInfo);
    }

    public void cancelOrDeclineMeetingSafe(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        cancelOrDeclineMeetingSafe(exchangeIds, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE, actionInfo);
    }

    public void cancelOrDeclineMeetingSafe(
            ListF<ExchangeIdLogData> exchangeIds, MessageDispositionType dispositionType,
            CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        logger.debug("Trying to cancel meetings: " + exchangeIds);

        ListF<CalendarItemType> events = getEvents(exchangeIds.map(id -> EwsUtils.createItemId(id.exchangeId)), true);

        logger.debug("Found events to cancel : " + events.map(getItemIdAndChangeKeyF()));

        cancelOrDeclineItemsSafe(events.map(EwsUtils.calendarItemItemIdF()), dispositionType, operationType, exchangeIds, actionInfo);
    }

    public void cancelOrDeclineMeetingWithReply(String exchangeId, String replyBodyHtml, EwsActionLogData actionInfo) {
        Option<CalendarItemType> event = getEvents(Option.of(exchangeId).map(EwsUtils.createItemIdF()), true).singleO();
        if (!event.isPresent()) return;

        CancelCalendarItemType cancelItem = event.map(EwsUtils.calendarItemItemIdF().andThen(toCancelMeetingF())).get();
        ResponseMessageType response = createAndLogItems(
                Cf.list(cancelItem), Option.empty(), Option.empty(), EwsErrorCodes.CALENDAR_IS_NOT_ORGANIZER::equals,
                Cf.list(actionInfo.consExchangeId(exchangeId)), actionInfo.actionInfo).single().get2();

        if (ResponseClassType.ERROR != response.getResponseClass()) return;

        if (EwsErrorCodes.CALENDAR_IS_NOT_ORGANIZER.equals(response.getResponseCode())) {
            DeclineItemType declineItem = toDeclineMeetingType(cancelItem, Option.of(replyBodyHtml));

            MessageDispositionType disposition = EnvironmentType.TESTS != environmentType
                    ? MessageDispositionType.SEND_AND_SAVE_COPY
                    : MessageDispositionType.SAVE_ONLY;

            response = createAndLogItems(
                    Cf.list(declineItem), Option.of(disposition), Option.empty(), Function1B.falseF(),
                    Cf.list(actionInfo.consExchangeId(exchangeId)), actionInfo.actionInfo).single().get2();
        }
        if (ResponseClassType.ERROR == response.getResponseClass()) {
            throw new EwsErrorResponseException("Failed to cancel or decline meeting", response);
        }
    }

    public void deleteEvents(ListF<ExchangeIdLogData> exchangeIds, ActionInfo actionInfo) {
        deleteEvents(exchangeIds, CalendarItemCreateOrDeleteOperationType.SEND_TO_NONE, actionInfo);
    }

    public void deleteEvents(
            ListF<ExchangeIdLogData> exchangeIds, CalendarItemCreateOrDeleteOperationType operationType, ActionInfo actionInfo)
    {
        deleteAndLogItems(exchangeIds, Option.of(operationType), actionInfo);
    }

    private void cancelOrDeclineItemsSafe(
            ListF<ItemIdType> itemIds, MessageDispositionType dispositionType,
            CalendarItemCreateOrDeleteOperationType operationType, ListF<ExchangeIdLogData> logDatas, ActionInfo actionInfo)
    {
        // XXX ssytnik: add pagination, like in getEvents, reuse maxPageSize,
        // then also simplify cancelMasterAndSingleMeetings and cancelMeetings
        Tuple2List<ItemType, ResponseMessageType> cancellingResponses = createAndLogItems(
                itemIds.map(toCancelMeetingF()), Option.of(dispositionType), Option.of(operationType),
                EwsErrorCodes.CALENDAR_IS_NOT_ORGANIZER::equals, logDatas, actionInfo);

        logger.info("Cancel or decline item safe");
        ListF<DeclineItemType> itemIdsToDecline = Cf.arrayList();
        for (Tuple2<ItemType, ResponseMessageType> cancellingResponse : cancellingResponses) {
            ResponseClassType result = cancellingResponse._2.getResponseClass();
            if (result != ResponseClassType.SUCCESS) {
                if (cancellingResponse._2.getResponseCode().equals(EwsErrorCodes.CALENDAR_IS_NOT_ORGANIZER)) {
                    itemIdsToDecline.add(toDeclineMeetingType(cancellingResponse._1, Option.<String>empty()));
                } else {
                    logger.error("Got " + result.name() + " when cancelling " + EwsUtils.findItemId(cancellingResponse._1).getId() +
                            ": " + cancellingResponse._2.getMessageText());
                }
            }
        }

        logger.debug("Trying to decline meetings: " + itemIdsToDecline.map(getDeclineItemTypeItemIdF()));
        Tuple2List<ItemType, ResponseMessageType> decliningResponses = createAndLogItems(
                itemIdsToDecline, Option.of(dispositionType), Option.of(operationType),
                Function1B.falseF(), logDatas, actionInfo);

        for (Tuple2<ItemType, ResponseMessageType> decliningResponse : decliningResponses) {
            ResponseClassType result = decliningResponse._2.getResponseClass();
            if (result != ResponseClassType.SUCCESS) {
                logger.error("Got " + result.name() + " when declining " + EwsUtils.findItemId(decliningResponse._1).getId() +
                        ": " + decliningResponse._2.getMessageText());
            }
        }

    }

    private Function<CalendarItemType, Tuple2<String, String>> getItemIdAndChangeKeyF() {
        return calendarItemType -> Tuple2.tuple(calendarItemType.getItemId().getId(), calendarItemType.getItemId().getChangeKey());
    }

    private Function<DeclineItemType, String> getDeclineItemTypeItemIdF() {
        return declineItemType -> declineItemType.getReferenceItemId().getId();
    }

    public void cancelMasterAndSingleMeetings(Email email, Instant startTs, Instant endTs) {
        cancelMasterAndSingleMeetings(
                email, startTs, endTs, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE);
    }

    public void cancelMasterAndSingleMeetings(Email email, Instant startTs, Instant endTs,
            MessageDispositionType dispositionType, CalendarItemCreateOrDeleteOperationType operationType)
    {
        InstantInterval interval = new InstantInterval(startTs, endTs);
        cancelMasterAndSingleMeetings(email, interval, dispositionType, operationType);
    }

    public void cancelMeetings(Email email, Instant startTs, Instant endTs) {
        cancelMeetings(email, startTs, endTs, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE);
    }

    public void cancelMeetings(Email email, Instant startTs, Instant endTs,
            MessageDispositionType dispositionType, CalendarItemCreateOrDeleteOperationType operationType)
    {
        InstantInterval interval = new InstantInterval(startTs, endTs);
        ListF<String> exchangeIds = findInstanceEventIds(email, interval);
        for (String exchangeId : exchangeIds) {
            cancelOrDeclineMeetingSafe(
                    Cf.list(ExchangeIdLogData.test(exchangeId)),
                    dispositionType,
                    operationType,
                    ActionInfo.exchangeTest());
        }
    }

    public void cancelMasterAndSingleMeetings(Email email, InstantInterval interval) {
        cancelMasterAndSingleMeetings(
                email, interval, DEFAULT_DISPOSITION_TYPE, DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE);
    }

    public void cancelMasterAndSingleMeetings(Email email, InstantInterval interval,
            MessageDispositionType dispositionType, CalendarItemCreateOrDeleteOperationType operationType)
    {
        cancelMasterAndSingleMeetings(email, Option.of(interval), dispositionType, operationType);
    }

    private void cancelMasterAndSingleMeetings(Email email, Option<InstantInterval> interval,
            MessageDispositionType dispositionType, CalendarItemCreateOrDeleteOperationType operationType)
    {
        ListF<String> exchangeIds = findMasterAndSingleEventIds(email, interval);
        // cancel by one, because of possible huge amount of events
        for (String exchangeId : exchangeIds) {
            cancelOrDeclineMeetingSafe(
                    Cf.list(ExchangeIdLogData.test(exchangeId)),
                    dispositionType,
                    operationType,
                    ActionInfo.exchangeTest());
        }
    }

    public boolean setUserDecision(
            String exchangeId, Decision decision, Option<String> message, MessageDispositionType dispositionType,
            CalendarItemCreateOrDeleteOperationType operationType, EwsActionLogData actionInfo)
    {

        ListF<WellKnownResponseObjectType> decisionItems;

        Function2<WellKnownResponseObjectType, String, WellKnownResponseObjectType> addReferenceIdAndBody =
                (decisionItem, id) -> {
                    decisionItem.setReferenceItemId(new ItemIdType());
                    decisionItem.getReferenceItemId().setId(id);
                    message.forEach(m -> {
                        BodyType body = new BodyType();
                        body.setBodyType(BodyTypeType.HTML);
                        body.setValue(m);
                        decisionItem.setBody(body);
                    });
                    return decisionItem;
                };

        switch (decision) {
            case UNDECIDED:
                return false;
            case YES:
                decisionItems = Cf.list(addReferenceIdAndBody.apply(new AcceptItemType(), exchangeId));
                break;
            case NO:
                // Exchange does not apply decline to occurrences, so they have to be declined manually
                // First all occurrences, then the master event, that's why a .reverse() is used,
                ListF<String> exchangeIds = getEvent(exchangeId).map(this::getEventAndItsModifiedOccurrences)
                        .getOrElse(Cf.list()).reverse().map(EwsUtils.calendarItemExchangeIdF());
                decisionItems = exchangeIds.map(id -> addReferenceIdAndBody.apply(new DeclineItemType(), id));
                logger.info("found exchange ids " + exchangeIds.toString());
                break;
            case MAYBE:
                decisionItems = Cf.list(addReferenceIdAndBody.apply(new TentativelyAcceptItemType(), exchangeId));
                break;
            default:
                throw new IllegalStateException("Unsupported decision type " + decision);
        }

        Tuple2List<ItemType, ResponseMessageType> result = createAndLogItems(
                decisionItems, Option.of(dispositionType), Option.of(operationType), Function1B.falseF(),
                Cf.list(actionInfo.consExchangeId(exchangeId)), actionInfo.actionInfo);

        return result.isNotEmpty() && result.get2().forAll(r -> r.getResponseClass() == ResponseClassType.SUCCESS);
    }

    private Tuple2List<ItemType, ResponseMessageType> createAndLogItems(
            ListF<? extends ItemType> items,
            Option<MessageDispositionType> dispositionO, Option<CalendarItemCreateOrDeleteOperationType> invitationsO,
            Function1B<String> ignoreErrorForLogging, ListF<ExchangeIdLogData> logDatas, ActionInfo actionInfo)
    {
        MapF<String, ExchangeIdLogData> dataByExchangeId = logDatas.toMapMappingToKey(e -> e.exchangeId);

        MessageDispositionType disposition = dispositionO.getOrElse(DEFAULT_DISPOSITION_TYPE);
        CalendarItemCreateOrDeleteOperationType invitations = invitationsO.getOrElse(DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE);

        Try<Tuple2List<ItemType, ResponseMessageType>> result = Try.tryCatchThrowable(
                () -> ewsProxy.createItems(Option.empty(), items, disposition, invitations));

        Tuple2List<ItemType, Try<ResponseMessageType>> responses = result.isFailure()
                ? items.toTuple2List(i -> i, i -> Try.failure(result.getThrowable()))
                : result.get().map2(Try::success);

        responses.forEach(res -> {
            String exchangeId = res.get1() instanceof ResponseObjectCoreType
                    ? ((ResponseObjectCoreType) res.get1()).getReferenceItemId().getId()
                    : res.get1().getItemId().getId();

            EwsCallOperation operation = res.get1() instanceof CancelCalendarItemType ? EwsCallOperation.CANCEL_MEETING
                    : res.get1() instanceof DeclineItemType ? EwsCallOperation.DECLINE_MEETING
                    : res.get1() instanceof AcceptItemType ? EwsCallOperation.ACCEPT_MEETING
                    : res.get1() instanceof WellKnownResponseObjectType ? EwsCallOperation.REPLY_MEETING
                    : EwsCallOperation.CREATE_EVENT; // XXX

            dataByExchangeId.getO(exchangeId).forEach(logData -> {
                if (!res.get2().isSuccess() || !ignoreErrorForLogging.apply(res.get2().get().getResponseCode())) {
                    eventsLogger.log(new EwsCallLogEventJson(operation, logData.logEventId, Optional.empty(),
                            Optional.of(disposition), Either.left(invitations), getError(res.get2()).toOptional()), actionInfo);
                }
            });
        });
        return getOrThrow(result);
    }

    private void deleteAndLogItems(
            ListF<ExchangeIdLogData> exchangeIds,
            Option<CalendarItemCreateOrDeleteOperationType> invitationsO, ActionInfo actionInfo)
    {
        MapF<String, ExchangeIdLogData> dataByExchangeId = exchangeIds.toMapMappingToKey(e -> e.exchangeId);

        CalendarItemCreateOrDeleteOperationType invitations = invitationsO.getOrElse(DEFAULT_CREATE_OR_DELETE_OPERATION_TYPE);

        Try<Tuple2List<String, ResponseMessageType>> result = Try.tryCatchThrowable(
                () -> ewsProxy.deleteEvents(exchangeIds.map(e -> e.exchangeId), invitations));

        Tuple2List<String, Try<ResponseMessageType>> responses = result.isFailure()
                ? exchangeIds.toTuple2List(i -> i.exchangeId, i -> Try.failure(result.getThrowable()))
                : result.get().map2(Try::success);

        responses.forEach(res -> dataByExchangeId.getO(res.get1()).forEach(logData -> {
            if (!res.get2().isSuccess() || !EwsErrorCodes.ITEM_NOT_FOUND.equals(res.get2().get().getResponseCode())) {
                eventsLogger.log(new EwsCallLogEventJson(
                        EwsCallOperation.DELETE_EVENT, logData.logEventId,
                        Optional.empty(), Optional.empty(), Either.left(invitations), getError(res.get2()).toOptional()), actionInfo);
            }
        }));

        getOrThrow(result).forEach(resp -> {
            if (resp.get2().getResponseClass() == ResponseClassType.ERROR) {
                if (!EwsErrorCodes.ITEM_NOT_FOUND.equals(resp.get2().getResponseCode())) {
                    throw new EwsErrorResponseException("Failed to delete events", resp.get2());
                }
            }
        });
    }

    private static Option<String> getError(Try<? extends ResponseMessageType> response) {
        if (response.isFailure()) {
            return Option.of(ExceptionUtils.getAllMessages(response.getThrowable()));

        } else if (response.get().getResponseClass() != ResponseClassType.SUCCESS) {
            return Option.of(EwsErrorResponseException.message(response.get()));
        }
        return Option.empty();
    }

    private static <T> T getOrThrow(Try<T> result) {
        if (result.isFailure()) {
            throw ExceptionUtils.throwException(result.getThrowable());

        } else {
            return result.get();
        }
    }
}
