package ru.yandex.calendar.frontend.ews.proxy;

import java.net.URL;

import javax.net.ssl.TrustManager;
import javax.xml.ws.BindingProvider;

import com.microsoft.schemas.exchange.services._2006.messages.ExchangeServicePortType;
import com.microsoft.schemas.exchange.services._2006.messages.ExchangeServices;
import lombok.val;
import org.apache.cxf.configuration.jsse.TLSClientParameters;
import org.apache.cxf.configuration.security.AuthorizationPolicy;
import org.apache.cxf.endpoint.Client;
import org.apache.cxf.frontend.ClientProxy;
import org.apache.cxf.transport.http.HTTPConduit;
import org.apache.cxf.transports.http.configuration.ConnectionType;
import org.apache.cxf.transports.http.configuration.HTTPClientPolicy;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.unistat.UnistatConfiguration;
import ru.yandex.calendar.util.ssl.SslUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.chrono.ChronoProxy;

@Configuration
@Import({ UnistatConfiguration.class })
public class EwsProxyWrapperContextConfiguration {
    private static final String WSDL_2010_LOCATION_PATH = "/Services2010.wsdl";

    @Value("${ews.domain}")
    private String domain;
    @Value("${ews.mailbot2013.login}")
    private String login2013;
    @Value("${ews.mailbot2013.password}")
    private String password2013;
    @Value("${ews.endpoint2013.url}")
    private String ewsEndpointUrl2013;
    @Value("${ews.request.attempts}")
    private int ewsRequestAttempts;
    @Value("${ews.connect.timeout}")
    private int ewsConnectTimeout;
    @Value("${ews.receive.timeout}")
    private int ewsReceiveTimeout;
    @Value("${ews.jetty.http.port}")
    private int ewsJettyHttpPort;

    @Autowired
    private EnvironmentType environmentType;

    @Bean
    public EwsProxy ewsProxy() {
        ExchangeServicePortType proxy2013 = createExchangeServicePort(
                ewsEndpointUrl2013, WSDL_2010_LOCATION_PATH, login2013, password2013, domain.split("\\.", 2)[0]);

        val userMailbox2013 = new Email(login2013 + "@" + domain);

        return ChronoProxy.newChronoProxy(new EwsProxyImpl(
                proxy2013, userMailbox2013,
                environmentType, ewsJettyHttpPort));
    }

    private ExchangeServicePortType createExchangeServicePort(
            String ewsEndpointUrl, String wsdlLocationPath, String login, String password, String domain)
    {
        // create client proxy
        URL wsdlLocation = EwsProxyWrapperContextConfiguration.class.getResource(wsdlLocationPath);
        ExchangeServices services = new ExchangeServices(wsdlLocation); // wsdlLocation);
        ExchangeServicePortType proxy = services.getExchangeServicePort();
        // create client
        Client client = ClientProxy.getClient(proxy);
        HTTPConduit conduit = (HTTPConduit) client.getConduit();
        // http basic authentication
        AuthorizationPolicy authorization = new AuthorizationPolicy();
        authorization.setUserName(domain + "\\" + login);
        authorization.setPassword(password);
        authorization.setAuthorizationType("Basic");
        conduit.setAuthorization(authorization);
        // initialize all trusting ssl connection
        TLSClientParameters tlsParams = new TLSClientParameters();
        tlsParams.setTrustManagers( new TrustManager[]{ SslUtils.getAllTrustingTrustManager() } );
        tlsParams.setDisableCNCheck(true);
        conduit.setTlsClientParameters(tlsParams);
        // setup client policy for NTLM authentication
        HTTPClientPolicy httpClientPolicy = new HTTPClientPolicy();
        httpClientPolicy.setAllowChunking(false);
        httpClientPolicy.setConnection(ConnectionType.KEEP_ALIVE);
        httpClientPolicy.setConnectionTimeout(ewsConnectTimeout);
        httpClientPolicy.setReceiveTimeout(ewsReceiveTimeout);
        conduit.setClient(httpClientPolicy);
        // set end point
        ((BindingProvider) proxy).getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, ewsEndpointUrl);

        return ExchangeLoggingServiceProxy.of(proxy);
    }

    @Bean
    public EwsProxyWrapper ewsProxyWrapper(EventsLogger eventsLogger) {
        return new EwsProxyWrapper(
                EwsMonitoringProxy.newEwsMonitoringProxy(ewsMonitoring(), ewsProxy(), ewsRequestAttempts),
                environmentType,
                eventsLogger
        );
    }

    @Bean
    public EwsMonitoring ewsMonitoring() {
        return new EwsMonitoring();
    }

    @Bean
    public EwsProxyChecker ewsProxyChecker() {
        return new EwsProxyChecker(Duration.standardSeconds(30));
    }
}
