package ru.yandex.calendar.frontend.ews.proxy;

import java.io.StringWriter;
import java.util.Optional;

import javax.jws.WebParam;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Marshaller;
import javax.xml.namespace.QName;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.microsoft.schemas.exchange.services._2006.messages.BaseRequestType;
import com.microsoft.schemas.exchange.services._2006.messages.BaseResponseMessageType;
import com.sun.xml.bind.marshaller.NamespacePrefixMapper;
import lombok.SneakyThrows;
import lombok.val;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.reflection.ClassX;

public class ExchangeRequestLogger {
    public static final String NAME = "exchange-requests-json";
    private static final Logger logger = LoggerFactory.getLogger(NAME);

    private static final MapF<Class<?>, JAXBContext> jaxbContexts = Cf.concurrentHashMap();

    public static void log(ExchangeInvocation invocation, BaseRequestType request) {
        log(invocation, "request", invocation.getRequestParam(), request);
    }

    public static void log(ExchangeInvocation invocation, BaseResponseMessageType response) {
        log(invocation, "response", invocation.getResponseParam(), response);
    }

    public static void log(ExchangeInvocation invocation, Throwable error) {
        val requestLogData = new ExchangeRequestLogData();
        requestLogData.type = "error";
        requestLogData.error = Optional.of(ExceptionUtils.prettyPrint(error));
        log(invocation, requestLogData);
    }

    @SneakyThrows
    private static void log(ExchangeInvocation invocation, String type, WebParam param, Object value) {
        val clazz = ClassX.wrap(value.getClass()).uncheckedCast().getClazz();
        val element = new JAXBElement<>(new QName(param.targetNamespace(), param.name()), clazz, value);

        val context = jaxbContexts.computeIfAbsent(clazz, ExchangeRequestLogger::newJaxbContext);

        val marshaller = context.createMarshaller();

        marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
        marshaller.setProperty(Marshaller.JAXB_FRAGMENT, true);
        marshaller.setProperty(PrefixMapper.PROPERTY_NAME, new PrefixMapper());

        val writer = new StringWriter();
        marshaller.marshal(element, writer);

        val requestLogData = new ExchangeRequestLogData();
        requestLogData.type = type;
        requestLogData.body = Optional.of(writer.toString());
        log(invocation, requestLogData);
    }

    @SneakyThrows
    private static JAXBContext newJaxbContext(Class<?> clazz) {
        return JAXBContext.newInstance(clazz);
    }

    private static void log(ExchangeInvocation invocation, ExchangeRequestLogData requestLogData) {
        requestLogData.actionInfo = CalendarRequest.getCurrentO().map(CalendarRequestHandle::getActionInfo)
                .getOrElse(new ActionInfo(ActionSource.UNKNOWN, RequestIdStack.current().getOrElse(""), Instant.now()));
        requestLogData.num = invocation.getNum();
        requestLogData.method = invocation.getMethod();
        try {
            logger.info(EventsLogger.mapper.writeValueAsString(requestLogData));
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static class PrefixMapper extends NamespacePrefixMapper {
        private static final String PROPERTY_NAME = "com.sun.xml.bind.namespacePrefixMapper";

        public String getPreferredPrefix(String namespaceUri, String suggestion, boolean required) {
            switch (namespaceUri) {
                case "http://schemas.microsoft.com/exchange/services/2006/types": return "";
                case "http://schemas.microsoft.com/exchange/services/2006/messages": return "m";
                default: return suggestion;
            }
        }
    }
}
