package ru.yandex.calendar.frontend.ews.replace;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.ItemChangeDescriptionType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.exp.EventToCalendarItemConverter;
import ru.yandex.calendar.frontend.ews.exp.EwsModifyingItemId;
import ru.yandex.calendar.frontend.ews.proxy.EwsActionLogData;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.proxy.ExchangeIdLogData;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.MainEventInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeout.Handle;
import ru.yandex.misc.time.InstantInterval;

public class EwsResourceMeetingReplacer {
    private static final Logger logger = LoggerFactory.getLogger(EwsResourceMeetingReplacer.class);

    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private EventToCalendarItemConverter eventToCalendarItemConverter;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private EventDao eventDao;


    public void replaceUserOrganized(Email email) {
        replace(loadMainEventsInfo(findUserOrganizedMainEventIds(email)), ActionInfo.adminManager());
    }

    public ListF<ExchangeResourceMeeting> findUserOrganizedResourcesMeetingsCreatedByExchange(Email email) {
        return findResourcesMeetingsCreatedByExchangeByMainEventIds(findUserOrganizedMainEventIds(email));
    }

    public void replaceByMainEventId(long mainEventId) {
        replace(loadMainEventsInfo(Cf.list(mainEventId)).single(), ActionInfo.adminManager());
    }

    public ListF<ExchangeResourceMeeting> findResourcesMeetingsCreatedByExchangeByMainEventId(long mainEventId) {
        return findResourcesMeetingsCreatedByExchangeByMainEventIds(Cf.list(mainEventId));
    }

    public ListF<ExchangeResourceMeeting> findResourcesMeetingsCreatedByExchangeByMainEventIds(ListF<Long> ids) {
        return findResourcesMeetingsInExchange(loadMainEventsInfo(ids))
                .filter(ExchangeResourceMeeting.wasCreatedFromYaCalendarF().notF());
    }

    public ListF<ExchangeResourceMeeting> findResourcesMeetingsInExchangeByMainEventId(long mainEventId) {
        return findResourcesMeetingsInExchange(loadMainEventsInfo(Cf.list(mainEventId)));
    }

    public ListF<ExchangeResourceMeeting> findResourcesMeetingsInExchange(ListF<MainEventInfo> mainEvents) {
        return mainEvents
                .flatMap(MainEventInfo::getEventsWithRelations)
                .flatMap(findResourcesMeetingInExchangeF());
    }

    public void replace(ListF<MainEventInfo> mainEvents, ActionInfo actionInfo) {
        for (MainEventInfo mainEvent : mainEvents) {
            logger.info("Processing " + LogMarker.MAIN_EVENT_ID.format(mainEvent.getMainEventId()));
            replace(mainEvent, actionInfo);
        }
    }

    public void replace(MainEventInfo mainEvent, ActionInfo actionInfo) {
        ListF<Long> resourceIds = mainEvent.getEventsWithRelations()
                .flatMap(EventWithRelations::getResourceIds).stableUnique();

        Handle tltHandle = ThreadLocalTimeout.push(Duration.standardSeconds(60), "replacing meetings in exchange");
        try {
            lockTransactionManager.lockAndDoInTransaction(
                    () -> updateLock2.lockEvent(Option.of(mainEvent.getMainEvent().getExternalId())),
                    () -> doReplace(mainEvent, actionInfo.withNow(Instant.now())));

        } finally {
            tltHandle.popSafely();
        }
    }

    private ListF<Long> findUserOrganizedMainEventIds(Email email) {
        PassportUid uid = userManager.getUidByEmail(email).getOrThrow("user not found ", email);
        return eventDao.findUserOrganizedEvents(uid).map(Event.getMainEventIdF()).stableUnique();
    }

    private ListF<MainEventInfo> loadMainEventsInfo(ListF<Long> mainEventIds) {
        return eventInfoDbLoader.getMainEventInfos(
                Option.<PassportUid>empty(),
                mainEventDao.findMainEventsByIds(mainEventIds),
                ActionSource.UNKNOWN);
    }

    private void doReplace(final MainEventInfo mainEvent, final ActionInfo actionInfo) {
        ListF<ExchangeResourceMeeting> meetingsCreatedByExchange = findResourcesMeetingsInExchange(Cf.list(mainEvent))
                .filter(ExchangeResourceMeeting.wasCreatedFromYaCalendarF().notF());

        if (meetingsCreatedByExchange.isEmpty()) {
            logger.info("No meetings to replace were found");
            return;
        }
        final ReplaceResourcesInfo resourcesInfo = getResourcesToReplaceMeetings(meetingsCreatedByExchange, mainEvent);
        final MasterCreateData masterCreateData = convertMaster(mainEvent);
        final MapF<Long, RecurrenceCreateData> recurrenceCreateData = convertRecurrences(mainEvent);

        ListF<ResourceMeeting> createdMeetings = cancelStoredIfException(stored -> {
            ListF<ResourceMeeting> masterMeetings = createMasterMeetings(
                    resourcesInfo.getResourcesToReplaceMaster(), masterCreateData, actionInfo);

            stored.addAll(masterMeetings);

            for (long eventId : mainEvent.getRecurrenceEventInfos().map(EventInfo::getEventId)) {
                ListF<ResourceMeeting> recurrencesMeetings = createRecurrenceMeetings(
                        resourcesInfo.getRecurrenceReplaceResourcesInfo(eventId),
                        masterMeetings, recurrenceCreateData.getOrThrow(eventId), actionInfo);

                stored.addAll(recurrencesMeetings);
            }
            saveUpdateExchangeIds(stored, actionInfo);
        }, actionInfo);

        cancelOrDeclineMeetings(meetingsCreatedByExchange.filter(ExchangeResourceMeeting.isMasterOrSingleF()), actionInfo);
        reportStats(meetingsCreatedByExchange, loadMeetingsFromExchange(createdMeetings));
    }

    private void reportStats(ListF<ExchangeResourceMeeting> oldMeetings, ListF<ExchangeResourceMeeting> newMeetings) {
        logger.info("{} master and recurrence meetings were replaced on {} resources",
                newMeetings.size(), newMeetings.map(ResourceMeeting.getResourceIdF()).unique().size());

        Function<? super ExchangeResourceMeeting, Tuple2<Long, Long>> pairF = Tuple2.join(
                ExchangeResourceMeeting.getEventIdF(), ExchangeResourceMeeting.getResourceIdF());

        MapF<Tuple2<Long, Long>, Integer> oldConflictsCountByPair =
                oldMeetings.toMap(pairF, ExchangeResourceMeeting.getConflictingMeetingCountF());

        for (ExchangeResourceMeeting n : newMeetings.filter(ExchangeResourceMeeting.isMasterOrSingleF())) {
            int oldConflictsCount = oldConflictsCountByPair.getO(pairF.apply(n)).getOrElse(0);
            int newConflictsCount = n.getConflictingMeetingCount();

            if (oldConflictsCount != newConflictsCount) {
                Email email = ResourceRoutines.getResourceEmail(n.getResource());
                logger.warn("Resource {} had {} conflicts with event {} but now has {}",
                        email, oldConflictsCount, n.getEventId(), newConflictsCount);
            }
        }
    }

    private void saveUpdateExchangeIds(final ListF<ResourceMeeting> meetings, final ActionInfo actionInfo) {
        for (ResourceMeeting m : meetings) {
            eventResourceDao.saveUpdateExchangeId(m.getResource().getId(), m.getEventId(), m.getExchangeId(), actionInfo);
        }
    }

    private ListF<ResourceMeeting> createMasterMeetings(
            final ListF<ResourceParticipantInfo> resources, final MasterCreateData createData, ActionInfo actionInfo)
    {
        return cancelStoredIfException(stored -> {
            val logId = new EventIdLogDataJson(createData.getEventInfo());

            stored.addAll(createMasterOrSingleMeetings(resources, createData.getCreateData(), logId, actionInfo));
            cancelOccurrences(stored.map(ResourceMeeting.getResourceIdF()), stored, createData.getExdates(), logId, actionInfo);
        }, actionInfo);
    }

    private ListF<ResourceMeeting> createRecurrenceMeetings(
            final RecurrenceReplaceResourcesInfo replaceResourcesInfo,
            final ListF<ResourceMeeting> masterMeetings, final RecurrenceCreateData createData, ActionInfo actionInfo)
    {
        return cancelStoredIfException(stored -> {
            val logId = new EventIdLogDataJson(createData.getEventInfo());

            stored.addAll(modifyOccurrences(
                    replaceResourcesInfo.getResourcesToModifyOccurrence(), masterMeetings, createData, actionInfo));

            stored.addAll(createMasterOrSingleMeetings(
                    replaceResourcesInfo.getResourcesToCreateSingleMeeting(), createData.getCreateData(), logId, actionInfo));

            cancelOccurrences(
                    replaceResourcesInfo.getResourceIdsToCancelOccurrence(),
                    masterMeetings, Cf.list(createData.getRecurrenceId()), logId, actionInfo);
        }, actionInfo);
    }

    private ListF<ResourceMeeting> createMasterOrSingleMeetings(
            ListF<ResourceParticipantInfo> resources,
            CalendarItemType createData, EventIdLogDataJson eventId, ActionInfo actionInfo)
    {
        return cancelProcessedIfException(resources, r -> {
            EwsActionLogData logData = new EwsActionLogData(UidOrResourceId.resource(r), eventId, actionInfo);

            Email email = resourceRoutines.getExchangeEmail(r.getResource());

            return new ResourceMeeting(r, new ExchangeIdLogData(
                    ewsProxyWrapper.createEvent(email, createData, logData), UidOrResourceId.resource(r), eventId));
        }, actionInfo);
    }

    private ListF<ResourceMeeting> modifyOccurrences(
            ListF<ResourceParticipantInfo> resources, ListF<ResourceMeeting> masters,
            RecurrenceCreateData createData, ActionInfo actionInfo)
    {
        final MapF<Long, ResourceMeeting> masterByResId = masters.toMapMappingToKey(ResourceMeeting.getResourceIdF());

        return cancelProcessedIfException(resources, r -> {
            val occurrenceExchangeId = getOccurrenceId(masterByResId.getOrThrow(r.getResourceId()), createData.getRecurrenceId());
            val id = EwsModifyingItemId.fromExchangeId(occurrenceExchangeId);

            val logId = new EventIdLogDataJson(createData.getEventInfo());
            val logData = new EwsActionLogData(UidOrResourceId.resource(r), logId, actionInfo);

            return new ResourceMeeting(r, new ExchangeIdLogData(
                    ewsProxyWrapper.updateItem(id, createData.getUpdateData(), logData).get(),
                    UidOrResourceId.resource(r), logId));
        }, actionInfo);
    }

    private void cancelOccurrences(
            ListF<Long> resourceIds, ListF<ResourceMeeting> masters,
            ListF<Instant> occurrenceStarts, EventIdLogDataJson eventIdLogData, ActionInfo actionInfo)
    {
        final MapF<Long, ResourceMeeting> masterByResId = masters.toMapMappingToKey(ResourceMeeting.getResourceIdF());

        for (long resourceId : resourceIds) {
            for (Instant start : occurrenceStarts) {
                String occurrenceExchangeId = getOccurrenceId(masterByResId.getOrThrow(resourceId), start);
                ewsProxyWrapper.cancelMeetingsSafe(Cf.list(new ExchangeIdLogData(
                        occurrenceExchangeId, UidOrResourceId.resource(resourceId), eventIdLogData)), actionInfo);
            }
        }
    }

    private String getOccurrenceId(ResourceMeeting master, Instant occurrenceStart) {
        Email email = resourceRoutines.getExchangeEmail(master.getResource());
        InstantInterval interval = new InstantInterval(occurrenceStart, occurrenceStart);
        ListF<String> instanceIds = ewsProxyWrapper.findInstanceEventIds(email, interval);

        return ewsProxyWrapper.findRecurringMasterIdsByInstanceIds(instanceIds)
                .findBy2(Cf2.isSomeF(master.getExchangeId())).get().get1();
    }

    private void cancelOrDeclineMeetings(ListF<? extends ResourceMeeting> meetings, ActionInfo actionInfo) {
        ewsProxyWrapper.cancelOrDeclineMeetingSafe(meetings.map(ResourceMeeting::getExchangeIdLogData), actionInfo);
    }

    private ListF<ResourceMeeting> cancelProcessedIfException(
            final ListF<ResourceParticipantInfo> resources,
            final Function<ResourceParticipantInfo, ResourceMeeting> performer, ActionInfo actionInfo)
    {
        return cancelStoredIfException(stored -> {
            for (ResourceParticipantInfo r : resources) {
                stored.add(performer.apply(r));
            }
        }, actionInfo);
    }

    private ListF<ResourceMeeting> cancelStoredIfException(
            Function1V<ArrayListF<ResourceMeeting>> callback, ActionInfo actionInfo)
    {
        ArrayListF<ResourceMeeting> stored = new ArrayListF<ResourceMeeting>();
        try {
            callback.apply(stored);
            return stored;

        } catch (Exception e) {
            cancelOrDeclineMeetings(stored, actionInfo);
            throw ExceptionUtils.throwException(e);
        }
    }

    private ReplaceResourcesInfo getResourcesToReplaceMeetings(
            ListF<ExchangeResourceMeeting> replacingMeetings, MainEventInfo mainEvent)
    {
        MapF<Long, ListF<ResourceParticipantInfo>> replacingByEventId = replacingMeetings
                .toTuple2List(ExchangeResourceMeeting.getEventIdF(), ExchangeResourceMeeting.getResourceParticipantF())
                .groupBy1();

        Function<Long, ListF<ResourceParticipantInfo>> replacingByEventIdF =
                id -> replacingByEventId.getOrElse(id, Cf.list());

        Function<ResourceParticipantInfo, Long> resourceIdF = ResourceParticipantInfo.getResourceIdF();

        EventWithRelations masterEvent = mainEvent.getMasterEventsWithRelations().single();
        ListF<ResourceParticipantInfo> masterResources = masterEvent.getResourceParticipantsYaTeamAndSyncWithExchange();
        ListF<ResourceParticipantInfo> resourcesToReplaceMaster = replacingByEventIdF.apply(masterEvent.getId());

        MapF<Long, RecurrenceReplaceResourcesInfo> recurrencesInfo = Cf.hashMap();
        for (EventWithRelations recurrence : mainEvent.getRecurrenceEventsWithRelations()) {
            ListF<ResourceParticipantInfo> resources = recurrence.getResourceParticipantsYaTeamAndSyncWithExchange();
            ListF<ResourceParticipantInfo> replacingResources = replacingByEventIdF.apply(recurrence.getId());

            ListF<Long> toCancel = masterResources.map(resourceIdF)
                    .filter(resources.map(resourceIdF).containsF().notF())
                    .filter(resourcesToReplaceMaster.map(resourceIdF).containsF());
            ListF<ResourceParticipantInfo> toModify = resources
                    .filter(resourceIdF.andThen(resourcesToReplaceMaster.map(resourceIdF).containsF()));
            ListF<ResourceParticipantInfo> toSingle = replacingResources
                    .filter(resourceIdF.andThen(masterResources.map(resourceIdF).containsF().notF()));

            recurrencesInfo.put(recurrence.getId(), new RecurrenceReplaceResourcesInfo(toModify, toSingle, toCancel));
        }
        return new ReplaceResourcesInfo(resourcesToReplaceMaster, recurrencesInfo);
    }

    private Function<EventWithRelations, ListF<ExchangeResourceMeeting>> findResourcesMeetingInExchangeF() {
        return e -> findResourcesMeetingInExchange(e);
    }

    private ListF<ExchangeResourceMeeting> loadMeetingsFromExchange(ListF<ResourceMeeting> meetings) {
        return findMeetingsByExchangeIds(meetings.toTuple2List(
                ExchangeResourceMeeting.getResourceParticipantF(),
                ExchangeResourceMeeting.getExchangeIdF()));
    }

    private ListF<ExchangeResourceMeeting> findResourcesMeetingInExchange(EventWithRelations event) {
        ListF<ResourceParticipantInfo> resources = event.getResourceParticipantsYaTeamAndSyncWithExchange();

        Function<ResourceParticipantInfo, Long> resourceIdF = ResourceParticipantInfo.getResourceIdF();

        ListF<ExchangeResourceMeeting> byExchangeIds = findMeetingsByExchangeIds(
                resources.zipWithFlatMapO(ResourceParticipantInfo.getExchangeIdF()));

        ListF<ResourceParticipantInfo> notFound = resources.filter(
                resourceIdF.andThen(byExchangeIds.map(ExchangeResourceMeeting.getResourceIdF()).containsF().notF()));

        ListF<ExchangeResourceMeeting> found = byExchangeIds.plus(findMeetingsByExternalId(notFound, event));
        notFound = resources
                .filter(resourceIdF.andThen(found.map(ExchangeResourceMeeting.getResourceIdF()).containsF().notF()));

        if (notFound.isNotEmpty()) {
            logger.warn("Event {} not found on resources {}", event.getId(), notFound.map(ParticipantInfo.getEmailF()));
        }
        return found;
    }

    private ListF<ExchangeResourceMeeting> findMeetingsByExternalId(
            ListF<ResourceParticipantInfo> resources, EventWithRelations event)
    {
        Validate.forAll(resources, ResourceParticipantInfo.getEventIdF().andThenEquals(event.getId()));

        InstantInterval interval = new InstantInterval(event.getEvent().getStartTs(), event.getEvent().getEndTs());
        Tuple2List<ResourceParticipantInfo, String> exchangeIds = Tuple2List.arrayList();

        for (ResourceParticipantInfo r : resources) {
            Email email = resourceRoutines.getExchangeEmail(r.getResource());
            Option<String> id = ewsProxyWrapper.findMasterAndSingleOrInstanceEventId(
                    email, interval, event.getExternalId(), event.getRecurrenceId().isPresent());

            if (id.isPresent()) {
                exchangeIds.add(r, id.get());
            }
        }
        return findMeetingsByExchangeIds(exchangeIds);
    }

    private ListF<ExchangeResourceMeeting> findMeetingsByExchangeIds(
            Tuple2List<ResourceParticipantInfo, String> ownedExchangeIds)
    {
        ListF<UnindexedFieldURIType> fields = Cf.list(
                UnindexedFieldURIType.CALENDAR_CONFLICTING_MEETING_COUNT,
                UnindexedFieldURIType.CALENDAR_CALENDAR_ITEM_TYPE,
                UnindexedFieldURIType.CALENDAR_UID,
                UnindexedFieldURIType.CALENDAR_RECURRENCE_ID);

        ListF<String> extendedFields = Cf.list(
                EwsUtils.EXTENDED_PROPERTY_SOURCE, EwsUtils.EXTENDED_PROPERTY_RECURRENCE_ID);

        MapF<String, CalendarItemType> foundByExchangeId = ewsProxyWrapper
                .getEvents(ownedExchangeIds.get2(), fields, extendedFields)
                .toMapMappingToKey(EwsUtils.calendarItemExchangeIdF());

        return Cf2.flatBy2(ownedExchangeIds.map2(foundByExchangeId::getO)).map(ExchangeResourceMeeting::new);
    }

    private MasterCreateData convertMaster(MainEventInfo mainEvent) {
        EventInfo e = mainEvent.getMasterEventInfos().single();
        CalendarItemType item = eventToCalendarItemConverter.convertToCalendarItem(
                e.getEventWithRelations(), e.getRepetitionInstanceInfo());

        return new MasterCreateData(e, item, e.getRepetitionInstanceInfo().getExdates().map(Rdate.getStartTsF()));
    }

    private MapF<Long, RecurrenceCreateData> convertRecurrences(MainEventInfo mainEvent) {
        final Event masterEvent = mainEvent.getMasterEventInfos().single().getEvent();

        return mainEvent.getRecurrenceEventInfos().toMap(e -> {
            CalendarItemType createData = eventToCalendarItemConverter.convertToCalendarItem(
                    e.getEventWithRelations(),
                    e.getRepetitionInstanceInfo());

            ListF<ItemChangeDescriptionType> updateData = eventToCalendarItemConverter.convertToChangeDescriptions(
                    e.getEventWithRelations(),
                    e.getRepetitionInstanceInfo(),
                    eventChangesFinder.getEventChangesInfoForExchangeForNewRecurrence(masterEvent, e.getEvent()));

            return Tuple2.tuple(
                    e.getEventId(), new RecurrenceCreateData(e, createData, e.getRecurrenceId().get(), updateData));
        });
    }
}
