package ru.yandex.calendar.frontend.ews.subscriber;

import java.util.ArrayList;
import java.util.List;

import com.microsoft.schemas.exchange.services._2006.types.ResponseClassType;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Minutes;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.ews.EwsResourceSubscriptionFailedException;
import ru.yandex.calendar.frontend.ews.EwsSubjectNotSubscribedException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.YtEwsSubscriptionDao;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscriptionFields;
import ru.yandex.calendar.logic.contact.UnivContact;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class ExchangeSubscriber {
    private static final Logger logger = LoggerFactory.getLogger(ExchangeSubscriber.class);

    @Autowired
    private UserManager userManager;
    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private YtEwsSubscriptionDao ytEwsSubscriptionDao;
    @Autowired
    private ResourceDao resourceDao;


    public EwsSubscribeResult subscribeUser(PassportUid uid) {
        Email email = userManager.getLdEmailByUid(uid);
        return subscribeToPushInner(UidOrResourceId.user(uid), email);
    }

    public EwsSubscribeResult subscribeResource(long resourceId) {
        Resource resource = resourceDao.findResourceById(resourceId);
        final Email resourceEmail = resourceRoutines.getExchangeEmail(resource);
        return subscribeToPushInner(UidOrResourceId.resource(resourceId), resourceEmail);
    }

    public EwsSubscribeResult subscribeUserToPull(PassportUid uid) {
        return subscribeToPullInner(UidOrResourceId.user(uid), userManager.getLdEmailByUid(uid));
    }

    public EwsSubscribeResult subscribeResourceToPull(long resourceId) {
        return subscribeToPullInner(UidOrResourceId.resource(resourceId),
                resourceRoutines.getExchangeEmail(resourceDao.findResourceById(resourceId)));
    }

    private EwsSubscribeResult subscribeToPushInner(UidOrResourceId subjectId, Email email) {
        logger.debug("Email for subscription: " + email);
        EwsSubscribeResult ewsSubscribeResult = ewsProxyWrapper.subscribeToPush(email);
        if (ewsSubscribeResult.getSubscriptionId().isPresent()) {
            String subscrId = ewsSubscribeResult.getSubscriptionId().get();
            Option<YtEwsSubscription> ewsSubscrO = findSubscriptionOBySubjectId(subjectId);
            YtEwsSubscription ewsSubscr = new YtEwsSubscription();

            ewsSubscr.setSubscriptionId(subscrId);
            ewsSubscr.setSubscriptionTs(AuxDateTime.NOWTS());
            if (subjectId.isUser()) {
                ewsSubscr.setUid(subjectId.getUid());
            } else {
                ewsSubscr.setResourceId(subjectId.getResourceId());
            }
            ewsSubscr.setEmail(ewsSubscribeResult.getLastAttemptEmail());
            if (ewsSubscrO.isPresent()) {
                ewsSubscr.setId(ewsSubscrO.get().getId());
                ytEwsSubscriptionDao.updateYtEwsSubscription(ewsSubscr);
            } else {
                ytEwsSubscriptionDao.saveYtEwsSubscription(ewsSubscr);
            }
        } else {
            logger.debug("Couldn't subscribe email: " + email);
        }
        return ewsSubscribeResult;
    }

    private EwsSubscribeResult subscribeToPullInner(UidOrResourceId subjectId, Email email) {
        logger.debug("Email for subscription: " + email);
        EwsSubscribeResult result = ewsProxyWrapper.subscribeToPull(email, Option.empty());

        if (result.getResponseMessage().getResponseClass() == ResponseClassType.ERROR) {
            throw new EwsSubjectNotSubscribedException(result.getResponseMessage().getMessageText());
        }
        Option<YtEwsSubscription> subscription = findSubscriptionOBySubjectId(subjectId);
        YtEwsSubscription data = new YtEwsSubscription();

        data.setPullSubscriptionId(result.getSubscriptionId().getOrThrow("no subscription id"));
        data.setPullWatermark(result.getWatermark().getOrThrow("no subscription watermark"));
        data.setPullSubscriptionTs(Instant.now());
        data.setEmail(email);
        data.setUid(subjectId.getUidO());
        data.setResourceId(subjectId.getResourceIdO());

        if (subscription.isPresent()) {
            data.setId(subscription.get().getId());
            ytEwsSubscriptionDao.updateYtEwsSubscription(data);
        } else {
            ytEwsSubscriptionDao.saveYtEwsSubscription(data);
        }
        return result;
    }

    public Option<YtEwsSubscription> findSubscriptionOBySubjectId(UidOrResourceId subjectId) {
        if (subjectId.isUser()) {
            return ytEwsSubscriptionDao.findUserSubscription(subjectId.getUid());
        } else if (subjectId.isResource()) {
            return ytEwsSubscriptionDao.findResourceSubscription(subjectId.getResourceId());
        } else {
            throw new IllegalStateException("Unknown type of subjectId: " + subjectId);
        }
    }

    public YtEwsSubscription findSubscriptionBySubjectId(UidOrResourceId subjectId) {
        Option<YtEwsSubscription> resO = findSubscriptionOBySubjectId(subjectId);
        return resO.getOrThrow(EwsSubjectNotSubscribedException.consF(subjectId));
    }

    public ListF<YtEwsSubscription> findSubscriptionsBySubjectIds(CollectionF<UidOrResourceId> ids) {
        return Cf.<YtEwsSubscription>list()
                .plus(ytEwsSubscriptionDao.findUsersSubscriptions(ids.filterMap(UidOrResourceId::getUidO)))
                .plus(ytEwsSubscriptionDao.findResourcesSubscriptions(ids.filterMap(UidOrResourceId::getResourceIdO)));
    }

    /**
     * @param email @yandex-team, @msft (don't use), but not @ld
     */
    public YtEwsSubscription findSubscriptionByEmail(Email email) {
        final Option<UidOrResourceId> subjectIdO = userManager.getSubjectIdByEmail(email);
        return findSubscriptionBySubjectId(subjectIdO.getOrThrow("Subject is not found by email: " + email));
    }


    public ListF<EwsSubscribeResult> resubscribeExpired() {
        ListF<EwsSubscribeResult> subscribeResult = Cf.arrayList();
        final Duration duration = Minutes.minutes(EwsUtils.STATUS_FREQUENCY_IN_MINUTES + 1).toStandardDuration();
        Instant expiredTsBound = AuxDateTime.NOWTS().minus(duration);
        ListF<YtEwsSubscription> expiredSubscriptions = ytEwsSubscriptionDao.findExpiredSubscripitons(expiredTsBound);
        subscribeResult.addAll(resubscribeUsers(expiredSubscriptions));
        subscribeResult.addAll(resubscribeResources(expiredSubscriptions));
        return subscribeResult;
    }

    private ListF<EwsSubscribeResult> resubscribeUsers(
            ListF<YtEwsSubscription> expiredSubscriptions)
    {
        ListF<PassportUid> expiredUids =
                expiredSubscriptions.map(YtEwsSubscriptionFields.UID.getF()).filterNotNull();
        return bulkSubscribe(expiredUids);
    }

    private ListF<EwsSubscribeResult> resubscribeResources(
            ListF<YtEwsSubscription> expiredSubscriptions)
    {
        ListF<EwsSubscribeResult> subscribeResult = Cf.arrayList();
        ListF<Long> resourceIds =
                expiredSubscriptions.map(YtEwsSubscriptionFields.RESOURCE_ID.getF()).filterNotNull();
        for (Long resourceId : resourceIds) {
            subscribeResult.add(subscribeResource(resourceId));
        }
        return subscribeResult;
    }

    public ListF<EwsSubscribeResult> updateAllResourceSubscriptions() {
        ListF<EwsSubscribeResult> subscribeResult = Cf.arrayList();
        ListF<Long> activeResourceIds = resourceDao.findAllYaTeamActiveResourceIdsThatSyncWithExchange();
        ListF<Long> subscribedResourceIds = ytEwsSubscriptionDao.findSubscribedResourceIds();
        AuxColl.removeCommon(activeResourceIds, subscribedResourceIds);
        logger.info("Resources: " +
                "non-active to unsubscribe: " + subscribedResourceIds.size() + ", " +
                "active to subscribe: " + activeResourceIds.size());
        // unsubscribe non-active resources
        ytEwsSubscriptionDao.deleteYtEwsSubscriptionsByResourceIds(subscribedResourceIds);
        // subscribe new active resources
        val failedResources = new ArrayList<Long>();

        for (Long resourceId : activeResourceIds) {
            try {
                subscribeResult.add(subscribeResource(resourceId));
            } catch (Exception e) {
                logger.error("Failed to subscribe resource with id " + resourceId, e);
                failedResources.add(resourceId);
            }
        }

        if (!failedResources.isEmpty()) {
            val msg = String.format("The following resource ids were not synchronized: %s. See log for details",
                    StreamEx.of(failedResources).joining(", "));
            throw new EwsResourceSubscriptionFailedException(msg);
        }

        return subscribeResult;
    }

    public ListF<EwsSubscribeResult> subscribeAllUsers() {
        List<UnivContact> yaTeamContacts = StreamEx.of(userManager.getAllYtUsers())
            .flatMap(user -> UnivContact.fromUser(user).stream())
            .toImmutableList();
        List<Email> yaTeamEmails = StreamEx.of(yaTeamContacts)
                .map(UnivContact::getEmail)
                .filter(email -> email.getDomain().equals(PassportAuthDomain.YANDEX_TEAM_RU.getDomain()))
                .toImmutableList();
        return subscribeUsers(yaTeamEmails);
    }

    public ListF<EwsSubscribeResult> subscribeUsers(List<Email> yaTeamEmails) {
        ListF<PassportUid> yaTeamUids = Cf.toArrayList(userManager.getFlatUidValuesByEmailsSafe(Cf.list(yaTeamEmails)));
        logger.debug("Found uids: " + yaTeamUids.size());

        // Subscribe only new (non-subscribed) users
        ListF<PassportUid> subscribedUids = ytEwsSubscriptionDao.findSubscribedUserUids().map(PassportUid::cons);
        yaTeamUids.removeAllTs(subscribedUids.unique());
        logger.debug("Already subscribed uids: " + subscribedUids.size() + ", left to subscribe: " + yaTeamUids.size());

        return bulkSubscribe(yaTeamUids);
    }

    public void unsubscribeDismissedUsers() {
        ytEwsSubscriptionDao.removeDismissedUsersSubscriptions();
    }

    private ListF<EwsSubscribeResult> bulkSubscribe(CollectionF<PassportUid> yaTeamUids) {
        ListF<EwsSubscribeResult> results = Cf.arrayList();
        for (PassportUid yaTeamUid : yaTeamUids) {
            try {
                results.add(subscribeUser(yaTeamUid));
            } catch (Exception e) {
                logger.warn(e, e);
            }
        }
        return results;
    }
}
