package ru.yandex.calendar.frontend.ews.sync;

import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import com.microsoft.schemas.exchange.services._2006.messages.GetEventsResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseClassType;
import lombok.val;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.ews.EwsErrorCodes;
import ru.yandex.calendar.frontend.ews.EwsErrorResponseException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.YtEwsSubscriptionDao;
import ru.yandex.calendar.frontend.ews.hook.EwsNotificationEventHandler;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.frontend.ews.subscriber.EwsSubscribeResult;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.log.log4j.appender.AppenderContextHolder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author dbrylev
 */
public class EwsEventsPuller {
    private static final Logger logger = LoggerFactory.getLogger(EwsEventsPuller.class);

    private final DynamicProperty<Integer> threads = new DynamicProperty<>("ewsPullThreads", 1);

    @Autowired
    private YtEwsSubscriptionDao ytEwsSubscriptionDao;
    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private EwsNotificationEventHandler ewsNotificationEventHandler;
    @Autowired
    private EwsMonitoring ewsMonitoring;

    public EwsPullResult pullAll() {
        ListF<YtEwsSubscription> subscriptions = ytEwsSubscriptionDao.findAllPullSubscriptions();

        logger.info("Working on {} subscriptions", subscriptions.size());

        subscriptions = subscriptions.sorted(YtEwsSubscription.getLastPullTsF()
                .andThen(Cf2.f(Option::getOrNull)).andThenNaturalComparator().nullLowC());

        val counter = new AtomicInteger();
        val failCounter = new AtomicInteger();

        val executor = Executors.newFixedThreadPool(threads.get());
        try {
            Function<Function0<Void>, Function0<Void>> withThreadLocalsF = f ->
                    MasterSlaveContextHolder.withStandardThreadLocalsF(AppenderContextHolder.inheritAppendersF(f));

            executor.invokeAll(subscriptions.map(subscription -> withThreadLocalsF.apply(() -> {
                val rid = RequestIdStack.current().getOrElse(RequestIdStack.generateId());
                val ridHandle = RequestIdStack.pushReplace(rid + "_" + counter.incrementAndGet());

                val requestHandle = CalendarRequest.push(
                        ActionSource.EXCHANGE_PULL, "Synchronizing subscription " + subscription.getEmail().get());
                try {
                    ThreadLocalTimeout.check();
                    pull(subscription, requestHandle.getActionInfo());

                } catch (Exception e) {
                    ewsMonitoring.reportIfEwsException(e);
                    logger.error("Failed to synchronize subscription {}: {}", subscription.getEmail().get(), e);
                    failCounter.incrementAndGet();
                } finally {
                    requestHandle.popSafely();
                    ridHandle.popSafely();
                }
                return null;
            })));
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } finally {
            executor.shutdownNow();
        }
        ThreadLocalTimeout.check();
        return new EwsPullResult(counter.get(), failCounter.get());
    }

    public void pull(YtEwsSubscription subscription, ActionInfo actionInfo) {
        Email email = subscription.getEmail().getOrThrow("no email");
        logger.info("Synchronization for {} started", email);
        do {
            GetEventsResponseMessageType response = ewsProxyWrapper.pull(
                    subscription.getPullSubscriptionId().getOrThrow("no subscription id"),
                    subscription.getPullWatermark().getOrThrow("no subscription watermark"));

            if (Cf.list(EwsErrorCodes.INVALID_WATERMARK, EwsErrorCodes.SUBSCRIPTION_NOT_FOUND)
                    .containsTs(response.getResponseCode()))
            {
                ewsProxyWrapper.unsubscribeSafe(subscription.getPullSubscriptionId().get());

                logger.info("Resubscribing {} using previously stored watermark", email);
                EwsSubscribeResult subscribe = ewsProxyWrapper.subscribeToPull(email, subscription.getPullWatermark());

                if (Cf.list(EwsErrorCodes.INVALID_WATERMARK, EwsErrorCodes.EVENT_NOT_FOUND)
                        .containsTs(subscribe.getResponseMessage().getResponseCode()))
                {
                    logger.error("Got stale subscription {}", subscription);
                    subscribe = ewsProxyWrapper.subscribeToPull(email, Option.empty());
                }
                if (subscribe.getResponseMessage().getResponseClass() == ResponseClassType.ERROR) {
                    throw new EwsErrorResponseException(subscribe.getResponseMessage());
                }
                subscription = updateResubscription(subscription, subscribe);
                response = ewsProxyWrapper.pull(subscribe.getSubscriptionId().get(), subscribe.getWatermark().get());
            }
            if (response.getResponseClass() == ResponseClassType.ERROR) {
                throw new EwsErrorResponseException("Failed to pull " + email, response);
            }
            ewsNotificationEventHandler.handle(
                    EwsUtils.getSubjectId(subscription), response.getNotification(), actionInfo.withNow(Instant.now()));

            subscription = updateWatermark(subscription, Cf.x(response.getNotification()
                    .getCopiedEventOrCreatedEventOrDeletedEvent()).last().getValue().getWatermark());

            if (!Boolean.TRUE.equals(response.getNotification().isMoreEvents())) return;
        } while (true);
    }

    private YtEwsSubscription updateResubscription(YtEwsSubscription subscription, EwsSubscribeResult subscribeResult) {
        YtEwsSubscription data = new YtEwsSubscription();
        data.setId(subscription.getId());
        data.setPullSubscriptionId(subscribeResult.getSubscriptionId().getOrThrow("no subscription id"));
        data.setPullWatermark(subscribeResult.getWatermark().getOrThrow("no subscription watermark"));
        data.setPullSubscriptionTs(Instant.now());

        ytEwsSubscriptionDao.updateYtEwsSubscription(data);
        return subscription.withFields(data);
    }

    private YtEwsSubscription updateWatermark(YtEwsSubscription subscription, String watermark) {
        YtEwsSubscription data = new YtEwsSubscription();
        data.setId(subscription.getId());
        data.setPullWatermark(watermark);
        data.setLastPullTs(Instant.now());

        ytEwsSubscriptionDao.updateYtEwsSubscription(data);
        return subscription.withFields(data);
    }
}
