package ru.yandex.calendar.frontend.ews.sync;

import java.util.concurrent.atomic.AtomicInteger;

import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import io.micrometer.core.instrument.MeterRegistry;
import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.frontend.ews.EwsBadEmailException;
import ru.yandex.calendar.frontend.ews.EwsUtils;
import ru.yandex.calendar.frontend.ews.ExchangeEmailManager;
import ru.yandex.calendar.frontend.ews.YtEwsSubscriptionDao;
import ru.yandex.calendar.frontend.ews.imp.EwsImportStatus;
import ru.yandex.calendar.frontend.ews.imp.EwsImporter;
import ru.yandex.calendar.frontend.ews.imp.ExchangeEventDataConverter;
import ru.yandex.calendar.frontend.ews.proxy.EwsProxyWrapper;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.beans.generated.YtEwsIgnoredEvent;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInstanceStatusChecker;
import ru.yandex.calendar.logic.event.ExchangeEventSynchData;
import ru.yandex.calendar.logic.ics.EventInstanceStatusInfo;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.TranslitUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeout.Handle;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author ssytnik
 */
public class EwsEventsSynchronizer {
    private static final Logger logger = LoggerFactory.getLogger(EwsEventsSynchronizer.class);
    // settings
    private static final boolean HANDLE_USER_SUBSCRIPTIONS = true;
    /**
     * true for quicker initial filling (temporary), false for stable, definite behavior
     */
    private static final boolean SHUFFLE_SUBSCRIPTIONS = false;
    private static final Duration SLEEP_BETWEEN_SUBSCRIPTIONS = Duration.standardSeconds(2);
    private static final String APPLICATION_EWS_EVENTS_SYNCHRONIZER_ERROR_METRIC = "application.ewsEventsSynchronizer" +
            ".error";
    private static final String APPLICATION_EWS_SUBSCRIPTION_ERROR_METRIC = "application.ewsEventsSynchronizer" +
            ".subscription.error";
    private static final String APPLICATION_EWS_EVENTS_SYNCHRONIZER_IGNORE_METRIC = "application" +
            ".ewsEventsSynchronizer.ignore";
    private static final String APPLICATION_EWS_SUBSCRIPTION_PROCESSED_METRIC = "application.ewsEventsSynchronizer" +
            ".processed";

    public final DynamicProperty<ListF<String>> fullSyncLogins = new DynamicProperty<>("ewsFullSyncLogins", Cf.list());

    @Autowired
    private EwsProxyWrapper ewsProxyWrapper;
    @Autowired
    private YtEwsSubscriptionDao ytEwsSubscriptionDao;
    @Autowired
    private EventInstanceStatusChecker eventInstanceStatusChecker;
    @Autowired
    private EwsImporter ewsImporter;
    @Autowired
    private IgnoredEventDao ignoredEventDao;
    @Autowired
    private IgnoredEventManager ignoredEventManager;
    @Autowired
    private ExchangeEmailManager exchangeEmailManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private MeterRegistry registry;


    public SyncStats synchronizeAll(boolean dryRun, boolean onlyFullSyncLogins, boolean onlyUsersSync) {
        long start = System.currentTimeMillis();
        logger.info(String.format("SynchronizeAll(dryRun = %s, onlyFullSyncLogins = %s, onlyUsersSync = %s) started",
                dryRun, onlyFullSyncLogins, onlyUsersSync));

        SyncStats overallStats = new SyncStats();
        AtomicInteger processedSubscriptions = new AtomicInteger(0);
        try {
            ListF<YtEwsSubscription> subscriptions = ytEwsSubscriptionDao.findAllSubscriptions();
            if (!HANDLE_USER_SUBSCRIPTIONS) {
                logger.info("Now removing user subscriptions from " + subscriptions.size() + " one(s)");
                subscriptions = subscriptions.filterNot(a -> a.getUid().isPresent());
            }
            if (onlyFullSyncLogins) {
                subscriptions = subscriptions
                        .filter(s -> s.getEmail().exists(e -> fullSyncLogins.get().containsTs(e.getLocalPart())));
            }
            if (onlyUsersSync) {
                subscriptions = subscriptions.filter(s -> s.getUid().isPresent());
            }

            logger.info("Working on " + subscriptions.size() + " subscription(s)");

            if (SHUFFLE_SUBSCRIPTIONS) {
                subscriptions = Random2.R.shuffle(subscriptions);
                logger.info("Subscription(s) are shuffled");
            }
            subscriptions.stream().forEach(subscription -> {
                if (Duration.standardMinutes(120).getMillis() < (System.currentTimeMillis() - start)) {
                    logger.warn("Time is almost over, I stop subscriptions processing");
                }
                CalendarRequestHandle requestHandle = CalendarRequest.push(ActionSource.EXCHANGE_SYNCH,
                        "Synchronizing subscription " + subscription.getEmail());
                try {
                    SyncStats subStats = synchronizeBySubscription(subscription, dryRun, requestHandle.getActionInfo());
                    overallStats.add(subStats);
                    processedSubscriptions.incrementAndGet();
                } finally {
                    requestHandle.pop();
                }
                logger.debug("Processing #" + processedSubscriptions + " subscribtions");
            });
        } finally {
            long end = System.currentTimeMillis();
            String diffStr = TimeUtils.millisecondsToSecondsString(end - start);
            logger.info(
                    "synchronizeAll() finished, took " + diffStr + " for " +
                            processedSubscriptions + " subscription(s). " + overallStats
            );
        }
        return overallStats;
    }

    public SyncStats synchronizeByEmail(Email email, boolean dryRun, ActionInfo actionInfo) {
        UidOrResourceId subjectId = userManager.getSubjectIdByEmail(email).getOrThrow("subject not found");
        Email exchangeEmail = exchangeEmailManager.getExchangeEmailByAnyEmailOrGetAsIs(email);

        return synchronize(subjectId, exchangeEmail, dryRun, actionInfo);
    }

    public SyncStats synchronizeByEmailForAdmin(Email email) {
        ActionInfo actionInfo = ActionInfo.adminManagerForEwsSynchronizer();
        return synchronizeByEmail(email, false, actionInfo);
    }

    public SyncStats synchronizeBySubscription(YtEwsSubscription subscription, boolean dryRun, ActionInfo actionInfo) {
        UidOrResourceId subjectId = EwsUtils.getSubjectId(subscription);
        Email exchangeEmail = exchangeEmailManager.getExchangeEmailBySubscription(subscription);

        return synchronize(subjectId, exchangeEmail, dryRun, actionInfo);
    }

    private SyncStats synchronize(UidOrResourceId subjectId, Email exchangeEmail, boolean dryRun,
                                  ActionInfo actionInfo) {
        SyncStats stats = new SyncStats();

        final String commonStatus = "Synchronization for " + subjectId + " (" + exchangeEmail + ")";
        logger.info(commonStatus + " started");
        final long start = System.currentTimeMillis();

        try {
            ListF<CalendarItemType> exchangeEvents = findExchangeEvents(exchangeEmail);
            logger.info(commonStatus + " found " + exchangeEvents.size() + " events");
            for (CalendarItemType calendarItemPartial : exchangeEvents) {
                try {
                    synchronizeEvent(subjectId, calendarItemPartial, stats, dryRun, actionInfo);
                } catch (Exception e) {
                    logger.error(getToStringItem(calendarItemPartial) + " " + e.getMessage(), e);
                    registry.counter(APPLICATION_EWS_EVENTS_SYNCHRONIZER_ERROR_METRIC).increment();
                    stats.incErrorsCount();
                }
                if (isTimeToFinishLongWork()) {
                    logger.warn("Time is almost over, I stop subscription's events processing");
                    break;
                }
            }
        } catch (Exception e) {
            logger.error(commonStatus + " error " + e.getMessage(), e);
            registry.counter(APPLICATION_EWS_SUBSCRIPTION_ERROR_METRIC).increment();
            stats.incSubscriptionErrorsCount();
        } finally {
            String diffStr = TimeUtils.millisecondsToSecondsString(System.currentTimeMillis() - start);
            logger.info(commonStatus + " finished, took " + diffStr + ". " + stats);
        }

        return stats;
    }

    @NotNull
    private String getToStringItem(CalendarItemType calendarItemPartial) {
        return "EXCHANGE_ID: " +
                LogMarker.EXCHANGE_ID.format(calendarItemPartial.getItemId().getId()) +
                " EXT_ID: " +
                LogMarker.EXT_ID.format(StringUtils.defaultIfEmpty(calendarItemPartial.getUID(), "?")) +
                " Item type: " +
                (calendarItemPartial.isIsCancelled() ? "CANCELED, " : "") +
                calendarItemPartial.getCalendarItemType() +
                " Event name: " + calendarItemPartial.getSubject() +
                " LastModifiedTime: " + calendarItemPartial.getLastModifiedTime();
    }

    private void synchronizeEvent(
            UidOrResourceId subjectId, CalendarItemType calendarItemPartial,
            SyncStats stats, boolean dryRun, ActionInfo actionInfo) {
        logger.info(
                "Processing " + calendarItemPartial.getCalendarItemType().value() + " " +
                        "at " + calendarItemPartial.getStart() + ": " + calendarItemPartial.getSubject() + " " +
                        "(" + TranslitUtils.translit(calendarItemPartial.getSubject()) + ") - " + subjectId
        );

        // optimizations: faster go first; try to avoid slow getEvent() call
        // TODO can also check for invalid attendee emails, unknown timezone

        final String exchangeId = calendarItemPartial.getItemId().getId();
        Option<YtEwsIgnoredEvent> ignoredEventO = ignoredEventDao.findIgnoredEventByExchangeId(exchangeId);
        if (ignoredEventO.isPresent()) {
            logger.info("Pre-check: event was ignored: " + ignoredEventO.get().getReason());
            registry.counter(APPLICATION_EWS_EVENTS_SYNCHRONIZER_IGNORE_METRIC).increment();
            stats.incIgnoredCount();
            return;
        }

        /* 2011-12-14 ssytnik@ don't check organizer for human, too
        if (subjectId.isResource()) {
            logger.info("Pre-check: subject is resource: " + email + ", skip organizer check");
        } else {
            Option<Email> orgEmailO = organizerCacheDao.findOrganizerEmailByExchangeId(exchangeId);
            if (orgEmailO.isDefined() && !ExchangeEmailManager.emailsNormalizedByLdDomainEqual(email, orgEmailO.get()
            )) {
                logger.info("Pre-check: organizer not ok: " + orgEmailO.get());
                stats.incIgnoredCount();
                return;
            }
        }
        */

        ExchangeEventSynchData exchangeSynchData = EwsUtils.createExchangeSynchData(calendarItemPartial);
        EventInstanceStatusInfo exchangeStatusInfo = eventInstanceStatusChecker.getStatusByExchangeSynchData(
                subjectId, exchangeSynchData, Option.empty(), actionInfo);
        if (exchangeStatusInfo.isAlreadyUpdated()) {
            logger.info("Pre-check: event synchronization status: " + exchangeStatusInfo);
            registry.counter(APPLICATION_EWS_EVENTS_SYNCHRONIZER_IGNORE_METRIC).increment();
            stats.incIgnoredCount();
            return;
        }

        // slow part

        Option<CalendarItemType> calendarItemO = ewsProxyWrapper.getEvent(exchangeId);
        if (!calendarItemO.isPresent()) {
            logger.warn("Could not find event by exchange id: " + exchangeId); // XXX maybe get error message
            doIfNotDryRun(dryRun, "ignore absent exchange event", new Function0V() {
                public void apply() {
                    ignoredEventManager.storeIgnoredEventSafe(
                            exchangeId, IgnoreReason.NO_EVENT_IN_EXCHANGE, Option.empty());
                }
            });
            registry.counter(APPLICATION_EWS_EVENTS_SYNCHRONIZER_IGNORE_METRIC).increment();
            stats.incIgnoredCount();
            return;
        }

        CalendarItemType calendarItem = calendarItemO.get();
        // XXX speed up recurrence masters: call 'EwsImporter.createOrUpdateEventWithRecurrences' once,
        // use in checkOrganizerAndUpdateCaches, checkAttendeesValidityAndUpdateIgnoreCache and import

        /* 2011-12-14 ssytnik@ don't check organizer for human, too
        // ssytnik@: not interested in "ok" organizers cache, only save non-"ok" ones
        // besides result, it checks if organizer email(s) is valid and can ignore event
        if (!checkOrganizerAndUpdateCaches(calendarItem, email, dryRun)) {
            if (!subjectId.isResource()) {
                stats.incIgnoredCount();
                return;
            }
        }
        */

        if (!checkAttendeesValidityAndUpdateIgnoreCache(calendarItem, dryRun)) {
            registry.counter(APPLICATION_EWS_EVENTS_SYNCHRONIZER_IGNORE_METRIC).increment();
            stats.incIgnoredCount();
            return;
        }

        Handle tltHandle = ThreadLocalTimeout.push(Duration.standardSeconds(60), "importing exchange event");
        try {
            ListF<EwsImportStatus> statuses = ewsImporter.createOrUpdateEventWithRecurrences(
                    subjectId, calendarItem, actionInfo, dryRun);
            Validate.notEmpty(statuses);
            registry.counter(APPLICATION_EWS_SUBSCRIPTION_PROCESSED_METRIC).increment();
            stats.incProcessedCount();
            for (EwsImportStatus status : statuses) {
                registry.counter(APPLICATION_EWS_SUBSCRIPTION_PROCESSED_METRIC + "." + status.toString().toLowerCase()).increment();
                stats.incProcessedByStatusCount(status);
            }
        } finally {
            tltHandle.popSafely();
        }
    }


    private ListF<CalendarItemType> findExchangeEvents(final Email exchangeEmail) {
        final InstantInterval instantInterval = new InstantInterval(LocalTime.now().minusHours(24).toDateTimeToday(),
                Duration.standardDays(90));
        final Option<InstantInterval> intervalO = Option.of(instantInterval);
        ListF<CalendarItemType> res = ewsProxyWrapper.findMasterAndSingleEvents(exchangeEmail, intervalO);

        logger.debug("Found " + res.size() + " events from mailbox for interval " + intervalO);
        return res;
    }

    /*
    private ListF<CalendarItemType> filterByOrganizer(final ListF<CalendarItemType> events, final Email email) {
        ListF<CalendarItemType> res = events.filter(new Function1B<CalendarItemType>() {
            public boolean apply(CalendarItemType a) {
                return eventBelongsTo(a, email);
            }
        });
        logger.debug("Filtered " + res.size() + " non-meetings / my meetings");

        return res;
    }


    private boolean checkOrganizerAndUpdateCaches(final CalendarItemType event, final Email email, boolean dryRun) {
        try {
            Option<Email> orgEmailO = ExchangeEventDataConverter.getOrganizerEmailSafe(event);
            boolean res;
            if (!orgEmailO.isPresent()) {
                // IsMeeting property is not applicable (== false for meetings with participants).
                // maybe: res = ExchangeEventDataConverter.getAllAttendees(event).isEmpty();
                res = event.getOrganizer() == null;
                if (!res) {
                    logger.warn("Organizer mailbox presents, but has empty email");
                    doIfNotDryRun(dryRun, "mark event as having empty organizer", () -> ignoredEventManager
                    .storeIgnoredEventSafe(
                            event.getItemId().getId(), IgnoreReason.BAD_ORGANIZER_EMAIL, Option.of("email is empty")));
                }
            } else {
                final Email orgEmail = orgEmailO.get();
                res = ExchangeEmailManager.emailsNormalizedByLdDomainEqual(email, orgEmail);
                if (!res) {
                    logger.info("- organizer not ok: " + orgEmail);
                    doIfNotDryRun(dryRun, "add organizer email to cache", () -> organizerCacheManager
                    .storeOrganizerCacheSafe(event.getItemId().getId(), orgEmail));
                }
            }
            if (res) { // maybe move to "!orgEmail.isEmpty()" branch; maybe do same checks as for organizer in master
             instance
                for (CalendarItemType occurrence : ewsProxyWrapper.getModifiedOccurrences(event)) {
                    ExchangeEventDataConverter.getOrganizerEmailSafe(occurrence);
                }
            }
            return res;
        } catch (final EwsBadEmailException e) {
            logger.warn("Could not check organizer: " + e);
            doIfNotDryRun(dryRun, "mark event as having bad organizer", new Function0V() { public void apply() {
                ignoredEventManager.storeIgnoredEventSafe(
                        event.getItemId().getId(), IgnoreReason.BAD_ORGANIZER_EMAIL, Option.of(e.getMessage()));
            } });
            return false; // ignore event with bad email
        }
    }
*/
    // XXX can merge with checkOrganizerAndUpdateCaches (need to move reason: 'bad_organizer_email' to 'bad_email')
    private boolean checkAttendeesValidityAndUpdateIgnoreCache(final CalendarItemType event, boolean dryRun) {
        ewsImporter.fixEmailsIfPossible(event);
        try {
            for (CalendarItemType eventOrOccurrence : ewsProxyWrapper.getEventAndItsModifiedOccurrences(event)) {
                ExchangeEventDataConverter.getAttendeeEmails(eventOrOccurrence);
            }
            return true;
        } catch (final EwsBadEmailException e) {
            logger.warn("Event contains bad attendee email: " + e);
            doIfNotDryRun(dryRun, "mark event as having bad attendee email",
                    () -> ignoredEventManager.storeIgnoredEventSafe(
                    event.getItemId().getId(), IgnoreReason.BAD_ATTENDEE_EMAIL, Option.of(e.getMessage())));
            return false; // ignore event with bad email
        }
    }

    private void doIfNotDryRun(boolean dryRun, String whatToDo, Function0V f) {
        if (dryRun) {
            logger.warn("Dry-run mode is on, do not " + whatToDo);
        } else {
            f.apply();
        }
    }

    /*
    private ListF<EventInstanceInfo> getCalendarEvents(YtEwsSubscription subscription, InstantInterval interval) {
        SubjectId subjectId = EwsUtils.getSubjectId(subscription);
        PassportUid uid;
        LayerIdPredicate layerIds;
        if (subjectId.isUser()) {
            uid = subjectId.getUid();
            layerIds = LayerIdPredicate.allForUser(uid, false);
        } else if (subjectId.isResource()) {
            Resource resource = resourceRoutines.loadById(subjectId.getResourceId());
            Layer layer = layerRoutines.getLayerById(resource.getLayerId());
            uid = new PassportUid(layer.getCreatorUid()); // for tests, != ResourceRoutines.MASTER_OF_RESOURCES
            layerIds = LayerIdPredicate.list(Cf.list(resource.getLayerId()), false);
        } else {
            throw new IllegalStateException("Unknown type of subjectId: " + subjectId);
        }
        EventGetProps egp = new EventGetProps();
        // TODO filter
        return eventRoutines.getSortedInstances(
            Option.some(uid), interval.getStart().toInstant(), Option.some(interval.getEnd().toInstant()),
            egp, layerIds, ActionSource.INTERNAL_API
        );
    }
    */

    /*
    // returns: @yandex-team only, but not: @ld, @msft
    public Email getEmailBySubjectId(SubjectId subjectId) {
        if (subjectId.isUser()) {
            final Option<Email> emailO = userManager.getEmailByUid(subjectId.getUid());
            if (emailO.isEmpty()) {
                throw new IllegalArgumentException("User email is not found by subjectId: " + subjectId);
            }
            return emailO.get();
        } else if (subjectId.isResource()) {
            return resourceRoutines.getResourceEmailByResourceId(subjectId.getResourceId());
        } else {
            throw new IllegalStateException("Unknown type of subjectId: " + subjectId);
        }
    }
    */


    // (RR) is it useful to oblige other long cron tasks (like CmdCalGetEvents) stop themselves on time?
    public boolean isTimeToFinishLongWork() {
        return ThreadLocalTimeout.expiresWithin(Duration.standardSeconds(30));
    }
}
