package ru.yandex.calendar.frontend.kiosk;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.OfficeFields;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.util.dates.InstantIntervalSet;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 */
class KioskManager {
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private TransactionTemplate transactionTemplate;

    public ListF<RoomState> getRoomStates(ListF<String> exchangeNames, final Instant now) {
        ListF<ResourceInfo> resourcesWithLayers = resourceDao.findResourceInfosByExchangeNames(exchangeNames);

        if (resourcesWithLayers.isEmpty()) {
            return Cf.list();
        }

        Check.C.hasSize(1, resourcesWithLayers.map(ResourceInfo.officeF().andThen(OfficeFields.ID.getF())).unique(),
                "Rooms are from different offices!");

        DateTimeZone officeTz = OfficeManager.getOfficeTimeZone(resourcesWithLayers.first().getOffice());
        //ListF<Long> resourceIds = resourcesWithLayers.map(ResourceInfo.resourceF()).map(ResourceFields.ID.getF());

        final Instant endOfDay = now.toDateTime(officeTz).plusDays(1).withTimeAtStartOfDay().toInstant();
        Tuple2List<ResourceInfo, InstantIntervalSet> busyIntervals = resourceScheduleManager.getResourceBusyIntervals(
                Option.<PassportUid>empty(), resourcesWithLayers, now, endOfDay, officeTz, getActionInfo());

        ListF<RoomState> roomStates = busyIntervals.map(new Function2<ResourceInfo, InstantIntervalSet, RoomState>() {
            public RoomState apply(ResourceInfo resourceInfo, InstantIntervalSet instantIntervals) {
                // @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#rezultat
                Option<InstantInterval> nearestEvent =
                        instantIntervals.mergeNeighborsCloserThan(Duration.standardMinutes(5)).getIntervals().firstO();

                // XXX city?
                if (!nearestEvent.isPresent()) {
                    return RoomState.nowFree(resourceInfo, new Duration(now, endOfDay), "Москва");
                } else if (nearestEvent.get().getStart().isAfter(now)) {
                    return RoomState.nowFree(resourceInfo, new Duration(now, nearestEvent.get().getStart()), "Москва");
                } else {
                    return RoomState.nowBusy(resourceInfo, new Duration(now, nearestEvent.get().getEnd()), "Москва");
                }
            }
        });

        return roomStates;
    }

    public ListF<Room> getRoomsByOfficeId(long officeId) {
        return resourceDao.findResourcesByOfficeId(officeId).map(Room.consF());
    }

    public long createMeetingInRoom(String exchangeName, Instant start, Duration duration) {
        final Resource resource = resourceDao.findResourceByExchangeName(exchangeName);
        final DateTimeZone officeTz = officeManager.getTimeZoneByOfficeId(resource.getOfficeId());

        final EventData eventData = new EventData();

        eventData.getEvent().setStartTs(start);
        eventData.getEvent().setEndTs(start.plus(duration));
        eventData.getEvent().setName("Встреча создана в киоске");
        eventData.getEvent().setIsAllDay(false);
        eventData.setTimeZone(officeTz);

        return transactionTemplate.execute(s -> {
            {
                long mainEventId = eventRoutines.createMainEvent(officeTz, getActionInfo());
                eventData.getEvent().setMainEventId(mainEventId);

                return eventRoutines.createUserOrFeedEvent(UidOrResourceId.resource(resource.getId()),
                        EventType.USER, mainEventId,
                        eventData, NotificationsData.create(Cf.<Notification>list()),
                        InvitationProcessingMode.SAVE_ATTACH, getActionInfo()
                ).getEvent().getId();
            }
        });
    }

    public void deleteMeetingInRoom(long eventId, String exchangeName) {
        eventRoutines.deleteEventsSafe(
                Option.<UserInfo>empty(), Cf.list(eventId), InvitationProcessingMode.SAVE_ATTACH, getActionInfo());
    }

    private ActionInfo getActionInfo() {
        return new ActionInfo(ActionSource.KIOSK, RequestIdStack.current().getOrElse("?"), Instant.now());
    }

}
