package ru.yandex.calendar.frontend.kiosk;

import java.io.IOException;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.monitoring.WebApiMonitoring;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.web.servlet.HttpServletResponseX;

/**
 * @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal
 */
@SuppressWarnings("serial")
public class KioskServlet extends HttpServlet {
    private static final String STATE_ROUTE = "/state";
    private static final String ROOMS_ROUTE = "/rooms";
    private static final String OCCUPY_ROUTE = "/occupy";
    private static final String RELEASE_ROUTE = "/release";
    private static final Logger logger = LoggerFactory.getLogger(KioskServlet.class);
    @Autowired
    private KioskManager kioskManager;
    @Autowired
    private WebApiMonitoring webApiMonitoring;

    @Override
    protected void service(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        val reqX = HttpServletRequestX.wrap(req);
        val pathInfo = reqX.getPathInfoO().getOrElse("");
        val handle = CalendarRequest.push(ActionSource.KIOSK, pathInfo);
        try {
            final JsonValue result;

            if (pathInfo.matches(STATE_ROUTE)) {
                result = state(reqX);
            } else if (pathInfo.matches(ROOMS_ROUTE)) {
                result = rooms(reqX);
            } else if (pathInfo.matches(OCCUPY_ROUTE)) {
                result = occupy(reqX);
            } else if (pathInfo.matches(RELEASE_ROUTE)) {
                result = release(reqX);
            } else {
                throw new IllegalArgumentException(pathInfo);
            }

            HttpServletResponseX.wrap(resp)
                    .writeContent(result.serialize().getBytes("UTF-8"), "application/json; charset=utf-8");
        } finally {
            handle.popSafely();
        }
    }

    /**
     * @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#sostojaniezadannyxperegovorok/stateget
     */
    private JsonValue state(HttpServletRequestX reqX) {
        try {
            val exchangeNames = Cf.list(reqX.getParameter("rooms").split(","));
            val states = kioskManager.getRoomStates(exchangeNames, new Instant());
            val result = new JsonArray(states.map(RoomState.toJsonF()));
            webApiMonitoring.reportSuccessExecution(reqX.getRequestURI());
            return result;
        } catch (Exception e) {
            webApiMonitoring.reportErrorExecution(reqX.getRequestURI(), e);
            throw e;
        }
    }

    /**
     * @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#spisokjetazhejjiperegovorokzadannogoofisa/roomsget
     */
    private JsonValue rooms(HttpServletRequestX reqX) {
        try {
            val officeId = Long.parseLong(reqX.getParameter("office_id"));
            val rooms = kioskManager.getRoomsByOfficeId(officeId);
            val result = new JsonArray(rooms.map(Room.toJsonf()));
            webApiMonitoring.reportSuccessExecution(reqX.getRequestURI());
            return result;
        } catch (Exception e) {
            webApiMonitoring.reportErrorExecution(reqX.getRequestURI(), e);
            throw e;
        }
    }

    /**
     * @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#bronirovanieperegovorki/occupypost
     */
    private JsonValue occupy(HttpServletRequestX reqX) {
        try {
            String exchangeName = reqX.getParameter("room");
            val durationMinutes = Integer.parseInt(reqX.getParameter("duration"));
            val eventId = kioskManager
                    .createMeetingInRoom(exchangeName, new Instant(), Duration.standardMinutes(durationMinutes));

            val result = new JsonObject(
                    Tuple2List.<String, JsonValue>tuple2List()
                            .plus1("status", JsonString.valueOf("ok"))
                            .plus1("event_id", JsonNumber.valueOf(eventId))
                            .plus1("message", JsonString.valueOf(""))
            );
            webApiMonitoring.reportSuccessExecution(reqX.getRequestURI());
            return result;
        } catch (Exception e) {
            webApiMonitoring.reportErrorExecution(reqX.getRequestURI(), e);
            logger.error(e, e);
            return new JsonObject(
                    Tuple2List.<String, JsonValue>tuple2List()
                            .plus1("status", JsonString.valueOf("error"))
                            .plus1("message", JsonString.valueOf("Ошибка"))
            );
        }
    }

    /**
     * http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#otmenabronirovanija/releasepost
     */
    private JsonValue release(HttpServletRequestX reqX) {
        try {
            val eventId = Long.parseLong(reqX.getParameter("event_id"));
            val exchangeName = reqX.getParameter("room");
            kioskManager.deleteMeetingInRoom(eventId, exchangeName);

            val result = new JsonObject(Tuple2List.<String, JsonValue>tuple2List()
                    .plus1("status", JsonString.valueOf("ok"))
                    .plus1("message", JsonString.valueOf("")));
            webApiMonitoring.reportSuccessExecution(reqX.getRequestURI());
            return result;
        } catch (Exception e) {
            webApiMonitoring.reportErrorExecution(reqX.getRequestURI(), e);
            logger.error(e, e);
            return new JsonObject(Tuple2List.<String, JsonValue>tuple2List()
                    .plus1("status", JsonString.valueOf("error"))
                    .plus1("message", JsonString.valueOf("Ошибка")));
        }
    }
}
