package ru.yandex.calendar.frontend.kiosk;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;

/**
* @author gutman
*/
class RoomState {

    private final ResourceInfo resourceInfo;
    private final boolean busy;
    private final Duration untilStateChange;
    private final String cityName;

    public RoomState(ResourceInfo resourceInfo, boolean busy, Duration untilStateChange, String cityName) {
        this.resourceInfo = resourceInfo;
        this.busy = busy;
        this.untilStateChange = untilStateChange;
        this.cityName = cityName;
    }

    public static Function<RoomState, JsonValue> toJsonF() {
        return new Function<RoomState, JsonValue>() {
            public JsonObject apply(RoomState roomState) {
                return roomState.toJson();
            }
        };
    }

    public static RoomState nowFree(ResourceInfo resourceInfo, Duration untilStateChange, String cityName) {
        return new RoomState(resourceInfo, false, untilStateChange, cityName);
    }

    public static RoomState nowBusy(ResourceInfo resourceInfo, Duration untilStateChange, String cityName) {
        return new RoomState(resourceInfo, true, untilStateChange, cityName);
    }

    public boolean isBusy() {
        return busy;
    }

    public Duration getUntilStateChange() {
        return untilStateChange;
    }

    /**
     * @url http://wiki.yandex-team.ru/Invite/dev/kiosk/cal#rezultat
     */
    public JsonObject toJson() {
        return new JsonObject(Tuple2List.<String, JsonValue>tuple2List()
                .plus1("name", JsonString.valueOf(resourceInfo.getResource().getExchangeName().getOrElse("undefined")))
                .plus1("state", JsonString.valueOf(busy ? "busy" : "free"))
                .plus1("duration", JsonNumber.valueOf(untilStateChange.getStandardSeconds() / 60))
                .plus1("office_id", JsonNumber.valueOf(resourceInfo.getOffice().getInviteId().getOrElse(0L))) // ?
                .plus1("office_name", JsonString.valueOf(resourceInfo.getOffice().getName()))
                .plus1("city", JsonString.valueOf(cityName))
                .plus1("floor_id", JsonNumber.valueOf(resourceInfo.getResource().getFloorId())));
    }
}
