package ru.yandex.calendar.frontend.mailhook;

import java.util.Optional;

import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.apache.commons.io.Charsets;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Try;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;

@Slf4j
public abstract class BaseMailhookService implements MailhookService {
    @Autowired
    protected IcsImporter icsImporter;
    @Autowired
    protected UserManager userManager;
    @Autowired
    protected EventInvitationManager eventInvitationManager;

    protected abstract MailhookData prepareMailhookData(HttpServletRequest req);

    @SneakyThrows
    @Override
    public void execute(HttpServletRequest req, HttpServletResponse response) {
        if (req.getContentLength() == 0)
            throw new RuntimeException("got empty request content length");

        val mailhookPostData = prepareMailhookData(req);
        log.debug("Mailhook servlet: got new message");
        val uid = new PassportUid(Long.parseLong(mailhookPostData.uid));
        val from = mailhookPostData.from;
        val to = mailhookPostData.to;
        Optional<Email> emailByUid = userManager.getEmailByUid(uid);
        emailByUid.ifPresent(email -> log.info(LogMarker.currentUser(uid, email)));
        log.info(LogMarker.MESSAGE_FROM.format(from));
        log.info(LogMarker.MESSAGE_TO.format(to));
        val messageId = mailhookPostData.messageId;
        val postfixId = mailhookPostData.postfixId;
        log.info(LogMarker.MESSAGE_ID.format(messageId));
        log.info(LogMarker.POSTFIX_ID.format(postfixId));

        val calendar = IcsUtils.parseBytesInGuessedEncoding(decodeIcsCalendar(mailhookPostData.icsCalendarData).getBytes(Charsets.UTF_8));
        val fromEmailTry = Try.tryCatchException(() -> new Email(new InternetAddress(from).getAddress()));
        val fromUid = fromEmailTry.mapCatchException(userManager::getUidByEmail).toOption().flatMapO(u -> u);
        ListF<Email> recipientEmails = Cf.arrayList();
        for (String address: Option.ofNullable(to).getOrElse("").split(",")) {
            try {
                recipientEmails.add(new Email(new InternetAddress(address).getAddress()));
            } catch (Exception ignored) {}
        }
        val recipientIds = eventInvitationManager.getParticipantIdsByEmails(recipientEmails).get2();

        icsImporter.importIcsFromMailhook(uid, fromUid, recipientIds, calendar);
    }

    abstract String decodeIcsCalendar(String rawCalendar);

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    static class MailhookData {
        @JsonProperty("yandex-uid")
        private String uid;
        private String from;
        private String to;
        @JsonProperty("message-id")
        private String messageId;
        @JsonProperty("postfix-id")
        private String postfixId;
        @JsonProperty("calendar-data")
        private String icsCalendarData;
    }
}
