package ru.yandex.calendar.frontend.mailhook;

import java.util.Optional;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import net.fortuna.ical4j.data.ParserException;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.postgresql.util.PSQLException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.TransientDataAccessException;

import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.AddrUtils;
import ru.yandex.calendar.frontend.HeaderUtils;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.calendar.monitoring.WebApiMonitoring;
import ru.yandex.calendar.tvm.TvmManager;
import ru.yandex.commune.a3.security.UnauthenticatedException;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.misc.db.masterSlave.MasterSlaveUnitUnavailableException;
import ru.yandex.misc.io.http.HttpStatus;

@Slf4j
public abstract class MailhookBaseCallbackServlet extends HttpServlet {
    @Autowired
    private WebApiMonitoring webApiMonitoring;

    @Autowired
    private EwsMonitoring ewsMonitoring;
    @Autowired
    private TvmManager tvmManager;

    public abstract MailhookService getMailhookService();

    private static <T extends Throwable> Optional<T> findFirstMatchingNestedException(Throwable e, Class<T> type) {
        return StreamEx.of(ExceptionUtils.getThrowableList(e))
                .select(type)
                .findFirst();
    }

    private static int getStatusCode(Exception e) {
        if (e instanceof UnauthenticatedException || e instanceof UnauthorizedException) {
            return HttpStatus.SC_403_FORBIDDEN;
        } else if (e instanceof IllegalArgumentException
                || e instanceof IllegalStateException
                || findFirstMatchingNestedException(e, ParserException.class).isPresent()) {
            return HttpStatus.SC_400_BAD_REQUEST;
        } else if (e instanceof TransientDataAccessException
                || e instanceof MasterSlaveUnitUnavailableException
                || findFirstMatchingNestedException(e, PSQLException.class).isPresent()) {
            return HttpStatus.SC_503_SERVICE_UNAVAILABLE;
        } else if (e instanceof CommandRunException) {
            val cre = (CommandRunException) e;
            if (cre.isSituation(Situation.TOO_LONG_EVENT)
                    || cre.isSituation(Situation.EVENT_LONGER_THAN_REP)
                    || cre.isSituation(Situation.ICS_PARSING_ERROR)
                    || cre.isSituation(Situation.BUSY_OVERLAP)) {
                return HttpStatus.SC_400_BAD_REQUEST;
            }
        }
        return HttpStatus.SC_500_INTERNAL_SERVER_ERROR;
    }

    @Override
    @SneakyThrows
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) {
        val handle = CalendarRequest.push(ActionSource.MAILHOOK, "mailhook");
        val headerReader = HeaderUtils.getHeaderReader(req);
        val ip = AddrUtils.getIp(headerReader);
        try {
            tvmManager.handleServiceTicket(headerReader, ActionSource.MAILHOOK);
            getMailhookService().execute(req, resp);
            webApiMonitoring.reportSuccessExecution(req.getRequestURI());
        } catch (Exception e) {
            if (e instanceof UnauthenticatedException || e instanceof UnauthorizedException) {
                log.error("Auth exception for IP {}", ip.orElse("UNKNOWN"), e);
            }
            log.error("Error occurred during import", e);
            ewsMonitoring.reportIfEwsException(e);
            webApiMonitoring.reportErrorExecution(req.getRequestURI(), e);
            resp.sendError(getStatusCode(e), e.getMessage());
        } finally {
            handle.popSafely();
        }
    }
}
