package ru.yandex.calendar.frontend.web.cmd;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.RepetitionFields;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.Completion;
import ru.yandex.calendar.logic.event.model.Priority;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.notification.Channel;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationRoutines;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.util.base.AuxBase;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.color.ColorUtils;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.validation.Captcha;
import ru.yandex.calendar.util.validation.Captcha.CaptchaData;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.misc.lang.StringUtils;

public class RequestDataConverter {

    public static EventUser convertEventUser(DataProvider uDp) {
        if (uDp == null) { return new EventUser(); }

        String availStr = uDp.getText(dpName(EventUserFields.AVAILABILITY), false);
        String completionStr = uDp.getText(dpName(EventUserFields.COMPLETION), false);
        String priorityStr = uDp.getText(dpName(EventUserFields.PRIORITY), false);
        String decisionStr = uDp.getText(dpName(EventUserFields.DECISION), false);

        EventUser eventUser = new EventUser();
        if (StringUtils.isNotEmpty(availStr)) { eventUser.setAvailability(Availability.R.fromValue(availStr)); }
        if (StringUtils.isNotEmpty(completionStr)) { eventUser.setCompletion(Completion.R.fromValue(completionStr)); }
        if (StringUtils.isNotEmpty(priorityStr)) { eventUser.setPriority(Priority.R.fromValue(priorityStr)); }
        if (StringUtils.isNotEmpty(decisionStr)) { eventUser.setDecision(Decision.R.valueOf(decisionStr)); }

        return eventUser;
    }

    public static ListF<Notification> convertNotifications(DataProvider nDp) {
        if (nDp == null) { return Option.empty(); }

        String ns = nDp.getText("s", false);

        if (ns != null) {
            return Cf.x(ns.split(";")).filterNot(String::isEmpty).map(s -> new Notification(
                    Channel.R.fromValue(StringUtils.substringBefore(s, ",")),
                    Duration.standardMinutes(Long.parseLong(StringUtils.substringAfter(s, ",")))));
        }

        String offsetsMinStr = nDp.getText("offsets-min", true);
        ListF<Duration> offsets = NotificationRoutines.splitOffsets(offsetsMinStr).map(AuxDateTime.minutesToDurationF());

        ListF<Notification> notifications = Cf.arrayList();

        if (Binary.parseBoolean(nDp.getText("is-email-notify", false))) {
            notifications.addAll(offsets.map(Notification::email));
        }
        if (Binary.parseBoolean(nDp.getText("is-sms-notify", false))) {
            notifications.addAll(offsets.map(Notification::sms));
        }
        if (Binary.parseBoolean(nDp.getText("is-svc-notify", false))) {
            notifications.addAll(offsets.map(Notification::svc));
        }
        if (Binary.parseBoolean(nDp.getText("is-ical-display-notify", false))) {
            notifications.addAll(offsets.map(Notification::display));
        }
        return notifications;
    }

    public static Repetition convertRepetition(DateTimeZone tz, DataProvider rDp, Instant eventStart) {
        if (rDp == null) { return RepetitionRoutines.createNoneRepetition(); }
        Repetition repetition = new Repetition();
        String dueTsStr = rDp.getText(dpName(RepetitionFields.DUE_TS), false);
        // ssytnik@: IcsEventDataConverter, ExchangeEventDataConverter use EventRoutines.calculateDueTs.
        // Here, we don't (need not +1 day for isAllDay case), unless we change format of web data.
        Option<Instant> dueTs = DateTimeFormatter.parseTimestampFromMachinesSafe(dueTsStr, tz);
        if (dueTs.isPresent()) {
            dueTs = Option.of(DateTimeFormatter.equalizeLtMsSafe(eventStart, dueTs.get(), 1, tz));
        }
        repetition.setDueTs(dueTs);
        RegularRepetitionRule rule = RegularRepetitionRule.R.fromValue(rDp.getText(dpName(RepetitionFields.TYPE), true));
        repetition.setType(rule);

        String rEachStr = rDp.getText(dpName(RepetitionFields.R_EACH), false);
        repetition.setREach(RepetitionUtils.calcREach(AuxBase.toNullableIntUnsafe(rEachStr)));
        if (rule == RegularRepetitionRule.WEEKLY) {
            String rWeeklyDaysStr = rDp.getText(dpName(RepetitionFields.R_WEEKLY_DAYS), false);
            if (rWeeklyDaysStr != null) {
                ListF<DayOfWeek> ds = Cf.x(rWeeklyDaysStr.split(",")).map(DayOfWeek.byNameF()).unique().sorted();
                rWeeklyDaysStr = ds.map(DayOfWeek.getDbValueF()).mkString(",");
            }
            repetition.setRWeeklyDays(rWeeklyDaysStr);
        }
        if (rule == RegularRepetitionRule.MONTHLY_DAY_WEEKNO) {
            String rMonthlyLastweek = rDp.getText(dpName(RepetitionFields.R_MONTHLY_LASTWEEK), false);
            repetition.setRMonthlyLastweek(Binary.parseBoolean(rMonthlyLastweek));
        }
        return repetition;
    }

    public static CaptchaData convertCaptchaData(DataProvider dp) {
        // XXX: is key allowed to be empty here?
        String paramKey = dp.getText(Captcha.PARAM_KEY, false);
        String userResponse = dp.getText(Captcha.PARAM_USER_RESPONSE, false);
        return new CaptchaData(paramKey, userResponse);
    }

    public static TodoList convertTodoList(DataProvider dp) {
        TodoList todoList = new TodoList();
        String todoListId = dp.getText("todo-list-id", false);
        if (StringUtils.isNotEmpty(todoListId)) {
            todoList.setId(Long.parseLong(todoListId));
        }
        String title = dp.getText(dpName(TodoListFields.TITLE), false);
        if (StringUtils.isNotEmpty(title)) {
            todoList.setTitle(title);
        }
        String description = StringUtils.defaultIfEmpty(dp.getText(dpName(TodoListFields.DESCRIPTION), false), "");
        todoList.setDescription(description);
        todoList.setExternalId(CalendarUtils.generateExternalId());
        return todoList;
    }

    // XXX ssytnik: maybe, link to convertTodoList
    public static Option<Long> convertEventId(DataProvider dp) {
        return Cf.Long.parseSafe(dp.getText("event-id", false));
    }

    public static TodoItem convertTodoItem(Option<DateTimeZone> tz, DataProvider dp) {
        TodoItem todoItem = new TodoItem();
        String id = dp.getText(dpName(TodoItemFields.ID), false); // NOTE: ssytnik: not 'todo-item-id', but 'id'
        if (StringUtils.isNotEmpty(id)) {
            todoItem.setId(Long.parseLong(id));
        }
        String todoListId = dp.getText(dpName(TodoItemFields.TODO_LIST_ID), false);
        if (StringUtils.isNotEmpty(todoListId)) {
            todoItem.setTodoListId(Long.parseLong(todoListId));
        }
        String title = dp.getText(dpName(TodoItemFields.TITLE), false);
        if (StringUtils.isNotEmpty(title)) {
            todoItem.setTitle(title);
        }
        if (tz.isPresent()) {
            String dueTsStr = dp.getText(dpName(TodoItemFields.DUE_TS), false);
            todoItem.setDueTs(DateTimeFormatter.toNullableTimestampUnsafe(dueTsStr, tz.get()));
        }
        String colorStr = dp.getText(dpName(TodoItemFields.COLOR), false);
        if (colorStr == null) {
            todoItem.setColorNull();
        } else {
            todoItem.setColor(ColorUtils.unformatColor(colorStr));
        }
        return todoItem;
    }

    // XXX ssytnik: maybe, link to convertTodoItem
    public static ListF<Long> convertTodoItemIds(DataProvider dp) {
      return AuxColl.splitToLongArray(dp.getText("ids", false));
    }

    // XXX move to common converter class
    public static String dpName(MapField<?> field) {
        return field.column().name().replace("_", "-");
    }
}
