package ru.yandex.calendar.frontend.web.cmd.generic;

import lombok.val;
import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.notification.EventNotifications;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.NotificationXmlizer;
import ru.yandex.calendar.logic.sharing.InvitationXmlizer;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.PermXmlizer;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Base class for get-event commands that obtain detailed information
 * about single event instance starting at a given date (day) (optional),
 * at the specified format
 * @author ssytnik
 */
public abstract class CmdGetEventBase extends ValidatableXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdGetEventBase.class);

    protected String eventTsStr; // for different validations
    private Option<Integer> sequence = Option.empty();

    @Autowired
    protected EventRoutines eventRoutines;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    protected EventInvitationManager eventInvitationManager;
    @Autowired
    protected Authorizer authorizer;
    @Autowired
    private PermXmlizer permXmlizer;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    // Authorized
    public CmdGetEventBase(
            String cmdTagName, AuthInfo ai,
            String eventTsStr, Option<Integer> sequence, String isForShowStr)
    {
        this(cmdTagName, ai, eventTsStr, sequence, isForShowStr, null);
    }
    // Authorized
    public CmdGetEventBase(
            String cmdTagName, AuthInfo ai, String eventTsStr, Option<Integer> sequence,
            String isForShowStr, String tzId)
    {
        super(cmdTagName, ai, tzId);
        init(eventTsStr, sequence, isForShowStr);
    }
    // Anonymous
    public CmdGetEventBase(
            String cmdTagName, String eventTsStr, Option<Integer> sequence,
            String isForShowStr, String tzId)
    {
        super(cmdTagName, tzId);
        init(eventTsStr, sequence, isForShowStr);
    }

    private void init(String eventTsStr, Option<Integer> sequence, String isForShowStr) {
        this.eventTsStr = StringUtils.defaultIfEmpty(eventTsStr, null); // just better logging
        this.sequence = sequence;
    }

    protected final Element buildGetEventXmlResponse( // returns created 'eEvent' element
            XmlCmdContext ctx, Option<PassportUid> uid2O,
            Option<Long> layerIdO, long eventId, Option<String> linkSignKey)
    {

        final Element eRoot = ctx.getRootElement();

        Instant eventTs = DateTimeFormatter.toNullableTimestampUnsafe(eventTsStr, DateTimeZone.UTC);
        // Add general info to root element
        CalendarXmlizer.setAttr(eRoot, "uid2", uid2O.map(PassportUid::getUid).getOrNull());
        CalendarXmlizer.setAttr(eRoot, "layer-id", layerIdO.getOrNull());
        CalendarXmlizer.setAttr(eRoot, "event-id", eventId);
        CalendarXmlizer.setAttr(eRoot, "event-ts", eventTsStr);
        logger.debug(
            "uid2: " + uid2O + ", layerId: " + layerIdO + ", " +
            "event-id: " + eventId + ", event-ts: " + eventTsStr
        );
        // Get event instance, then check permission. In that order, because
        // mainly everything is ok, so that ER.getSingleInstance's work
        // will be useful, and permission check will be very quick then.
        // 'ei' is not null here.
        final Option<Instant> startMs; // start of test interval
        if (eventTs == null) {
            startMs = Option.<Instant>empty();
        } else {
            startMs = Option.of(new Instant(eventTs.getMillis()));
        }
        EventInstanceInfo eventInstanceInfo = eventWebManager.getEvent(
                uid2O, eventId, startMs, sequence, getActionInfo());

        val user2InfoO = obtainUserInfoO(uid2O);
        val eventWithRelations = eventInstanceInfo.getEventWithRelations();

        user2InfoO.toOptional()
            .ifPresentOrElse(
                user -> {
                    val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, eventWithRelations);
                    authorizer.ensureCanViewEvent(user, eventAuthInfo, ActionSource.WEB);
                },
                () -> {
                    val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(eventWithRelations);
                    authorizer.ensureCanViewEvent(eventAuthInfo, ActionSource.WEB);
                }
            );
        val layer = layerIdO.filterMap(eventInstanceInfo.getEventWithRelations()::findLayerById);
        Element eEvent = eventRoutines.getElement(
                user2InfoO, eventInstanceInfo, layer, Option.empty(), Option.empty(), tz, linkSignKey, ActionSource.WEB);

        Repetition repetition = eventInstanceInfo.getRepInstInfo().getRepetition().getOrNull();
        Participants invs = eventInstanceInfo.getEventWithRelations().getParticipants();

        boolean outResourceInfo = passportAuthDomainsHolder.containsYandexTeamRu();
        eEvent.addContent(InvitationXmlizer.serializeParticipants(invs, uidO, outResourceInfo));

        // Layer - additional info (permissions)
        if (layerIdO.isPresent()) {
            Element eLayer = eEvent.getChild("layer"); // must exist because is always created by ER.getElm()
            permXmlizer.appendLayerPermElms(eLayer, user2InfoO, layerIdO.get(), ActionSource.WEB, LayerAction.DETACH_EVENT);
        }
        Option<EventNotifications> notifications = eventInstanceInfo
                .getEventUserWithNotifications()
                .map(EventUserWithNotifications.getNotificationsF());
        // Event user - additional info (notification)
        if (notifications.isPresent()) {
            // tag must exist because a) we do not care of 'can-view' (though, anyway,
            // it is 'true' now): event-user is always seen, and b) event-user exists
            Element eUser = eEvent.getChild("user");
            NotificationXmlizer.appendElmForWeb(eUser, notifications.get().getNotifications());
        }
        // Repetition
        RepetitionRoutines.addElms(eEvent, eventInstanceInfo.getInterval().getStartMillis(), repetition, tz);

        eRoot.addContent(eEvent);
        return eEvent;
    }

} //~
