package ru.yandex.calendar.frontend.web.cmd.generic;

import javax.annotation.PostConstruct;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jdom.Document;
import org.jdom.Element;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.calendar.CalendarDataSourceStatus;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarVersion;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.ParameterValidationException;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.stat.CountAndTimeStats;
import ru.yandex.calendar.logic.stat.ExceptionCountStats;
import ru.yandex.calendar.monitoring.WebApiMonitoring;
import ru.yandex.calendar.tvm.exceptions.TvmUnauthenticatedException;
import ru.yandex.calendar.tvm.exceptions.TvmUnauthorizedException;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.net.CalendarHostnameUtils;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.web.action.minispring.ActionPrepare;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.misc.xml.jdom.JdomUtils;
import ru.yandex.misc.xml.support.XmlWriteFormat;

/**
 * Runs given command. If unexpected error occurs, it creates servant-failed tagged document
 * with a command-run-exception tag within which explains occurred error.
 */
@Slf4j
public class CommandExecutor implements ApplicationContextAware {
    private ApplicationContext applicationContext;
    @Value("${dump.response:-false}")
    private boolean dumpResponse;

    private static CountAndTimeStats methodCallStats = null;
    private static ExceptionCountStats exceptionCountStats = null;

    @Autowired
    private WebApiMonitoring webApiMonitoring;
    @Autowired
    private CalendarDataSourceStatus dataSourceStatus;

    protected static ListF<ClassX<?>> findActionClasses() {
        return
            ClassFinder.findClasses("ru.yandex.calendar.frontend.web.cmd.**.Cmd*")
                .filter(ClassX.wrap(Command.class)::isAssignableFrom)
            ;
    }

    private static class ActionClassesHolder {
        static ListF<ClassX<?>> actionClasses = findActionClasses();
    }

    private MapF<ClassX<?>, ActionPrepare> actionPreparers;

    @PostConstruct
    public void init() {
        actionPreparers = ActionClassesHolder.actionClasses.toMapMappingToValue(
                actionClazz -> new ActionPrepare(actionClazz, applicationContext));
    }

    public Document executeCommandAsXml(String method, XmlCommand cmd) {
        return executeCommandAsXml(method, cmd, ActionSource.WEB);
    }

    // If exception occurs, logs its stack trace no matter of what type it is
    public Document executeCommandAsXml(String method, XmlCommand cmd, ActionSource actionSource) {
        val handle = MasterSlaveContextHolder.push(cmd.getMasterSlavePolicy());
        val tltHandle = ThreadLocalTimeout.push(cmd.getTimeout(), "action " + cmd.getActionName());
        val sdcHandle = Ndc.push("a=" + cmd.getActionName());
        // XXX add other action sources - command executor can be used not only by web, e.g. by cron
        val requestHandle = CalendarRequest.push(
                cmd.getRemoteInfo(), actionSource, "action " + cmd.getActionName(), cmd.getActionName());
        try {
            actionPreparers.getOrThrow(ClassX.getClass(cmd)).prepare(cmd);
            log.info(LogMarker.ACTION.format(cmd.getActionName()));
            log.info("Started '" + cmd.getTagName() + "' @ '" + method + "'" + cmd.getLogTail());
            Document resDoc = null;
            Throwable throwable = null;
            boolean isAdmissible = false;
            String status;
            val startMs = AuxDateTime.NOW();
            try {
                resDoc = cmd.execute();
                status = "succeeded";
            } catch (CommandRunException e) {
                throwable = e;
                val situationO = e.getSituation();
                isAdmissible = e.isAdmissibleError() || situationO.isPresent();
                status =
                    (isAdmissible ? "admissible error" : "command run exception") +
                    (situationO.isPresent() ? " (situation: " + situationO.get() + ")" : "") +
                    " in";
            } catch (ParameterValidationException e) {
                throwable = e;
                isAdmissible = true;
                status = "parameter validation failed in";
            } catch (TvmUnauthorizedException | TvmUnauthenticatedException e) {
                throwable = e;
                isAdmissible = true;
                status = "tvm exception in";
            } catch (PermissionDeniedUserException e) {
                throwable = e;
                isAdmissible = true;
                status = "permission denied in";
            } catch (Throwable t) {
                throwable = t;
                status = "unexpected throwable in";
            }
            if (throwable != null) {
                addThrowableToStats(throwable); // NOTE: we include admissible errors
                resDoc = CommandErrors.createServantFailedDoc(cmd, throwable);
            }
            val totalMs = AuxDateTime.NOW() - startMs;
            val totalMsStr = TimeUtils.millisecondsToSecondsString(totalMs);
            val msg = status + " '" + cmd.getTagName() + "', " + "command took " + totalMsStr;

            if (throwable == null) {
                log.info(msg);
            } else if (isAdmissible) {
                log.warn(msg, throwable);
            } else {
                log.error(msg, throwable);
            }

            if (methodCallStats != null) {
                methodCallStats.add(method, totalMs);
            }
            if (throwable == null || isAdmissible) {
                webApiMonitoring.reportSuccessExecution(method);
            } else {
                webApiMonitoring.reportErrorExecution(method, throwable);
            }

            setCommonAttributes(resDoc.getRootElement(), method, cmd, totalMs);

            if (dataSourceStatus.isMasterUnavailable()) {
                CalendarXmlizer.setAttr(resDoc.getRootElement(), "master-db-unavailable", true);
            }

            if (dumpResponse) {
                val resFormatted = JdomUtils.I.writeDocumentToString(resDoc, XmlWriteFormat.compactFormat());
                log.debug("Response XML is: {}", resFormatted);
            }
            return resDoc;
        } catch (RuntimeException e) {
            log.error("Unexpected exception in action executing routines", e);
            throw e;
        } finally {
            requestHandle.popSafely();
            sdcHandle.popSafely();
            tltHandle.popSafely();
            handle.popSafely();
        }
    }

    private void addThrowableToStats(Throwable t) {
        if (exceptionCountStats != null) {
            if (t.getCause() != null) {
                addThrowableToStats(t.getCause());
            }
            exceptionCountStats.add(t.getClass().getName(), null);
        }
    }

    private void setCommonAttributes(Element eRoot, String method, XmlCommand cmd, long totalMs) {
        // http://wiki.yandex-team.ru/j/VerstkaXml
        CalendarXmlizer.setAttr(eRoot, "exec-duration", TimeUtils.millisecondsToSecondsString(totalMs));
        CalendarXmlizer.setAttr(eRoot, "hostname", HostnameUtils.localHostname());
        CalendarXmlizer.setAttr(eRoot, "host-id", CalendarHostnameUtils.getLocalhostId().getOrElse(""));
        CalendarXmlizer.setAttr(eRoot, "version", CalendarVersion.VERSION.toString());
        CalendarXmlizer.setAttr(eRoot, "method", method);
        CalendarXmlizer.setAttr(eRoot, "action", cmd.getActionName());
        if (RequestIdStack.current().isPresent()) {
            CalendarXmlizer.setAttr(eRoot, "req-id", RequestIdStack.current().get());
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
