package ru.yandex.calendar.frontend.web.cmd.generic;

import Yandex.RequestPackage.RequestData;
import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.RemoteInfo;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.run.ParameterValidationException;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.Environment;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.NoDateTimeZone;

/**
 * Xml command that
 * <ol>
 * <li>handles authentication/authorization for the user who asks an information;</li>
 * <li>creates a date time formatter with appropriate time zone</li>
 * </ol>
 * @author ssytnik
 */
public abstract class UserXmlCommand extends XmlCommand {
    private static final String EMPTY_PTK = "_EMPTY_"; // this value is not possible
    private static final AuthInfo NO_AUTH_INFO = null;
    private static final String NO_TZ_ID = null;

    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    protected UserManager userManager;

    protected final Option<AuthInfo> authInfo;

    /**
     * Auth uid (user who asks for an information)
     */
    protected final Option<PassportUid> uidO;

    protected Option<UserInfo> userInfoO;
    /**
     * Depends on constructor
     */
    private boolean mustHaveAuth;
    /**
     * Optional time zone id
     */
    private final String tzId;
    /**
     * Optional private token that can specify a resource
     * to be accessed without access rights check
     */
    private String privateToken;

    protected DateTimeZone tz;

//    /**
//     * Provides information for ANONYMOUS USER without predefined timezone
//     * (ssytnik: (?) in this case, {@link #getTzData()}} must be overridden)
//     * @param cmdTagName command tag name
//     * @param conf configuration
//     * @param requestData request parameters
//     */
//    public UserXmlCommand(String cmdTagName, RequestData requestData) {
//        this(cmdTagName, requestData, NO_AUTH_INFO, NO_TZ_ID, false);
//    }
//    /** Shortcut for {@link #UserXmlCommand(String, RequestData)} */
//    public UserXmlCommand(String cmdTagName) {
//        this(cmdTagName, NO_REQUEST_DATA);
//    }

    /**
     * Provides information for ANONYMOUS USER.
     * Inheritor command should collect resource
     * and check if user has access rights to it
     * @param cmdTagName command tag name
     * @param requestData request parameters
     * @param tzId timezone id to which we must convert dates and times
     */
    public UserXmlCommand(String cmdTagName, RequestData requestData, String tzId) {
        this(cmdTagName, requestData, NO_AUTH_INFO, tzId, false);
    }
    /** Shortcut for {@link #UserXmlCommand(String, RequestData, String)} */
    public UserXmlCommand(String cmdTagName, String tzId) {
        this(cmdTagName, NO_REQUEST_DATA, tzId);
    }

    public UserXmlCommand(String cmdTagName) {
        this(cmdTagName, NO_TZ_ID);
    }

    /**
     * Provides information for AUTHORIZED USER and specified timezone.
     * Inheritor command should collect resource
     * and check if user has access rights to it.
     * @param tzId timezone id to which we must convert dates and times
     */
    public UserXmlCommand(String cmdTagName, RequestData requestData, AuthInfo ai, String tzId) {
        this(cmdTagName, requestData, ai, tzId, true);
    }
    /** Shortcut for {@link #UserXmlCommand(String, RequestData, AuthInfo, String)} */
    public UserXmlCommand(String cmdTagName, AuthInfo ai, String tzId) {
        this(cmdTagName, NO_REQUEST_DATA, ai, tzId);
    }

    /**
     * Provides information for AUTHORIZED USER.
     * Inheritor command should collect resource
     * and check if user has access rights to it.
     * Behaves exactly like ctor with 'ai', and 'tzId' == null.
     */
    public UserXmlCommand(String cmdTagName, RequestData requestData, AuthInfo ai) {
        this(cmdTagName, requestData, ai, NO_TZ_ID, true);
    }
    /** Shortcut for {@link #UserXmlCommand(String, RequestData, AuthInfo)} */
    public UserXmlCommand(String cmdTagName, AuthInfo ai) {
        this(cmdTagName, NO_REQUEST_DATA, ai);
    }


    protected final Option<UserInfo> obtainUserInfoO(Option<PassportUid> uidO) {
        return userManager.getUserInfos(uidO).singleO();
    }

    /** Common ctor **/
    private UserXmlCommand(String cmdTagName, RequestData requestData, AuthInfo ai, String tzId, boolean mustHaveAuth) {
        super(cmdTagName, requestData);
        this.authInfo = Option.ofNullable(ai);
        this.uidO = authInfo.map(info -> info.uid).filterMap(PassportUidOrZero::toPassportUidO);
        this.tzId = tzId;
        this.mustHaveAuth = mustHaveAuth;
        this.privateToken = null;
    }

    protected Option<DateTimeZone> getTimezone() {
        if (StringUtils.isNotEmpty(tzId)) {
            return Option.of(AuxDateTime.getVerifyDateTimeZoneSafe(tzId)
                    .getOrThrow(() -> new ParameterValidationException("Timezone is not recognized: " + tzId)));
        } else if (uidO.isPresent()) {
            return Option.of(dateTimeManager.getTimeZoneForUid(uidO.get()));
        } else {
            return Option.empty();
        }
    }

    /**
     * Expects non-empty private token in order to obtain a private resource
     * @param privateToken private token that specifies resource. Must be set.
     */
    protected void setPrivateToken(String privateToken) {
        // This is called from the ctor, and we should not throw exceptions here
        //if (!AuxBase.isSet(privateToken)) { throw new IllegalArgumentException("privateToken"); }
        this.privateToken = (StringUtils.isNotEmpty(privateToken) ? privateToken : EMPTY_PTK);
    }

    protected boolean isPrivateTokenSet() {
        return StringUtils.isNotEmpty(privateToken);
    }

    protected void setMustHaveAuth(boolean mustHaveAuth) {
        this.mustHaveAuth = mustHaveAuth;
    }

    @Override
    public RemoteInfo getRemoteInfo() {
        return authInfo.map(ai -> new RemoteInfo(ai.ip, ai.yandexuid)).getOrElse(super::getRemoteInfo);
    }

    @Override
    protected String getLogTail() {
        return (
            ", auth-uid = " + uidO +
            (StringUtils.isNotEmpty(tzId) ? ", tz-id = " + tzId : "")
        );
    }

    @Override
    protected final void buildXmlResponse(XmlCmdContext ctx) {
        if (mustHaveAuth && !uidO.isPresent()) {
            Element responseElement = new Element("response");
            CalendarXmlizer.setAttr(responseElement, "no-auth", "yes");
            ctx.getRootElement().addContent(responseElement);
            return;
        }

        // Note: "tz-id" attribute can be added here, if needed for makeup
        // We log auth-uid, timezone in CommandUtils already so we save a line here

        if (isPrivateTokenSet()) { obtainPrivateResource(privateToken); }
        // NOW: we check everything we need in command logic, except isPrivateTokenSet() cases

        tz = getTimezone().getOrElse(() -> Environment.isProductionOrPre() ? DateTimeZone.UTC : new NoDateTimeZone());

        userInfoO = obtainUserInfoO(uidO);
        buildXmlResponseU(ctx);
    }

    // CAN BE OVERRIDDEN //

    // Must be overridden if private token is given
    // (to obtain resource with given private token)
    protected void obtainPrivateResource(String privateToken) {
        throw new UnsupportedOperationException();
    }

    // ABSTRACT //

    protected abstract void buildXmlResponseU(XmlCmdContext ctx);
}
