package ru.yandex.calendar.frontend.web.cmd.generic;

import javax.annotation.Nullable;

import Yandex.RequestPackage.RequestData;
import org.jdom.Document;
import org.jdom.Element;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.TransactionCallback;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.RemoteInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.util.dates.Timeouts;
import ru.yandex.calendar.util.idlent.RequestWrapper;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.MasterSlaveUnitUnavailableException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.reflection.ClassX;

/**
 * Basic command that returns an xml response
 * @author ssytnik
 */
public abstract class XmlCommand implements Command {
    private static final Logger logger = LoggerFactory.getLogger(XmlCommand.class);
    protected static final RequestData NO_REQUEST_DATA = null;
    private final String tagName;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Nullable
    private final RequestData requestData;
    @Nullable
    protected final RequestWrapper rw;
    /**
     * if true, then transaction will be used. Can be set if:
     * - multiple batch update operations are performed;
     * - several command runs from separate threads can.
     * work bad due to incorrect data from slave db-s
     * (however, if so, force r/w mode can be used)
     */
    private boolean requiresDbTransaction;

    private ActionInfo actionInfo = null; // lazy


    public XmlCommand(String tagName) {
        this(tagName, NO_REQUEST_DATA);
    }

    public XmlCommand(String tagName, RequestData requestData) {
        this.tagName = tagName;
        this.requestData = requestData;
        this.rw = requestData != null ? new RequestWrapper(requestData) : null;

        this.requiresDbTransaction = false;
    }

    // Puts given parameter to both log and xml
    // TODO Improve it! Because for now, for dtf params, UTC timezone is used
    protected void debugLogXml(Element rootElement, String logPrefix, String xmlAttrName, Object value) {
        logger.debug(logPrefix + value);
        CalendarXmlizer.setAttr(rootElement, xmlAttrName, String.valueOf(value));
    }
    protected void debugLogXml(XmlCmdContext ctx, String logPrefix, String xmlAttrName, Object value) {
        debugLogXml(ctx.getRootElement(), logPrefix, xmlAttrName, value);
    }

    public String getTagName() {
        return tagName;
    }

    public String getActionName() {
        return StringUtils.uncapitalize(this.getClass().getSimpleName());
    }

    protected void setRequiresDbTransaction(boolean requiresDbTransaction) {
        this.requiresDbTransaction = requiresDbTransaction;
    }

    /** for logging, e.g. @ {@link CommandExecutor} */
    protected String getLogTail() {
        return "";
    }

    /** don't forget to change timeout also in makeup */
    public Duration getTimeout() {
        return Timeouts.getForCommand();
    }

    protected ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }

    public RemoteInfo getRemoteInfo() {
        return new RemoteInfo(Option.empty(), Option.empty());
    }


    @Override
    public MasterSlavePolicy getMasterSlavePolicy() {
        String simpleName = ClassX.getClass(this).getSimpleName();
        if (simpleName.startsWith("CmdGet") || simpleName.startsWith("CmdSvcGet"))
            return MasterSlavePolicy.R_MS;
        else
            return MasterSlavePolicy.RW_M;
    }

    public Document execute() {
        // Create response doc and add common params
        Element rootElement = new Element(tagName);
        Document resDoc = new Document(rootElement);

        if (requestData != null) {
            RequestWrapper rw = new RequestWrapper(requestData);
            rw.logRequest();
            // XXX: according to leonya, verstka depends on this behaviour
            rw.addXmlElementTo(rootElement);
        }
        final XmlCmdContext ctx = new XmlCmdContext(rootElement);
        //logger.debug(EXECUTE_PREFIX + ": pre-init complete");

        try {
            boolean runCommand = true;
            if (runCommand) {
                if (requiresDbTransaction) {
                    transactionTemplate.execute(new TransactionCallback<Object>() {
                        public Object doInTransaction(TransactionStatus status) {
                            buildXmlResponse(ctx);
                            return null;
                        }
                    });
                } else {

                    buildXmlResponse(ctx);

                }
            }

        } catch (MasterSlaveUnitUnavailableException msuue) {
            if (msuue.getMasterSlavePolicy().equals(MasterSlavePolicy.RW_M)) {
                throw CommandRunException.createSituation(msuue, Situation.MASTER_UNAVAILABLE);
            } else {
                throw msuue;
            }
        }

        return resDoc;
    }


    protected abstract void buildXmlResponse(XmlCmdContext ctx);
}
