package ru.yandex.calendar.frontend.web.cmd.run.api;

import Yandex.Tag;
import Yandex.TagHolder;
import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Hours;
import org.joda.time.Instant;
import org.joda.time.Minutes;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarDataSourceStatus;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.Staff;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class CmdCheckUpdateActiveOffice extends UserXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdCheckUpdateActiveOffice.class);

    private static final String CMD_TAG = "check-update-active-office";

    private static final Duration NOT_FOUND_CACHE_DURATION = Hours.THREE.toStandardDuration();
    private static final Duration OFFICE_CHANGED_CACHE_DURATION = Minutes.minutes(10).toStandardDuration();

    @Autowired
    private Staff staff;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private CalendarDataSourceStatus dataSourceStatus;

    private final String ipAddressStr;
    private final TagHolder tag;

    public CmdCheckUpdateActiveOffice(AuthInfo ai, String ipAddressStr, TagHolder tag) {
        super(CMD_TAG, ai);
        this.ipAddressStr = ipAddressStr;
        this.tag = tag;
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        val ipAddress = parseIpAddress();
        if (!ipAddress.isPresent()) {
            sendOk(ctx, NOT_FOUND_CACHE_DURATION);
            return;
        }
        if (dataSourceStatus.isMasterUnavailable()) {
            sendOk(ctx, Duration.standardMinutes(1));
            return;
        }
        Option<Integer> officeCenterId;
        try {
            officeCenterId = staff.getOfficeIdByIp(ipAddress.get());

        } catch (RuntimeException e) {
            logger.error("Failed to get office by ip", e);
            sendError(ctx);
            return;
        }
        val officeId = officeCenterId.filterMap(officeManager.getOfficeIdByCenterIdF());

        if (officeCenterId.isPresent() && !officeId.isPresent()) {
            logger.warn("Office not found by center id {}", officeCenterId.get());
        }
        if (!officeId.isPresent() && !uidO.exists(userManager::isYamoneyUser)) {
            sendOk(ctx, NOT_FOUND_CACHE_DURATION);
            return;
        }

        val settings = settingsRoutines.getSettingsByUid(uidO.get()).getYt().get();

        val data = new SettingsYt();
        data.setActiveOfficeId(officeManager.chooseActiveOfficeId(officeId, settings.getTableOfficeId(), uidO.get()));
        data.setActiveOfficeDetectTs(Instant.now());

        val handle = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
        try {
            settingsRoutines.updateSettingsYtByUid(data, uidO.get());
            sendOk(ctx, OFFICE_CHANGED_CACHE_DURATION);

        } catch (RuntimeException e) {
            logger.error("Failed to update active office id ", e);
            sendError(ctx);
        } finally {
            handle.popSafely();
        }
    }

    @Override
    public MasterSlavePolicy getMasterSlavePolicy() {
        return MasterSlavePolicy.R_MS;
    }

    private void sendOk(XmlCmdContext ctx, Duration cacheDuration) {
        Instant now = Instant.now();
        tag.value = new Tag();
        tag.value.lastModified = (int) (now.getMillis() / 1000);
        tag.value.expireTime = (int) (now.plus(cacheDuration).getMillis() / 1000);
        tag.value.notModified = false;

        CalendarXmlizer.appendElm(ctx.getRootElement(), "ok");
    }

    private void sendError(XmlCmdContext ctx) {
        tag.value = new Tag();
        CalendarXmlizer.appendElm(ctx.getRootElement(), "error");
    }

    private Option<IpAddress> parseIpAddress() {
        return IpAddress.parseIpAddressO(ipAddressStr).map(ip -> {
            if (!ip.isIpv4Address() && ip.toIpv6Address().isIpv4MappedAddress()) {
                return ip.toIpv6Address().toMappedIpv4Address();
            } else {
                return ip;
            }
        });

    }
}
