package ru.yandex.calendar.frontend.web.cmd.run.api;

import org.jdom.Element;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.XmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.ui.CmdGetHolidaysA;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.holiday.XmlDayInfoHandler;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.PassportUids;
import ru.yandex.misc.lang.StringUtils;

/**
 * Obtains all information about holidays and weekdays
 * @author akirakozov
 */
public class CmdGetHolidays extends XmlCommand {
    private static final String CMD_TAG = "get-holidays";

    private String startDateStr;
    private String endDateStr;
    private int countryId;
    private long uid;
    private String outModeStr;
    private String forYandexStr;
    private Option<String> whoAmIO;

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private OfficeManager officeManager;

    public CmdGetHolidays(String startDateStr, String endDateStr, int countryId, String outModeStr) {
        this(startDateStr, endDateStr, countryId, 0, outModeStr, Binary.FALSE_STR, Option.<String>empty());
    }

    public CmdGetHolidays(
            String startDateStr, String endDateStr,
            int countryId, long uid, String outModeStr,
            String forYandexStr, String whoAmI)
    {
        this(startDateStr, endDateStr, countryId, uid, outModeStr, forYandexStr, Option.of(whoAmI));
    }

    private CmdGetHolidays(
            String startDateStr, String endDateStr,
            int countryId, long uid, String outModeStr,
            String forYandexStr, Option<String> whoAmIO)
    {
        super(CMD_TAG);
        this.startDateStr = startDateStr;
        this.endDateStr = endDateStr;
        this.countryId = countryId;
        this.uid = uid;
        this.outModeStr = outModeStr;
        this.forYandexStr = forYandexStr;
        this.whoAmIO = whoAmIO;
    }

    @Override
    protected void buildXmlResponse(XmlCmdContext ctx) {
        Element rootElement = ctx.getRootElement();
        LocalDate endDate;
        LocalDate startDate = DateTimeFormatter.toNullableDateUnsafe(startDateStr);
        if (startDate == null) {
            throw new CommandRunException("Start date must be specified");
        }
        if (StringUtils.isNotEmpty(endDateStr)) {
            endDate = DateTimeFormatter.toNullableDateUnsafe(endDateStr);
        } else {
            endDateStr = startDateStr;
            endDate = startDate;
        }

        if (PassportUids.isValid(uid) && PassportUid.cons(uid).isYandexTeamRu()) {
            Option<SettingsYt> settingsYt = settingsRoutines.getSettingsByUid(PassportUid.cons(uid)).getYt();

            countryId = settingsYt.isPresent() && settingsYt.get().getTableOfficeId().isPresent()
                    ? officeManager.getCountryIdByOfficeId(settingsYt.get().getTableOfficeId().get())
                    : countryId;
        }

        OutputMode outMode = OutputMode.find(outModeStr);
        boolean forYandex = Binary.parseBoolean(forYandexStr);
        // Add input parameters
        CalendarXmlizer.setAttr(rootElement, "country-id", countryId);
        CalendarXmlizer.setAttr(rootElement, "start-date", startDateStr);
        CalendarXmlizer.setAttr(rootElement, "end-date", endDateStr);
        CalendarXmlizer.setAttr(rootElement, "out-mode", outModeStr);
        CalendarXmlizer.setAttr(rootElement, "for-yandex", forYandex);
        // Generate output
        XmlDayInfoHandler diHandler = new XmlDayInfoHandler(true, CmdGetHolidaysA.isOutLabelsMode(outMode), Option.empty());
        HolidayRoutines.processDates(startDate, endDate, countryId, outMode, forYandex, diHandler);
        rootElement.addContent(diHandler.getElement());
    }
}
