package ru.yandex.calendar.frontend.web.cmd.run.api;

import org.jdom.Element;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.notification.NotificationSendManager;
import ru.yandex.calendar.logic.notification.Reminder;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * Returns information about (at most) given number of nearest user notifications
 * @author ssytnik
 */
public class CmdGetNearestNotification extends ValidatableXmlCommand {
    private static final String CMD_TAG = "get-nearest-ntf";

    @Autowired
    private NotificationSendManager notificationSendManager;

    // start (NOW), end
    private final String startTsStr;
    private final String endTsStr;
    private final String limitStr;

    public CmdGetNearestNotification(AuthInfo ai, String startTsStr, String endTsStr, String limitStr) {
        super(CMD_TAG, ai);
        this.startTsStr = startTsStr;
        this.endTsStr = endTsStr;
        this.limitStr = limitStr;
    }

    @Override
    public void validate() {
        // start is actually required, end or limit are optional (but not both)
        RequestValidator.validateOptional(RequestValidator.TIMESTAMP, "startTs", startTsStr);
        RequestValidator.validateOptional(RequestValidator.EXT_TIMESTAMP, "endTs", endTsStr);
        RequestValidator.validateOptional(RequestValidator.NON_NEGATIVE, "limit", limitStr);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        // Defaults are: NOW .. +inf, 3 events, all layers.
        Instant startTs = DateTimeFormatter.toTimestamp(startTsStr, new Instant(), tz);
        Instant endTs = DateTimeFormatter.toNullableExtTimestampUnsafe(startTs, endTsStr, tz);
        int limit = Cf.Integer.parseSafe(limitStr).getOrElse(3);

        ListF<Reminder> reminders = notificationSendManager
                .getUserDisplayNotifications(uidO.get(), new InfiniteInterval(startTs, Option.ofNullable(endTs)), limit);

        // Generate XML output
        Element eRoot = ctx.getRootElement();
        CalendarXmlizer.setDtfAttr(eRoot, "start-ts", startTs, tz);
        CalendarXmlizer.setDtfAttr(eRoot, "end-ts", endTs, tz);
        CalendarXmlizer.setAttr(eRoot, "limit", limit);
        final long now = AuxDateTime.NOW(); // now snap ts

        for (Reminder reminder : reminders) {
            Element eReminder = new Element("reminder");
            CalendarXmlizer.setAttr(eReminder, "name", reminder.getEventName());
            CalendarXmlizer.setAttr(eReminder, "diff-ms", reminder.getNotifyTs().getMillis() - now);
            CalendarXmlizer.setDtfAttr(eReminder, "send-ts", reminder.getNotifyTs(), tz);
            CalendarXmlizer.setDtfAttr(eReminder, "start-ts", reminder.getEventInstanceStart(), tz);
            eRoot.addContent(eReminder);
        }
    }
}
