package ru.yandex.calendar.frontend.web.cmd.run.api;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import Yandex.Request;
import org.jdom.Element;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.ics.imp.IcsImportStats;
import ru.yandex.calendar.logic.ics.imp.LayerReference;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.Timeouts;
import ru.yandex.calendar.util.idlent.RequestWrapper;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * NOTE: maybe rename to importIcal
 * If url is set, command gets ics file contents from there and parse.
 * If not, then command uses given file contents.
 * As for layer id: it is optional EVEN if user has custom layers.
 * NOTE: ics files can be very big so that servant won't be able
 * to add all of them quickly (within one second)
 * @author ssytnik
 */
public class CmdIcsAddEvent extends UserXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdIcsAddEvent.class);

    private static final String CMD_TAG = "ics-add-event";
    private static final String MAIN_METHOD_NAME = "CmdIcsAddEvent.buildXmlResponseU";
    private static final Pattern FILENAME_PATTERN = Pattern.compile(
        "(?:^.*[\\\\/]\\??)?([^\\?\\.\\\\/#]+)" // supports russian resource (file) names
    );

    @Autowired
    private IcsFeedManager icsFeedManager;

    private Request request;
    private String contentsName;
    private String url;
    private Option<String> layerInfo;
    private String isFeedStr = null;

    private enum IcsSource { URL, REMOTE_FILE }

    // <valid input> ::= (conf & ai) & <ics source> & (layerInfo & isFeedStr)
    // <ics source>  ::= (request & contentsName) | url | (rawText & url==srcUrl)
    private CmdIcsAddEvent(
            AuthInfo ai,
            Request request, String contentsName, String url, String rawText,
            String layerInfo)
    {
        super(CMD_TAG, RequestWrapper.getRequestDataSilent(request), ai);

        // NOW: this was split into separate transactions (in IcsRoutines)
        //setRequiresDbTransaction(false);
        // Due to POST possibility, parameters will be initialized later
        this.request = request;
        this.contentsName = contentsName; // by name
        this.url = (url == null ? null : StringUtils.trim(url));
        this.layerInfo = Option.ofNullable(layerInfo);
    }

    // (directly called by 'Calendar.apiIcsAddEvent')
    public CmdIcsAddEvent(
            AuthInfo ai, Request request, String contentsName, String url,
            String layerInfo)
    {
        this(ai, request, contentsName, url, null, layerInfo);
    }

    // (directly called by 'Calendar.apiIcsAddEventFromMail')
    public CmdIcsAddEvent(AuthInfo ai, String rawText, String decisionStr) {
        this(ai, null, null, null, rawText, "DEFAULT");
    }

    // (directly called by 'Calendar.apiIcsAddEventByText')
    public CmdIcsAddEvent(
            AuthInfo ai,
            String srcUrl, String rawText,
            String layerInfo) {
        this(ai, null, null, srcUrl, rawText, layerInfo);
    }

    @Override
    public Duration getTimeout() {
        return Timeouts.getForCommand(Duration.standardMinutes(5));
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        IcsSource icsSource;
        byte[] contents = null;
        String layerNameSource;

        if (StringUtils.isNotEmpty(url)) {
            icsSource = IcsSource.URL;
            layerNameSource = UrlUtils.urlDecode(url);
        } else {
            contents = request == null ? null : RequestWrapper.getRemoteFile(request, contentsName);
            if (contents != null && contents.length > 0) {
                icsSource = IcsSource.REMOTE_FILE;
                layerNameSource = Option.ofNullable(RequestWrapper.getRemoteFilename(request, contentsName))
                        .getOrThrow("can not get filename");
            } else {
                final String msg = MAIN_METHOD_NAME + ": all ics sources are unavailable";
                throw new CommandRunException(msg);
            }
        }

        String contentsInfo = "{" + (contents == null ? 0 : contents.length) + " bytes}";
        String newLayerName = getLayerNameByFilenameOrUrlMagic(layerNameSource);
        boolean isFeed = Binary.parseBoolean(isFeedStr);
        // Debug
        Element rootElement = ctx.getRootElement();
        debugLogXml(rootElement, "chosen source = ", "chosen-source", icsSource);
        debugLogXml(rootElement, "url           = ", "url", url);
        debugLogXml(rootElement, "layer info    = ", "layer-info", layerInfo.getOrNull());
        debugLogXml(rootElement, "layer name src= ", "layer-name-src", layerNameSource);
        debugLogXml(rootElement, "layer name    = ", "layer-name", newLayerName);
        debugLogXml(rootElement, "contents info = ", "contents-info", contentsInfo);
        debugLogXml(rootElement, "is feed?      = ", "is-feed", isFeed);

        if (!isFeed) {
            LayerReference layerReference = LayerReference.parse(layerInfo.getOrElse(""));
            IcsImportStats result = icsSource == IcsSource.URL ?
                icsFeedManager.importContent(uidO.get(), url, newLayerName, layerReference) :
                icsFeedManager.importContent(uidO.get(), contents, newLayerName, layerReference);

            rootElement.addContent(result.createEventStatElem());
            rootElement.addContent(result.getTodoImportStats().createTodosStatElem());
        } else {
            icsFeedManager.subscribe(uidO.get(), url, newLayerName);
            CalendarXmlizer.appendElm(rootElement, "delayed", true);
        }
    }

    private String getLayerNameByFilenameOrUrlMagic(String source) {
        Matcher m = FILENAME_PATTERN.matcher(source); // check + fill 1st group
        if (!m.lookingAt()) {
            logger.warn("can not parse filename or url " + source);
            return "calendar";
        }
        return StringUtils.capitalize(m.group(1));
    }

    public void setIsFeed(String isFeedStr) { this.isFeedStr = isFeedStr; }
}
