package ru.yandex.calendar.frontend.web.cmd.run.api;

import org.jdom.Element;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.notification.ControlDataNotification;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;

/**
 * Deletes event by given external id and (implicitly) and event-user records.
 * Also deletes related entities.
 * @author ssytnik
 */
public class CmdSvcDelEventExtId extends UserXmlCommand {
    private static final String CMD_TAG = "service-delete-event-ext-id";

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventWebManager eventWebManager;

    private PassportSid sid;
    private Option<PassportUid> uid2O;
    private String eventExtId;
    private String applyToFutureEventsStr;

    public CmdSvcDelEventExtId(
            AuthInfo ai, PassportSid sid,
            String eventExtId, String applyToFutureEventsStr) {
        super(CMD_TAG, ai);
        init(sid, uidO, eventExtId, applyToFutureEventsStr);
    }
    // Ctor for 'apiSvcDelEvtExtIdCd'
    public CmdSvcDelEventExtId(
            AuthInfo ai, PassportSid sid,
            Option<PassportUid> uid2O, String controlData,
            String eventExtId, String applyToFutureEventsStr) {
        super(CMD_TAG, ai);
        init(sid, uid2O, eventExtId, applyToFutureEventsStr);
        setPrivateToken(controlData);
    }
    private void init(PassportSid sid, Option<PassportUid> uid2O, String eventExtId, String applyToFutureEventsStr) {
        this.sid = sid;
        this.uid2O = uid2O;
        this.eventExtId = StringUtils.defaultIfEmpty(eventExtId, null);
        this.applyToFutureEventsStr = applyToFutureEventsStr;
    }

    // OVERRIDES

    @Override
    protected void obtainPrivateResource(String privateToken) {
        // Here we need to validate uid2 using controlData (== privateToken)
        if (uid2O.isPresent()) {
            ControlDataNotification.ensureValid(sid, uid2O.get(), eventExtId, privateToken);
        } else {
            uid2O = uidO; // No uid2 => simply use auth-logic
        }
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        final PassportUid uid2 = uid2O.get();
        Long eId = eventRoutines.findMasterEventBySubjectIdAndExternalId(UidOrResourceId.user(uid2), eventExtId)
                .map(EventFields.ID.getF()).getOrNull();
        if (eId == null) {
            final String msg = "Event not found: sid = " + sid + ", ext. id = " + eventExtId;
            throw new CommandRunException(msg);
        }

        // Deleted event since its first instance
        ModificationInfo mi = eventWebManager.deleteEvent(
                userInfoO.get(), eId, Option.<Instant>empty(),
                Binary.parseBoolean(applyToFutureEventsStr), getActionInfo()
        );
        // Output
        final Element rootElement = ctx.getRootElement();
        CalendarXmlizer.setAttr(rootElement, "sid", sid.getValue());
        CalendarXmlizer.setAttr(rootElement, "event-ext-id", eventExtId);
        mi.appendXmlTo(rootElement, false); // we do not want split info
    } // buildXml


}
