package ru.yandex.calendar.frontend.web.cmd.run.api;

import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.CmdGetEventBase;
import ru.yandex.calendar.frontend.web.cmd.run.ui.event.CmdGetEvent;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.notification.ControlDataNotification;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * {@link CmdGetEvent}, but instead of <code>event_id</code> and
 * <code>layer_id</code>, it uses <code>sid</code> and <code>external_id</code>
 * @author ssytnik
 */
public class CmdSvcGetEvent extends CmdGetEventBase {
    private static final Logger logger = LoggerFactory.getLogger(CmdSvcGetEvent.class);

    private static final String CMD_TAG = "service-get-event";

    @Autowired
    private LayerDao layerDao;
    @Autowired
    private SettingsRoutines settingsRoutines;

    private PassportSid sid;
    private Option<PassportUid> uid2O;
    private String eventExtId;

    public CmdSvcGetEvent(
            AuthInfo ai,
            PassportSid sid, String eventExtId,
            String eventTsStr, String isForShowStr)
    {
        super(CMD_TAG, ai, eventTsStr, Option.<Integer>empty(), isForShowStr);
        init(sid, uidO, eventExtId);
    }
    /**
     * Ctor for 'apiSvcGetEventCd'
     * @param ai if optTzId is not specified
     * @param optTzId if ai's tzId needs to be overridden
     * @param sid identifies service
     * @param uid2O optional uid2 (to be cross-checked via controlData)
     * @param controlData service signature to verify uid2
     * @param eventExtId event external id
     * @param eventTsStr event timestamp
     * @param isForShowStr output format
     */
    public CmdSvcGetEvent(
            AuthInfo ai, String optTzId,
            PassportSid sid, Option<PassportUid> uid2O, String controlData, String eventExtId,
            String eventTsStr, String isForShowStr)
    {
        super(CMD_TAG, ai, eventTsStr, Option.<Integer>empty(), isForShowStr, optTzId);
        init(sid, uid2O, eventExtId);
        setPrivateToken(controlData);
    }
    private void init(PassportSid sid, Option<PassportUid> uid2O, String eventExtId) {
        this.sid = sid;
        this.uid2O = uid2O;
        this.eventExtId = StringUtils.defaultIfEmpty(eventExtId, null);
    }

    // OVERRIDES

    @Override
    protected void obtainPrivateResource(String privateToken) {
        // Here we need to validate uid2 using controlData (== privateToken)
        if (uid2O.isPresent()) {
            ControlDataNotification.ensureValid(sid, uid2O.get(), eventExtId, privateToken);
        } else {
            uid2O = uidO; // No uid2 => simply use auth-logic
        }
    }

    @Override
    public void validate() {
        RequestValidator.validateOptional(RequestValidator.TIMESTAMP, "eventTs", eventTsStr);
        RequestValidator.validateRequired(null, "eventExtId", eventExtId);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        final Element eRoot = ctx.getRootElement();
        final PassportUid uid2 = uid2O.get();
        Option<Long> layerId = layerDao.findServiceLayerIdByUidAndSid(uid2, sid);
        Option<Long> eventIdO;
        if (!layerId.isPresent()) {
            eventIdO = Option.empty();
        } else {
            eventIdO = eventRoutines.findMasterEventBySubjectIdAndExternalId(UidOrResourceId.user(uid2), eventExtId)
                .map(EventFields.ID.getF());
        }
        CalendarXmlizer.setAttr(eRoot, "sid", sid.getValue());
        CalendarXmlizer.setAttr(eRoot, "event-ext-id", eventExtId);
        logger.debug(("= sid: " + sid + ", eventExtId: " + eventExtId));
        if (!layerId.isPresent() || !eventIdO.isPresent()) {
            logger.debug(("Event or layer could not be found " +
            "(uid2: " + uid2 + ", layerId: " + layerId + ")"));
            Element eNoEvent = new Element("no-event");
            CalendarXmlizer.setAttr(eNoEvent, "uid2", uid2.getUid());
            CalendarXmlizer.setAttr(eNoEvent, "layer-id", layerId.getOrNull());
            // NEEDLESS: XmlUtils.setAttr(eNoEvent, "event-id", AuxBase.debugValue(eventId));
            eRoot.addContent(eNoEvent);
        } else {
            Element eEvent = buildGetEventXmlResponse(ctx, Option.of(uid2), layerId, eventIdO.get(), Option.empty());
            SettingsRoutines.appendCreatorUserLogin(eEvent, settingsRoutines.getSettingsByUid(uid2).getCommon());
        }
    }
}
