package ru.yandex.calendar.frontend.web.cmd.run.api;

import org.jdom.Element;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Second version of service-get-events
 * @author ssytnik
 */
public class CmdSvcGetEvents2 extends ValidatableXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdSvcGetEvents2.class);
    private static final String CMD_TAG = "service-get-events2";
    private static final String ANONYMOUS_ID_V2 = "Anonymous caller (v.2)";
    private static final String DEF_SHOW_LIMIT = null; // by default, show limit == limit

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private LayerRoutines layerRoutines;

    private String layerIdsStr;            // layers (ALL) - can be obtained via private token,
    private String startTsStr, endTsStr;   // start (NOW), end (+INF),
    private String limitStr, showLimitStr; // query limit (ALL), show limit (=query limit),
    private String onlyVisibleInUiStr;     // only visible in UI (FALSE), caller ID
                                           // (any string, required for method v3)
    private String whoAmI; // for diag only
    private Option<PassportUid> uid2O; // user (uid -> looks at -> uid2)
                                    // TODO: ssytnik: when working on CAL-1292
                                    // (private token: layer -> layer_user),
                                    // we'll need to modify 'ER.get', 'ER.getSortedInstances'
                                    // (uid, uid2; permissions checks apply to uid2, not uid),
                                    // and here: AuthInfo -> uid, Layer.creator_uid -> uid2.
    private final boolean showEvents; // set to false if you want to see count only

    // BEWARE: orders of parameters at 'this ctor' and 'init' do not match!
    public CmdSvcGetEvents2(
            AuthInfo ai, String optTzId, String layerIdsStr,
            String startTsStr, String endTsStr, String limitStr, String onlyVisibleInUiStr) {

        super(CMD_TAG, ai, optTzId);
        this.layerIdsStr = layerIdsStr;
        this.uid2O = uidO;
        this.startTsStr = startTsStr;
        this.endTsStr = endTsStr;
        this.limitStr = limitStr;
        this.showLimitStr = DEF_SHOW_LIMIT;
        this.onlyVisibleInUiStr = onlyVisibleInUiStr;
        this.whoAmI = ANONYMOUS_ID_V2;
        this.showEvents = true;
    }

    // BEWARE: orders of parameters at 'this ctor' and 'init' do not match!
    public CmdSvcGetEvents2(
            String tzId, String privateToken,
            String startTsStr, String endTsStr, String limitStr, String onlyVisibleInUiStr) {

        super(CMD_TAG, tzId);
        this.layerIdsStr = null;
        this.uid2O = Option.<PassportUid>empty();
        this.startTsStr = startTsStr;
        this.endTsStr = endTsStr;
        this.limitStr = limitStr;
        this.showLimitStr = DEF_SHOW_LIMIT;
        this.onlyVisibleInUiStr = onlyVisibleInUiStr;
        this.whoAmI = ANONYMOUS_ID_V2;
        setPrivateToken(privateToken);
        this.showEvents = true;
    }

    // idl method: apiSvcGetEvents3
    public CmdSvcGetEvents2(
            AuthInfo ai, String optTzId, String layerIdsStr,
            String startTsStr, String endTsStr, String limitStr, String showLimitStr,
            String onlyVisibleInUiStr, String whoAmI, boolean showEvents) {

        super(CMD_TAG, ai, optTzId);
        this.layerIdsStr = layerIdsStr;
        this.uid2O = uidO;
        this.startTsStr = startTsStr;
        this.endTsStr = endTsStr;
        this.limitStr = limitStr;
        this.showLimitStr = showLimitStr;
        this.onlyVisibleInUiStr = onlyVisibleInUiStr;
        this.whoAmI = whoAmI;
        this.showEvents = showEvents;
    }

    // idl method: apiSvcGetEvents3AP
    public CmdSvcGetEvents2(
            String tzId, String privateToken,
            String startTsStr, String endTsStr, String limitStr, String showLimitStr,
            String onlyVisibleInUiStr, String whoAmI) {

        super(CMD_TAG, tzId);
        this.layerIdsStr = null;
        this.uid2O = null;
        this.startTsStr = startTsStr;
        this.endTsStr = endTsStr;
        this.limitStr = limitStr;
        this.showLimitStr = showLimitStr;
        this.onlyVisibleInUiStr = onlyVisibleInUiStr;
        this.whoAmI = whoAmI;
        setPrivateToken(privateToken);
        this.showEvents = true;
    }

    @Override
    protected void obtainPrivateResource(String privateToken) {
        Layer l = layerRoutines.getByPrivateToken(privateToken);
        this.layerIdsStr = String.valueOf(l.getId());
        this.uid2O = Option.of(l.getCreatorUid()); // as if layer creator is looking over his own layer
    }

    @Override
    public void validate() {
        // start is, in fact, required, end or limit are optional (but not both)
        RequestValidator.validateOptional(RequestValidator.TIMESTAMP, "startTs", startTsStr);
        RequestValidator.validateOptional(RequestValidator.EXT_TIMESTAMP, "endTs", endTsStr);
        RequestValidator.validateOptional(RequestValidator.NON_NEGATIVE, "limit", limitStr);
        RequestValidator.validateOptional(RequestValidator.NON_NEGATIVE, "showLimit", showLimitStr);
        RequestValidator.validate(RequestValidator.LONG_ARRAY, "layerIds", layerIdsStr);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        // Defaults are: NOW .. +inf, 3 events, all layers.
        Instant startTs = DateTimeFormatter.toTimestamp(startTsStr, AuxDateTime.NOWTS(), tz);
        Instant endTs = DateTimeFormatter.toNullableExtTimestampUnsafe(startTs, endTsStr, tz);
        int limit = Cf.Integer.parseSafe(limitStr).getOrElse(3);
        int showLimit = Cf.Integer.parseSafe(showLimitStr).getOrElse(0);
        ListF<Long> layerIds = AuxColl.splitToLongArray(layerIdsStr);
        boolean onlyVisibleInUi = Binary.parseBoolean(onlyVisibleInUiStr);
        // Getting events / generate XML output
        Object endTsAttr = (endTs == null ? "+inf" : endTs);
        String layerIdsAttr = layerIds.toString();
        // Debug output
        Element rootElement = ctx.getRootElement();
        CalendarXmlizer.setAttr(rootElement, "layer-ids", layerIdsAttr);
        CalendarXmlizer.setAttr(rootElement, "uid2", uid2O.get().getUid());
        CalendarXmlizer.setDtfAttr(rootElement, "start-ts", startTs, tz);
        CalendarXmlizer.setDtfAttr(rootElement, "end-ts", endTsAttr, tz);
        CalendarXmlizer.setAttr(rootElement, "limit", limit);
        CalendarXmlizer.setAttr(rootElement, "show-limit", showLimit);
        CalendarXmlizer.setAttr(rootElement, "only-visible-in-ui", onlyVisibleInUi);
        // No need to setAttr(... "whoAmI" ...)
        logger.debug("= " +
            "layer-ids: " + layerIdsAttr + ", uid2: " + uid2O.get().getUid() + ", " +
            "start-ts: " + startTs + " UTC, end-ts: " + endTsAttr + " UTC, "
        );
        logger.debug("  " +
            "limit: " + limit + ", show-limit: " + showLimit + ", " +
            "only-visible-in-ui: " + onlyVisibleInUi + ", whoAmI: " + whoAmI
        );
        if (StringUtils.isEmpty(whoAmI)) {
            throw new CommandRunException("'whoAmI' is empty");
        }

        Instant startMs = startTs;
        Option<Instant> endMs = Option.ofNullable(endTs);

        EventLoadLimits limits = EventLoadLimits.intersectsInterval(new InfiniteInterval(startMs, endMs));
        final LayerIdPredicate layerIdPredicate;
        if (limit != 0) {
            limits = limits.withResultSize(limit);
        }
        if (AuxColl.isSet(layerIds)) {
            layerIdPredicate = LayerIdPredicate.list(Cf.x(layerIds), onlyVisibleInUi);
        } else {
            layerIdPredicate = LayerIdPredicate.allForUser(uid2O.get(), onlyVisibleInUi);
        }

        ListF<EventInstanceInfo> eventInfos = eventRoutines.getSortedInstancesIMayView(
                uid2O, EventGetProps.any(), layerIdPredicate, limits, ActionSource.WEB);
        int count = eventInfos.size();

        if (showEvents) {
            Option<UserInfo> user2InfoO = obtainUserInfoO(uid2O);
            Element eEvents = new Element("events");
            int eventsShown = 0; // 'showLimit' requires it. Also, we use it for '@shown'.
            for (EventInstanceInfo ei : eventInfos) {
                Option<Layer> layer = ei.getLayerId().filterMap(ei.getEventWithRelations()::findLayerById);
                Element eEvent = eventRoutines.getElement(
                        user2InfoO, ei, layer, Option.empty(), Option.empty(), tz, Option.<String>empty(), ActionSource.WEB);
                //String pubExtId = EventRoutines.getPublicExtId(ei.getE());
                //XmlUtils.appendElm(eEvent, "public-external-id", pubExtId);
                eEvents.addContent(eEvent);

                eventsShown++;
                if (0 < showLimit && showLimit <= eventsShown) { break; }
            } // for
            CalendarXmlizer.setAttr(eEvents, "count", count); // true size
            CalendarXmlizer.setAttr(eEvents, "shown", eventsShown); // shown size
            rootElement.addContent(eEvents);
            // Timezone information, mainly for RSS correct displaying
            rootElement.addContent(DateTimeFormatter.appendTzInfo(tz, new Element("user"), "timezone-javaid", true, null));
        }
        String iconName = (count > 9 ? "more" : Integer.toString(count));
        CalendarXmlizer.appendElm(rootElement, "icon-name", iconName);
    }

}
