package ru.yandex.calendar.frontend.web.cmd.run.ui;

import java.util.Optional;

import javax.annotation.Nullable;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.XmlCommand;
import ru.yandex.calendar.logic.event.grid.GrayedWeekNoAppender;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.holiday.XmlDayInfoHandler;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.commune.holidays.CityLookup;
import ru.yandex.commune.holidays.CountryLookup;
import ru.yandex.commune.holidays.Holiday;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.commune.holidays.HolidaysData;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.misc.lang.StringUtils;

public class CmdGetHolidaysA extends XmlCommand {
    private static final String CMD_TAG = "get-holidays-a";

    private final String viewTypeStr;
    private final String showDateStr;
    private final String forStr;

    public CmdGetHolidaysA(String showDateStr, String viewTypeStr, String forStr) {
        super(CMD_TAG);
        this.showDateStr = showDateStr;
        this.viewTypeStr = viewTypeStr;
        this.forStr = forStr;
    }

    @Override
    protected void buildXmlResponse(XmlCmdContext ctx) {
        Element rootElement = ctx.getRootElement();

        LocalDate showDate = DateTimeFormatter.toDate(showDateStr, new LocalDate(DateTimeZone.UTC));
        ViewType vt = StringUtils.isNotEmpty(viewTypeStr) ? ViewType.R.valueOf(viewTypeStr) : ViewType.getDefault();
        int countryId = lookupGeoId(forStr).orElse(GeobaseIds.RUSSIA);

        DayOfWeek startWeekday = WeekdayConv.getDefaultCals();
        LocalDate startDate = DateTimeFormatter.getViewTypeBoundLocalDate(showDate, ViewType.MONTH, startWeekday, true);
        LocalDate endDate = DateTimeFormatter.getViewTypeBoundLocalDate(showDate, ViewType.MONTH, startWeekday, false).minusDays(1);
        // Generate output
        GrayedWeekNoAppender grWnAppender = vt.createGrayedWeekNoAppender(showDate, startWeekday, DateTimeZone.UTC);
        XmlDayInfoHandler diHandler = new XmlDayInfoHandler(false, true, Option.of(grWnAppender));
        HolidayRoutines.processDates(startDate, endDate, countryId, OutputMode.ALL, false, diHandler);
        rootElement.addContent(diHandler.getElement());
    }

    public static SetF<Integer> knownGeoIds = HolidaysData.getHolidays()
            .map(Holiday::getCountryId).unique();

    public static Optional<Integer> lookupGeoId(@Nullable String s) {
        if (StringUtils.isEmpty(s)) return Optional.empty();

        Optional<Integer> foundById = Cf.x(StringUtils.split(s, ','))
                .filterMap(Cf.Integer.parseSafeF()).find(knownGeoIds.containsF())
                .toOptional();
        if (foundById.isPresent()) {
            return foundById;
        }

        Optional<Integer> foundByCity = CityLookup.lookupCountryByCity(s);
        if (foundByCity.isPresent()) {
            return foundByCity;
        }

        try {
            return Optional.of(CountryLookup.lookupGeobaseId(s));
        } catch (IllegalArgumentException e) {
            return Optional.empty();
        }
    }

    private static final ListF<OutputMode> INCLUDE_LABELS_OUTPUT_MODES = Cf.list(
            OutputMode.ALL,
            OutputMode.WITH_NAMES,
            OutputMode.HOLIDAYS_AND_WITH_NAMES);

    public static boolean isOutLabelsMode(OutputMode mode) {
        return INCLUDE_LABELS_OUTPUT_MODES.containsTs(mode);
    }
}
