package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.logic.event.ModificationInfo;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.event.web.IdOrExternalId;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * Deletes event and (implicitly) and event-user records.
 * Also deletes related entities.
 * NOTE: there is possible error: when we do not specify
 * layerId, system looks for event at user's layers.
 * This can be wrong if event is being deleted from layer
 * which user does not share (for now, this is impossible).
 * @author ssytnik
 */
public class CmdDeleteEvent extends ValidatableXmlCommand {
    private static final String CMD_TAG = "delete-event";

    @Autowired
    private EventWebManager eventWebManager;

    private final long eventId;
    private final String applyToFutureEventsStr;
    private final String eventTsStr;
    private final Option<Integer> sequence;

    public CmdDeleteEvent(
            AuthInfo ai, long eventId, String eventTsStr,
            Option<Integer> sequence, String applyToFutureEventsStr)
    {
        super(CMD_TAG, ai);
        this.eventId = eventId;
        this.eventTsStr = eventTsStr;
        this.applyToFutureEventsStr = applyToFutureEventsStr;
        this.sequence = sequence;
    }

    public CmdDeleteEvent(
            AuthInfo ai, long eventId, String eventTsStr,
            Option<Integer> sequence, String applyToFutureEventsStr, String timezoneId)
    {
        super(CMD_TAG, ai, timezoneId);
        this.eventId = eventId;
        this.eventTsStr = eventTsStr;
        this.applyToFutureEventsStr = applyToFutureEventsStr;
        this.sequence = sequence;
    }

    @Override
    public Duration getTimeout() {
        return Duration.standardSeconds(29);
    }

    @Override
    public void validate() {
        // ssytnik, 2009-10-26: presence needs to be changed 'true' -> 'null' [<=> whole event] (eventTs)
        RequestValidator.validateRequired(RequestValidator.POSITIVE, "eventId", eventId);
        RequestValidator.validateOptional(RequestValidator.TIMESTAMP, "eventTs", eventTsStr);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        final Element rootElement = ctx.getRootElement();

        Instant eventTs = DateTimeFormatter.toNullableTimestampUnsafe(eventTsStr, DateTimeZone.UTC);
        Option<Instant> dayStartMs = Option.ofNullable(AuxDateTime.getMsNullable(eventTs)).map(Instant::new);
        boolean applyToFutureEvents = Binary.parseBoolean(applyToFutureEventsStr);
        // Add info to root element
        CalendarXmlizer.setAttr(rootElement, "event-id", eventId);
        CalendarXmlizer.setAttr(rootElement, "apply-to-future-events", applyToFutureEvents);
        if (eventTs != null) {
            CalendarXmlizer.setDtfAttr(rootElement, "event-ts", eventTs, DateTimeZone.UTC); // ...actually use
        }

        ModificationInfo mi = eventWebManager.deleteEvent(
                userInfoO.get(), IdOrExternalId.id(eventId), sequence, dayStartMs, applyToFutureEvents, getActionInfo()
        );
        mi.appendXmlTo(rootElement, true);
    } // buildXml
}
