package ru.yandex.calendar.frontend.web.cmd.run.ui.event;


import java.util.NoSuchElementException;

import javax.annotation.Nullable;

import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.logic.contact.ContactRoutines;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervalsOrRefusal;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

public class CmdGetAvailabilities extends UserXmlCommand {

    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private ContactRoutines contactRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;

    private final String usersString;

    private final String dayString;
    private final Either<Integer, ViewType> daysOrViewType;

    private final boolean outParticipants;
    private final String officeIdString;

    private final Language lang;

    public CmdGetAvailabilities(
            AuthInfo ai, String users, String day, Either<Integer, String> daysOrViewType, boolean outParticipants,
            @Nullable String officeId, @Nullable String timezoneId, @Nullable String lang)
    {
        super("get-availabilities-day", ai, timezoneId);
        this.usersString = users;
        this.dayString = day;
        this.daysOrViewType = Either.fromOptions(
                daysOrViewType.leftO(),
                daysOrViewType.rightO().map(v -> ViewType.fromValueSafe(v).getOrElse(ViewType.DAY)));
        this.outParticipants = outParticipants;
        this.officeIdString = officeId;
        this.lang = Language.fromValueSafe(lang).getOrElse(Language.RUSSIAN);
    }

    private LocalDate day;
    private LocalDate getDay(DateTimeZone tz) {
        if (day == null) {
            if (StringUtils.isEmpty(dayString))
                day = new LocalDate(tz);
            else
                day = TimeUtils.localDate.parse(dayString);
        }
        return day;
    }

    private Option<Long> getOfficeId() {
        if (StringUtils.isNotEmpty(officeIdString)) {
            return Option.of(Long.parseLong(officeIdString));
        } else {
            return Option.empty();
        }
    }

    private Tuple2List<String, Option<Email>> getUsers() {
        Check.C.notEmpty(usersString);
        final Option<PassportAuthDomain> defaultDomainO = passportAuthDomainsHolder.getDefaultDomain();

        return Cf.x(usersString.split(",")).zipWith(string -> {
            if (string.equals("me")) {
                return Option.empty();
            } else {
                String domainSuffix = string.contains("@") || !defaultDomainO.isPresent() ?
                        "" : "@" + defaultDomainO.get().getDomain().getDomain();
                return Option.of(Emails.punycode(string + domainSuffix));
            }
        });
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        final DateTimeZone timeZone;
        Option<Long> officeId = getOfficeId();
        if (officeId.isPresent()) {
            timeZone = officeManager.getTimeZoneByOfficeId(officeId.get());
        } else {
            timeZone = this.tz;
        }

        final PassportUid uid = uidO.get();

        Element availabilies = new Element("availabilities");
        availabilies.setAttribute("date", getDay(timeZone).toString());

        LocalDate date = getDay(timeZone);
        DateTime midday = date.toDateTime(new LocalTime(12, 0), timeZone);

        InstantInterval scheduleInterval = daysOrViewType.fold(
                d -> new InstantInterval(
                        date.toDateTimeAtStartOfDay(timeZone).toInstant(),
                        date.plusDays(d).toDateTimeAtStartOfDay(timeZone).toInstant()),
                v -> new InstantInterval(
                        midday.property(v.getDateTimeFieldType()).roundFloorCopy(),
                        midday.property(v.getDateTimeFieldType()).roundCeilingCopy()));

        CalendarXmlizer.setDtfAttr(availabilies, "now", Instant.now(), tz);
        CalendarXmlizer.setAttr(availabilies, "tz", tz.getID());
        CalendarXmlizer.setAttr(availabilies, "tz-offset", tz.getOffset(scheduleInterval.getStart()));

        AvailabilityRequest req = AvailabilityRequest.interval(scheduleInterval).includeEventsInfo();

        Function<Option<Email>, Email> emailF = emailO -> emailO.getOrElse(
                () -> userManager.getEmailByUid(uid).orElseThrow(() -> new NoSuchElementException("email is unknown"))); // none means me

        Tuple2List<String, Email> requested = getUsers().map2(emailF);
        MapF<Email, String> requestedByEmail = requested.toMap(Tuple2.get2F(), Tuple2.get1F());

        MapF<Email, NameI18n> resourceNameByEmail = Cf2.flatBy2(
                resourceRoutines.findResourcesByEmails(requestedByEmail.keys())).toMap(
                        Tuple2::get1, t -> ResourceRoutines.getNameWithAlterNameI18n(t.get2())
                                .getOrElse(NameI18n.constant(t.get2().getExchangeName().get())));

        MapF<Email, NameI18n> contactNameByEmail =  Cf2.flatBy2(
                contactRoutines.getI18NamesByEmails(uid, requestedByEmail.keys())).toMap();

        Tuple2List<Email, AvailabilityIntervalsOrRefusal> avails = availRoutines.getAvailabilityIntervalssByEmails(
                uid, requested.get2(), req, getActionInfo());

        for (Tuple2<Email, AvailabilityIntervalsOrRefusal> t : avails) {

            Element availability = new Element("availability");
            availability.setAttribute("attendee", requestedByEmail.getOrThrow(t.get1()));
            availability.setAttribute("attendee-email", Emails.getUnicoded(t.get1()));

            Option<NameI18n> attendeeNameO = resourceNameByEmail.getO(t.get1()).orElse(contactNameByEmail.getO(t.get1()));
            if (attendeeNameO.isPresent()) {
                availability.setAttribute("attendee-name", attendeeNameO.get().getName(lang));
            }
            if (!t.get2().isRefusal()) {
                for (AvailabilityInterval interval : t.get2().getIntervals().forTz(timeZone).bounded().mergedByDays(timeZone)) {
                    availability.addContent(AvailabilityXmlizer.serializeAvailabilityInterval(uidO, interval, timeZone, outParticipants));
                }
            } else {
                Element refusal = new Element("refusal");
                refusal.setAttribute("reason", t.get2().getRefusalReason().toXmlName());
                availability.addContent(refusal);
            }

            availabilies.addContent(availability);
        }

        ctx.getRootElement().addContent(availabilies);

    }
} //~
