package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervalsOrRefusal;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.SubjectId;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

/**
 * Returns availability for a single (optionally specified) date
 * @author ssytnik
 */
public class CmdGetAvailabilityDay extends ValidatableXmlCommand {
    private static final String CMD_TAG = "get-availability-day";
    private static final int MARKERS_PER_DAY = 24;
    private static final int PARTS_PER_MARKER = 12;
    // Note: PARTS_IN_DAY should be divisor of AuxDateTime#MS_PER_DAY_UTC
    private static final int PARTS_IN_DAY = PARTS_PER_MARKER * MARKERS_PER_DAY;
    private static final long PART_LENGTH_MS = Duration.standardDays(1).getMillis() / PARTS_IN_DAY;

    private final Option<PassportUid> uid2;
    private final String showDateStr;

    @Autowired
    private AvailRoutines availRoutines;

    public CmdGetAvailabilityDay(AuthInfo ai, Option<PassportUid> uid2, String showDateStr) {
        super(CMD_TAG, ai);
        this.uid2 = uid2;
        this.showDateStr = showDateStr;
    }

    @Override
    public void validate() {
        //RequestValidator.validateRequired(RequestValidator.POSITIVE, "uid2", uid2);
        RequestValidator.validateOptional(RequestValidator.DATE, "showDate", showDateStr);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        final Element rootElement = ctx.getRootElement();

        // Get incoming parameters from the request (or use defaults instead)

        final LocalDate showDate = StringUtils.isNotEmpty(showDateStr) ?
            TimeUtils.localDate.parse(showDateStr) :
            new LocalDate(tz);
        Instant startMs = showDate.toDateTimeAtStartOfDay(tz).toInstant();
        final LocalDate nextDate = showDate.plusDays(1);
        Instant endMs = nextDate.toDateTimeAtStartOfDay(tz).toInstant();
        // Add info to root element
        CalendarXmlizer.setDtfAttr(rootElement, "current-ts", AuxDateTime.NOWTS(), tz);
        CalendarXmlizer.setAttr(rootElement, "show-date", showDate);
        // Get data
        SubjectId subject = SubjectId.uid(uid2.getOrElse(uidO.get()));
        AvailabilityRequest req = AvailabilityRequest.interval(startMs, endMs);

        AvailabilityIntervalsOrRefusal availability =
                availRoutines.getAvailabilityIntervals(uidO.get(), subject, req, ActionInfo.webTest());


        // Prepare answer
        Availability[] availPieces = new Availability[PARTS_IN_DAY];
        if (!availability.isRefusal()) {
            for (AvailabilityInterval ai : availability.getIntervals().bounded().merged()) {
                // Calculate event start and end milliseconds with [ms1 .. ms2) saturation
                InstantInterval i = ai.getInterval();
                Instant ems1 = ObjectUtils.max(i.getStart().toInstant(), startMs);
                Instant ems2 = ObjectUtils.min(i.getEnd().toInstant(), endMs);
                if (ems1.compareTo(ems2) < 0) { // if equal, k2 could be negative
                    long localMs1 = ems1.getMillis() - startMs.getMillis();
                    long localMs2 = ems2.getMillis() - startMs.getMillis();
                    // Calculate first and last interval indexes to mark as having event(s)
                    int k1 = (int) (localMs1 / PART_LENGTH_MS);
                    int k2 = (int) ((localMs2 - 1) / PART_LENGTH_MS);
                    Availability a = ai.getAvailability();
                    for (int k = k1; k <= k2; ++k) {
                        availPieces[k] = a;
                    }
                }
            }
        }


        // Generate XML output
        Element eIntervals = new Element("intervals");
        for (int k = 0; k < PARTS_IN_DAY; ++k) {
            Availability a = (availPieces[k] != null ? availPieces[k] : Availability.AVAILABLE);
            Element eInterval = new Element("interval").setText(a.toDbValue());
            CalendarXmlizer.setDtfAttr(eInterval, "time", LocalTime.fromMillisOfDay(k * PART_LENGTH_MS), DateTimeZone.UTC);
            if (k % PARTS_PER_MARKER == 0) { CalendarXmlizer.setAttr(eInterval, "marker", "1"); }
            eIntervals.addContent(eInterval);
        }
        rootElement.addContent(eIntervals);
    }
}
