package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import Yandex.RequestPackage.RequestData;
import org.jdom.Element;
import org.joda.time.Chronology;
import org.joda.time.Instant;
import org.joda.time.Interval;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.RequestDataConverter;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ReqProcessingXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.AuxBase;
import ru.yandex.calendar.util.data.AliasedRequestDataProvider;
import ru.yandex.calendar.util.data.DataProvider;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * WAS: shows availability / business of a given user in given time interval.
 * NOW: says whether busy overlap occurs if user creates new busy event here.
 * @author akirakozov
 */
public class CmdGetBusyOverlap extends ReqProcessingXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdGetBusyOverlap.class);
    private static final String CMD_TAG = "get-busy-overlap";

    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;

    public CmdGetBusyOverlap(AuthInfo ai, RequestData requestData) {
        super(CMD_TAG, requestData, ai);
    }

    @Override
    protected void buildXmlResponseRP(XmlCmdContext ctx, AliasedRequestDataProvider dp) {
        Option<UidOrResourceId> subjectIdO = getSubjectIdO(dp);
        Element eRoot = ctx.getRootElement();
        if (subjectIdO.isPresent()) {
            UidOrResourceId subjectId = subjectIdO.get();
            logger.debug("Determined subject id: " + subjectId);
            Instant startTs = DateTimeFormatter.toNullableTimestampUnsafe(dp.getText("start-ts", true), tz);
            Instant endTs = DateTimeFormatter.toNullableTimestampUnsafe(dp.getText("end-ts", true), tz);
            final PassportUid yandexUid = uidO.get();
            Chronology chrono = dateTimeManager.getChronologyForUid(yandexUid);
            Interval eventInterval = new Interval(startTs.getMillis(), endTs.getMillis(), chrono);
            Repetition r = null;
            DataProvider eDp = dp.getDataProvider("event", false);
            if (eDp != null) {
                DataProvider rDp = eDp.getDataProvider("repetition", false);
                r = RequestDataConverter.convertRepetition(tz, rDp, startTs);
            }

            Option<Long> eventId = eDp != null ? RequestDataConverter.convertEventId(eDp) : Option.<Long>empty();

            boolean busyOverlap = false;

            boolean canViewAvailability = authorizer.canViewAvailability(userInfoO.get(), subjectId);
            logger.debug("Busy check: uid = " + yandexUid + ", subject = " + subjectId + ", can-view-avail? " + canViewAvailability);
            if (canViewAvailability) {
                busyOverlap = availRoutines.busyOverlappingEventsInFuture(
                        uidO, subjectId, new Instant(0),
                        RepetitionInstanceInfo.create(eventInterval, Option.ofNullable(r)),
                        eventId, true, getActionInfo()).isNotEmpty();
            }

            CalendarXmlizer.setAttr(eRoot, "subject-id", subjectId.toString()); // debug
            CalendarXmlizer.setAttr(eRoot, "busy-overlap", busyOverlap);
        } else {
            logger.debug("No subject is determined - busy overlap is unknown");
            CalendarXmlizer.setAttr(eRoot, "busy-overlap", "unknown"); // update for CAL-1971
        }
    }

    private Option<UidOrResourceId> getSubjectIdO(AliasedRequestDataProvider dp) {
        Option<PassportUid> uid2O = Option
                .ofNullable(AuxBase.toNullableLongUnsafe(dp.getText("uid2", false)))
                .map(PassportUid::cons);
        if (!uid2O.isPresent()) {
            String emailStr = dp.getText("email", false);
            if (StringUtils.isEmpty(emailStr)) {
                throw new CommandRunException("Neither 'uid2', nor 'email' is given");
            }
            Email email = Emails.punycode(emailStr);
            Option<Long> resourceIdO = resourceRoutines.parseIdFromEmail(email);
            if (resourceIdO.isPresent()) {
                return Option.of(UidOrResourceId.resource(resourceIdO.get()));
            }
            uid2O = userManager.getUidByEmail(email);
            if (!uid2O.isPresent()) {
                logger.info("Could not find 'uid2' by 'email': " + email);
            }
        }
        return UidOrResourceId.notNullUserO(uid2O);
    }
}
