package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import javax.annotation.Nullable;

import lombok.val;
import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.layer.UserLayersSharing;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.InvitationXmlizer;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.net.AssertNoNetworkAccessStack;
import ru.yandex.misc.net.AssertNoNetworkAccessStackHandle;

public class CmdGetEventInfos extends UserXmlCommand {
    private static final String CMD_TAG = "get-event-infos";

    private final String eventIdsStr;
    private final Option<String> linkSignKey;

    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private Authorizer authorizer;

    public CmdGetEventInfos(AuthInfo ai, String eventIdsStr, @Nullable String linkSignKey) {
        super(CMD_TAG, ai);
        this.eventIdsStr = eventIdsStr;
        this.linkSignKey = StringUtils.notEmptyO(linkSignKey);
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        ListF<Long> eventIds = Cf.x(StringUtils.split(eventIdsStr, ",")).filterMap(Cf.Long.parseSafeF());

        ListF<EventWithRelations> events = MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_ANY, () -> {
            ListF<EventWithRelations> loaded = eventDbManager.getEventsWithRelationsByIdsSafe(eventIds);

            authorizer.loadBatchAndCacheAllRequiredForPermsCheck(uidO.get(), loaded);

            return loaded;
        });

        AssertNoNetworkAccessStackHandle h = AssertNoNetworkAccessStack.push();
        try {
            Element eventsElement = new Element("events");
            xmlizeEvents(events, eventsElement);
            ctx.getRootElement().addContent(eventsElement);
        } finally {
            h.popSafely();
        }
    }

    private void xmlizeEvents(ListF<EventWithRelations> events, Element eventsElement) {
        for (EventWithRelations event : events) {
            val eventElement = new Element("event");
            event.getEvent().appendXmlTo(eventElement, tz, EventFields.ID, EventFields.IS_ALL_DAY);

            val layerOfUser = event.findOwnUserLayer(uidO.get());
            if (layerOfUser.isPresent()) {
                CalendarXmlizer.appendElm(eventElement, "layer-id", layerOfUser.get().getId());
            }

            val source = ActionSource.WEB;
            val alreadyAttached = uidO.isPresent() && layerOfUser.isPresent();

            final var permissions = userInfoO.map(user -> {
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
                return authorizer.getEventPermissions(user, eventAuthInfo, source);
            }).getOrElse(() -> {
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(event);
                return authorizer.getEventPermissions(eventAuthInfo, source);
            });

            val mayView = permissions.contains(EventAction.VIEW);

            CalendarXmlizer.appendElm(eventElement, "can-attach", mayView && !alreadyAttached);
            CalendarXmlizer.appendElm(eventElement, "can-view", mayView);
            CalendarXmlizer.appendElm(eventElement, "can-edit", permissions.contains(EventAction.EDIT));
            CalendarXmlizer.appendElm(eventElement, "can-delete", permissions.contains(EventAction.DELETE));

            val userEventUser = event.findUserEventUser(uidO.get());
            val userIsParticipant = layerOfUser.isPresent()
                    && userEventUser.isPresent() && userEventUser.get().getDecision() != Decision.NO;
            CalendarXmlizer.appendElm(eventElement, "user-is-participant", userIsParticipant);

            if (mayView) {
                CalendarXmlizer.appendElm(eventElement, "name", event.getEvent().getName());
                CalendarXmlizer.appendElm(eventElement, "description", event.getEvent().getDescription());
                CalendarXmlizer.appendElm(eventElement, "description-html",
                        EventRoutines.getDescriptionHtmlForVerstka(event.getEvent().getDescription(), linkSignKey));
                eventElement.addContent(InvitationXmlizer.serializeParticipants(event.getParticipants(), uidO));
            } else {
                eventElement.addContent(InvitationXmlizer.serializeOrganizer(event.getParticipants(), uidO));
            }

            val repeating = event.getEvent().getRepetitionId().isPresent();
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(event, UserLayersSharing.empty());
            CalendarXmlizer.appendElm(eventElement, "is-repeating", repeating);
            CalendarXmlizer.appendElm(eventElement, "is-recurrence", event.getRecurrenceId().isPresent());
            CalendarXmlizer.appendElm(eventElement, "is-private", eventAuthInfo.isPrivate());
            CalendarXmlizer.appendElm(eventElement, "is-multi-office", event.isMultiOffice());

            CalendarXmlizer.appendElm(eventElement, "sequence", event.getEvent().getSequence());

            eventsElement.addContent(eventElement);
        }
    }
}
