package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import java.util.Calendar;
import java.util.Stack;

import javax.annotation.Nullable;

import org.jdom.Attribute;
import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.MutableDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.ui.CmdGetHolidaysA;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventsFilter;
import ru.yandex.calendar.logic.event.IntervalComparator;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.grid.GrayedWeekNoAppender;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.event.reminders.RemindersClient;
import ru.yandex.calendar.logic.event.reminders.RemindersEventInfo;
import ru.yandex.calendar.logic.event.repetition.EventIndentAndRepetitionAndPerms;
import ru.yandex.calendar.logic.event.repetition.EventIndentInterval;
import ru.yandex.calendar.logic.event.repetition.EventIndentIntervalAndPerms;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.dates.WeekdayConv;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.calendar.util.xml.TagReplacement;
import ru.yandex.commune.holidays.DayInfo;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.commune.holidays.MapDayInfoHandler;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder.PolicyHandle;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * Returns all events owned (created or shared) by current user.
 * In the future: OR user specified by uid.
 * - showDateStr - string representation of the date which must be included in the output
 * (if not set, then current date is used)
 * - viewType - "day", "week" or "month" (if not set, then "week" is used by default)
 *
 * @author ssytnik
 */
public class CmdGetEvents extends ValidatableXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdGetEvents.class);
    private static final String CMD_TAG = "get-events";

    private Option<PassportUid> uid2O;
    private Option<UserInfo> user2InfoO;
    private String layerIdsStr; // optional (default: all user's layers) / required for *A, *AP cases

    private final String showDateStr; // optional (default: current date)
    private final String viewTypeStr; // optional (default: 'week' for *A case, settings value otherwise)
    private final String holidaysFor;

    private final boolean indentsOnly; // if true, then no event information given but their indents
    private final boolean includeReminders;

    private final Option<String> linkSignKey;

    private boolean sSet;
    private Option<SettingsInfo> settings;
    private LocalDate showDate;
    private ViewType viewType;
    private DayOfWeek startWeekday;
    // used to show small calendar in the top-level corner of page
    private DateTime monthStart;
    private DateTime monthEnd;
    private MapF<LocalDate, ListF<EventsGroup>> dayGroups = Cf.hashMap();
    private MapF<LocalDate, EventsData> dayEvents = Cf.hashMap();
    private int vIndent; // event vIndent (initially, -1)
    // (Only calculated if 'show-is-all-day') current offset in
    // days from month start, current remaining event length

    private ListF<Long> layerIds;


    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private RemindersClient remindersClient;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;

    // CTORS //

    // Authorized
    // uiGetEvents/uiGetEventIndents ctor
    public CmdGetEvents(
            AuthInfo ai, String showDateStr, String viewTypeStr,
            String holidaysFor, boolean indentsOnly, boolean includeReminders, @Nullable String linkSignKey)
    {
        super(CMD_TAG, ai);
        this.uid2O = uidO;
        this.layerIdsStr    = null;

        this.showDateStr    = showDateStr;
        this.viewTypeStr    = viewTypeStr;
        this.holidaysFor    = holidaysFor;

        this.indentsOnly    = indentsOnly;
        this.includeReminders = includeReminders;
        this.linkSignKey = StringUtils.notEmptyO(linkSignKey);
    }

    // Anonymous
    // uiGetEventsA/AP, uiGetEventIndentsA/AP ctor (either layerIds OR privateToken should be specified, but not both)
    public CmdGetEvents(
            @Nullable AuthInfo ai, String tzId, String layerIdsStr, String privateToken,
            String showDateStr, String viewTypeStr, String holidaysFor, boolean indentsOnly, @Nullable String linkSignKey)
    {
        super(CMD_TAG, ai, tzId);
        setMustHaveAuth(ai != null);

        boolean layerIdsSet = StringUtils.isNotEmpty(layerIdsStr);
        boolean privateTokenSet = StringUtils.isNotEmpty(privateToken);
        if (layerIdsSet == privateTokenSet) {
            logger.error("Ctor(): !!! INVALID: layer ids set? " + layerIdsSet + "; private token set? " + privateTokenSet);
        }
        this.uid2O = uidO;
        this.layerIdsStr    = layerIdsStr;

        this.showDateStr    = showDateStr;
        this.viewTypeStr    = viewTypeStr;
        this.holidaysFor    = holidaysFor;

        this.indentsOnly    = indentsOnly; // default // if layerIds == null, then privateToken is set
        this.includeReminders = false;
        this.linkSignKey = StringUtils.notEmptyO(linkSignKey);

        if (privateTokenSet) {
            setPrivateToken(privateToken);
        }
    }

    // OVERRIDES

    @Override
    protected void obtainPrivateResource(String privateToken) {
        Layer l = layerRoutines.getByPrivateToken(privateToken);
        this.uid2O = Option.of(l.getCreatorUid()); // as if layer creator was looking over his own layer
        this.layerIdsStr = String.valueOf(l.getId());
    }

    @Override
    public void validate() {
        // As for showDate, we allow error here and fix at the main code
        RequestValidator.validateOptional(ViewType.AV, "viewType", viewTypeStr);
        RequestValidator.validate(RequestValidator.LONG_ARRAY, "layerIds", layerIdsStr);
    }

    public boolean showEvents(boolean isGrayed) {
        return !isGrayed || viewType.equals(ViewType.MONTH) || viewType.equals(ViewType.GREY_MONTH);
    }

    // This class is used in mapping { date_milliseconds => events data }
    public static class EventsData {
        private static final Duration MIN_EVENT_DURATION = Duration.standardMinutes(45);
        // These stacks are temporary (and all have the same size)
        private Stack<Element> tmpEvents = new Stack<Element>(); // for final putting calculated h.indent-r attribute
                                            // to event xml tag without iterating through list
        private Stack<Short> tmpHindentrs = new Stack<Short>(); // h.indent-r attributes can't be calculated once so
                                            // due to optimization reasons they are kept as map
                                            // used for calculating h.indent-r attribute
        private Stack<Long> tmpEndTimes = new Stack<Long>(); // is used for calculating h.indent and
                                            // h.indent-r attributes
        // These variables are preserved until xml output begins
        public Tuple2List<Element, Option<EventIndentIntervalAndPerms>> eventOrReminders = Tuple2List.arrayList();

        public int dayEventsCount = 0; // count of event parts in this day; used for
                                            // get-event-indents, may differ from events.size()
        // final/temporary all-day vblock indents
        public int ansVBlockIndent = -1;
        public int curVBlockIndent = -1;

        // Updates event end times stack with given new event interval,
        // then calculates and returns h.indent value for it (given new event).
        // Do not call this routine for all day events if they are shown separately.

        // eventId - cannot be null, it is Long for boxing optimization reasons only
        private void handleIndentAttributes(
                DateTimeZone tz, Element eEvent,
                long newStartMs, long newEndMs)
                {

            // '<=' below is for 'no space between events'
            while (tmpEndTimes.size() > 0 && tmpEndTimes.peek() <= newStartMs) {
                doPop();
            }

            // Adjust minimal event length
            // (IMPORTANT NOTE: if ONLY event length is 0, NOT if end < start + MIN.DUR)
            if (newStartMs == newEndMs /*&& newEndMs < newStartMs + MIN_EVENT_DURATION*/) {
                newEndMs = newStartMs + MIN_EVENT_DURATION.getMillis();
            }
            // Increase hindentr-s for the previous events.
            // At some moment, we can encounter event, for
            // which the stack of the other events on it was
            // bigger than it is now. So no need to continue.
            tmpEvents.push(eEvent);
            tmpHindentrs.push((short) 0);
            for (int i = tmpHindentrs.size() - 2; i >= 0; --i) {
                short newValue = (short) (tmpHindentrs.size() - 1 - i);
                if (newValue > tmpHindentrs.get(i).shortValue()) {
                    tmpHindentrs.setElementAt(newValue, i);
                } else {
                    break; // no need to continue, newValue will always be less than current item
                }
            }
            tmpEndTimes.push(newEndMs);
            // SetF h.indent and v.indent immediately;
            // h.indent-r will be set here or in finalFlush() during doPop() executing
            int hindent = tmpEndTimes.size() - 1;
            Calendar c = DateTimeFormatter.createCalendar(tz);
            c.setTimeInMillis(newStartMs);
            int vindent = c.get(Calendar.MINUTE) / 5;
            CalendarXmlizer.setAttr(eEvent, "hindent", hindent);
            CalendarXmlizer.setAttr(eEvent, "vindent", vindent);
        }

        public void finalFlush() {
            // flush hindentr attributes
            while (tmpEndTimes.size() > 0) {
                doPop();
            }
//            // NOTE: the code below is now unnecessary because we
//            // calculate ansVBlockIndent for non-show-all-day events
//            // flush vblock indent (if days does not contain
//            // show-is-all-day events, but contains usual ones)
//            if (ansVBlockIndent == -1) { ansVBlockIndent = curVBlockIndent + 1; }
        }

        private void doPop() {
            final Element eEvent = tmpEvents.pop();
            final Short hindentr = tmpHindentrs.pop();
            tmpEndTimes.pop();
            CalendarXmlizer.setAttr(eEvent, "hindent-r", hindentr);
        }
    } // EventsData class ends

    public static class EventsStack {
        public final Instant start;
        public int width;

        public EventsStack(Instant start) {
            this.start = start;
        }
    }

    // TODO Move this class to the frontend layer
    public static class EventsGroup {
        private static final Duration MIN_GROUP_LENGTH = Duration.standardMinutes(45);
        private static final Duration MIN_EVENT_LENGTH = Duration.standardMinutes(15);

        public static final Duration REMINDER_LENGTH = Duration.standardMinutes(30);

        public final Instant start;
        public final Instant end;

        public final EventsStack stack;
        public final int height;

        private final MapF<String, EventInGroup> eventById = Cf.hashMap();
        private final ListF<EventInGroup> eventsBranch = Cf.arrayList();
        private int size;

        public EventsGroup(InstantInterval interval, EventsStack stack, int height) {
            this.start = interval.getStart();
            this.end = getEndBoundedByMinLength(interval);
            this.stack = stack;
            this.height = height;
        }

        public void put(InstantInterval interval, String eventId) {
            interval = new InstantInterval(interval.getStart(), getEndBoundedByMinLength(interval));
            EventSubGroup subgroup;
            int pos;

            if (eventsBranch.isNotEmpty()) {
                ListF<EventInGroup> tail = eventsBranch.reverse().takeWhile(
                        EventInGroup.endsAfterF(interval.getStart()).notF());

                if (tail.isNotEmpty()) {
                    pos = eventsBranch.size() - tail.size();
                    subgroup = new EventSubGroup(pos, 1, true);

                    if (tail.map(EventInGroup.getSubgroupF()).unique().size() == 1) {
                        tail.forEach(EventInGroup.setSubgroupF(new EventSubGroup(pos, tail.size(), true)));
                    }
                    eventsBranch.subList(pos, eventsBranch.size()).clear();
                    eventsBranch.last().subgroup.actual = false;

                } else {
                    pos = eventsBranch.last().pos + 1;
                    subgroup = eventsBranch.last().subgroup;
                    subgroup.size += 1;
                }
            } else {
                pos = 0;
                subgroup = new EventSubGroup(0, 1, false);
            }
            eventsBranch.add(new EventInGroup(interval, pos, subgroup));
            eventById.put(eventId, eventsBranch.last());
            size = Math.max(size, pos + 1);
        }

        public EventInGroupPosition getPosition(String eventId) {
            EventInGroup event = eventById.getOrThrow(eventId);

            return new EventInGroupPosition(
                    event.pos, size,
                    Option.when(event.subgroup.actual, event.subgroup.start),
                    Option.when(event.subgroup.actual, event.subgroup.size));
        }

        private Instant getEndBoundedByMinLength(InstantInterval interval) {
            return ObjectUtils.max(interval.getEnd(), interval.getStart().plus(MIN_EVENT_LENGTH));
        }

        public static Function1B<EventsGroup> matchesF(final Instant nextStart) {
            return new Function1B<EventsGroup>() {
                public boolean apply(EventsGroup g) {
                    return g.start.plus(MIN_GROUP_LENGTH).isAfter(nextStart) && nextStart.isBefore(g.end);
                }
            };
        }

        public static Function1B<EventsGroup> endsAfterF(final Instant instant) {
            return new Function1B<EventsGroup>() {
                public boolean apply(EventsGroup g) {
                    return g.end.isAfter(instant);
                }
            };
        }
    }

    public static class EventSubGroup {
        public int start;
        public int size;
        public boolean actual;

        public EventSubGroup(int start, int size, boolean actual) {
            this.start = start;
            this.size = size;
            this.actual = actual;
        }
    }

    public static class EventInGroup {
        public final Instant start;
        public final Instant end;
        public final int pos;
        public EventSubGroup subgroup;

        public EventInGroup(InstantInterval interval, int pos, EventSubGroup subgroup) {
            this.start = interval.getStart();
            this.end = interval.getEnd();
            this.pos = pos;
            this.subgroup = subgroup;
        }

        public static Function1B<EventInGroup> endsAfterF(final Instant instant) {
            return new Function1B<EventInGroup>() {
                public boolean apply(EventInGroup e) {
                    return e.end.isAfter(instant);
                }
            };
        }

        public static Function<EventInGroup, EventSubGroup> getSubgroupF() {
            return new Function<EventInGroup, EventSubGroup>() {
                public EventSubGroup apply(EventInGroup e) {
                    return e.subgroup;
                }
            };
        }

        public static Function1V<EventInGroup> setSubgroupF(final EventSubGroup subgroup) {
            return new Function1V<EventInGroup>() {
                public void apply(EventInGroup e) {
                    e.subgroup = subgroup;
                }
            };
        }
    }

    public static class EventInGroupPosition {
        public final int groupPos;
        public final int groupSize;
        public final Option<Integer> subgroupStart;
        public final Option<Integer> subgroupSize;

        public EventInGroupPosition(
                int groupPos, int groupSize, Option<Integer> subgroupStart, Option<Integer> subgroupSize)
        {
            this.groupPos = groupPos;
            this.groupSize = groupSize;
            this.subgroupStart = subgroupStart;
            this.subgroupSize = subgroupSize;
        }
    }

    public static class EventOrReminder {
        private final Either<EventIndentIntervalAndPerms, RemindersEventInfo> either;

        public EventOrReminder(Either<EventIndentIntervalAndPerms, RemindersEventInfo> either) {
            this.either = either;
        }

        public static Function<EventIndentIntervalAndPerms, EventOrReminder> eventF() {
            return e -> new EventOrReminder(Either.left(e));
        }

        public static Function<RemindersEventInfo, EventOrReminder> reminderF() {
            return r -> new EventOrReminder(Either.right(r));
        }

        public InstantInterval getInterval(DateTimeZone tz) {
            return new InstantInterval(getStart(tz), getEnd(tz));
        }

        public Instant getStart(DateTimeZone tz) {
            return isEvent()
                    ? getEvent().getEventInterval().getStart().toInstant(tz)
                    : getReminder().getStart().toInstant();
        }

        public Instant getEnd(DateTimeZone tz) {
            return isEvent()
                    ? getEvent().getEventInterval().getEnd().toInstant(tz)
                    : getReminder().getStart().plus(EventsGroup.REMINDER_LENGTH).toInstant();
        }

        public boolean isEvent() {
            return either.isLeft();
        }

        public EventIndentIntervalAndPerms getEvent() {
            return either.getLeft();
        }

        public RemindersEventInfo getReminder() {
            return either.getRight();
        }

        public String getId() {
            return isEvent()
                    ? Cf.list(getEvent().getEventId(), getEvent().getIndent().getLayerOrResourceId()).mkString(":")
                    : Cf.list(getReminder().getClientId(), getReminder().getExternalId(), getReminder().getIdx()).mkString(":");
        }

        public static Comparator<EventOrReminder> comparator(final DateTimeZone tz) {
            final IntervalComparator intervalComparator = IntervalComparator.INSTANCE;
            final Comparator<EventIndentIntervalAndPerms> eventsComparator =
                    Comparator.<EventIndentIntervalAndPerms>constEqualComparator()
                            .thenComparing(intervalComparator.compose(i -> i.getIndent().getInterval()))
                            .thenComparing(Comparator.naturalComparator().compose(i -> i.getIndent().getLayerOrResourceId()));

            return new Comparator<EventOrReminder>() {
                public int compare(EventOrReminder e1, EventOrReminder e2) {
                    return e1.isEvent() && e2.isEvent()
                            ? eventsComparator.compare(e1.getEvent(), e2.getEvent())
                            : intervalComparator.compare(e1.getInterval(tz), e2.getInterval(tz));
                }
            };
        }

        public static Function1B<EventOrReminder> isEventF() {
            return new Function1B<EventOrReminder>() {
                public boolean apply(EventOrReminder e) {
                    return e.isEvent();
                }
            };
        }
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        final Element rootElement = ctx.getRootElement();

        user2InfoO = !uid2O.equals(uidO) ? obtainUserInfoO(uid2O) : userInfoO;
        sSet = uid2O.isPresent();
        settings = settingsRoutines.getSettingsByUidBatch(uid2O).values().singleO();

        Option<Settings> settingsCommon = settings.map(SettingsInfo.getCommonF());
        Option<SettingsYt> settingsYt = settings.filterMap(SettingsInfo.getYtF());

        showDate = DateTimeFormatter.toDate(showDateStr, new LocalDate(tz));

        if (StringUtils.isNotEmpty(viewTypeStr)) {
            viewType = ViewType.R.valueOf(viewTypeStr);
        } else if (sSet) {
            viewType = settingsCommon.get().getViewType();
        } else {
            viewType = ViewType.getDefault();
        }
        startWeekday = sSet ? settingsCommon.get().getStartWeekday() : WeekdayConv.getDefaultCals();
        GrayedWeekNoAppender grayerWeekNoAppender = viewType.createGrayedWeekNoAppender(showDate, startWeekday, tz);

        Tuple2<DateTime, DateTime> monthBounds = DateTimeFormatter.getViewTypeBounds(tz, showDate, ViewType.MONTH, startWeekday);
        monthStart = monthBounds._1;
        monthEnd = monthBounds._2;

        LayerIdPredicate layerIdPredicate;
        if (StringUtils.isNotEmpty(layerIdsStr)) {
            ListF<Long> layerIds = AuxColl.splitToLongArray(layerIdsStr);
            layerIdPredicate = LayerIdPredicate.list(layerIds, false);
        } else {
            layerIdPredicate = LayerIdPredicate.allForUser(uid2O.get(), true);
        }
        layerIds = eventRoutines.getLayerIds(layerIdPredicate);

        MapF<Long, Layer> layerById = layerRoutines.getLayersById(layerIds).toMapMappingToKey(Layer.getIdF());

        ListF<EventOrReminder> events = loadEvents();
        // NOW: we need to process all show-is-all-day events first; all others next.
        //      This is required to proper calculation of the vblock-indent attribute.
        ListF<EventOrReminder> simpleEvents = Cf.arrayList();
        for (EventOrReminder ei : events) {
            boolean showIsAllDay = ei.isEvent() && EventRoutines.getShowIsAllDay(ei.getEvent().getIndent().getTime(), tz);
            if (showIsAllDay) {
                addEventDaysToMap(ei.getEvent(), true);
            } else {
                addEventOrReminderToDaysGroups(ei);
                simpleEvents.add(ei);
            }
        }
        for (EventOrReminder ei : simpleEvents) {
            addEventOrReminderToDaysMap(ei, false);
        }

        // Debug logging
        LocalDate monthStartDt = monthStart.toLocalDate();
        LocalDate monthEndDt = monthEnd.minusDays(1).toLocalDate();
        final int countryId = settingsYt.isPresent() && settingsYt.get().getTableOfficeId().isPresent()
                ? officeManager.getCountryIdByOfficeId(settingsYt.get().getTableOfficeId().get())
                : CmdGetHolidaysA.lookupGeoId(holidaysFor).orElse(GeobaseIds.RUSSIA);

        String vtStr = viewType.toDbValue();
        logger.debug(
            "[:] given (s/d str.: " + showDateStr + ", v/t: " + viewTypeStr + ", " +
            "tz: " + tz.getID() + ", s/wd: " + startWeekday + "), " +
            "used (s/ts: " + showDate + ", " +
            "s/d: " + showDate + ", v/t: " + vtStr + "), " +
            "ts1: " + monthStartDt + ", " +
            "ts2: " + monthEndDt + ", " +
            "country: " + countryId
        );
        // Add info to root element
        CalendarXmlizer.setDtfAttr(rootElement, "current-ts", AuxDateTime.NOWTS(), tz); // local tz
        CalendarXmlizer.setAttr(rootElement, "current-ts-tz-offset", tz.getOffset(AuxDateTime.NOWTS()));

        CalendarXmlizer.setDtfAttr(rootElement, "show-date", showDate, tz); // local tz
        CalendarXmlizer.setAttr(rootElement, "view-type", vtStr);
        CalendarXmlizer.setDtfAttr(rootElement, "date1", monthStartDt, tz); // local tz
        CalendarXmlizer.setDtfAttr(rootElement, "date2", monthEndDt, tz); // inclusive, local tz
        CalendarXmlizer.setAttr(rootElement, "has-created-event", handleHasCreatedEvent(
                events.count(EventOrReminder.isEventF())) ? 1L : 0L);
        // Generate XML output
        //rootElement.addContent(new Element("timezone-javaid").setText(settings.getTimezoneJavaid()));
        MapDayInfoHandler diHandler = new MapDayInfoHandler();
        boolean forYandex = passportAuthDomainsHolder.containsYandexTeamRu();

        HolidayRoutines.processDates(monthStartDt, monthEndDt, countryId, OutputMode.ALL, forYandex, diHandler);

        SetF<Long> eventIdsToGetInfo = Cf.hashSet();
        MutableDateTime mdt;

        if (!indentsOnly) {
            for (mdt = new MutableDateTime(monthStart, tz); mdt.getMillis() < monthEnd.getMillis(); mdt.addDays(1)) {
                LocalDate dayDate = new LocalDate(mdt.getMillis(), tz);
                EventsData eventsData = dayEvents.getTs(dayDate);

                if (showEvents(grayerWeekNoAppender.isGrayed(dayDate)) && eventsData != null) {
                    eventIdsToGetInfo.addAll(eventsData.eventOrReminders
                            .filterMap(t -> t.get2().map(EventIndentIntervalAndPerms::getEventId)));
                }
            }
        }
        MapF<Long, EventInfo> eventInfoById = Cf.map();
        if (eventIdsToGetInfo.isNotEmpty()) {
            EventGetProps egp = EventGetProps.none();

            egp = egp.loadEventUserWithNotifications();
            egp = egp.loadMainEventField(Cf.list(MainEventFields.EXTERNAL_ID));
            egp = egp.loadEventFields(viewType == ViewType.DAY
                    ? Cf.list(EventFields.SEQUENCE, EventFields.NAME,
                            EventFields.LOCATION, EventFields.DESCRIPTION, EventFields.URL)
                    : Cf.list(EventFields.SEQUENCE, EventFields.NAME));

            ListF<EventIndentAndRepetitionAndPerms> indents = events.filterMap(e -> {
                if (e.isEvent() && eventIdsToGetInfo.containsTs(e.getEvent().getEventId())) {
                    return Option.of(e.getEvent().getIndentAndRepetitionAndPerms());
                } else {
                    return Option.empty();
                }
            });
            ListF<EventInfo> eventInfos = eventInfoDbLoader.getEventInfosByIndentsAndPerms(
                    user2InfoO, egp, indents, ActionSource.WEB);

            eventInfoById = eventInfos.toMapMappingToKey(EventInfo::getEventId);
        }

        Function2<EventInfo, Long, EventActions> actionsByEventInfoAndLayerId = (event, layerId) -> {
            Option<EventUser> eventUser = event.getEventUser();
            Option<Layer> layer = layerById.getO(layerId);

            return eventRoutines.getEventActions(
                    userInfoO.get(), event.getInfoForPermsCheck(), eventUser, layer, ActionSource.WEB);
        };
        actionsByEventInfoAndLayerId =
                userInfoO.isPresent() ? actionsByEventInfoAndLayerId.memoize() : (e, l) -> EventActions.empty();

        mdt = new MutableDateTime(monthStart, tz);
        while (mdt.getMillis() < monthEnd.getMillis()) {
            LocalDate dayDate = new LocalDate(mdt.getMillis(), tz);

            Element dayElement = new Element("day");
            CalendarXmlizer.setAttr(dayElement, "date", dayDate);
            boolean isGrayed = grayerWeekNoAppender.append(dayElement, dayDate);

            EventsData eventsData = dayEvents.getTs(dayDate);
            int dayEventsCount = 0;
            if (eventsData != null) {
                dayEventsCount = eventsData.dayEventsCount;
//                if (eventsData.getEvents().size() > 0) { // is this check needed? it seems that no
                    eventsData.finalFlush(); // finish calculating as stacks won't be empty
                    if (showEvents(isGrayed)) {
                        // Here we fill the day element with the events.
                        // Calculate 'next-id's first
                        String nextEIdStr = null;
                        for (int i = eventsData.eventOrReminders.size() - 1; i >= 0; --i) {
                            if (!eventsData.eventOrReminders.get(i).get2().isPresent()) continue;

                            Element eventElm = eventsData.eventOrReminders.get(i).get1();
                            if (StringUtils.isNotEmpty(nextEIdStr)) {
                                eventElm.setAttribute("next-id", nextEIdStr);
                            }
                            nextEIdStr = eventElm.getAttributeValue("id");
                        }
                        for (Tuple2<Element, Option<EventIndentIntervalAndPerms>> eventOrReminder
                                : eventsData.eventOrReminders)
                        {
                            Option<EventIndentIntervalAndPerms> indentO = eventOrReminder.get2();
                            Option<EventInfo> info = indentO.filterMap(
                                    Cf2.f(eventInfoById::getO).compose(EventIndentIntervalAndPerms::getEventId));

                            if (info.isPresent()) {
                                EventIndentIntervalAndPerms indent = indentO.get();

                                Tuple2<LocalDateTime, LocalDateTime> startEnd = getShowStartAndEnd(
                                        indent.getIndentInterval(), dayDate, tz);

                                EventActions actions = actionsByEventInfoAndLayerId.apply(
                                        info.get(), indent.getIndent().getLayerOrResourceId());

                                EventInstanceInfo ei = EventRoutines.toEventInstanceInfoFromLayer(
                                        indent.getIndentInterval(), info.get());

                                Element element = eventRoutines.getElementWithNotification(
                                        ei, user2InfoO, layerById.getO(indent.getIndent().getLayerOrResourceId()),
                                        Option.of(actions),
                                        startEnd.get1(), startEnd.get2(), tz, linkSignKey, ActionSource.WEB);

                                eventOrReminder.get1().getAttributes().forEach(a -> element.setAttribute(
                                        ((Attribute) a).getName(), ((Attribute) a).getValue()));

                                dayElement.addContent(element);
                            } else {
                                dayElement.addContent(eventOrReminder.get1());
                            }
                        }
                    } // showEvents?
                    CalendarXmlizer.setAttr(dayElement, "vblock-indent", eventsData.ansVBlockIndent < 0
                            ? eventsData.curVBlockIndent + 1
                            : eventsData.ansVBlockIndent);
//                } // if eventData contains events that were filled
            } // if eventsData is not null
            // ssytnik: the 'if 'below was added on 2009-09-04;
            if (dayEventsCount > 0) {
                CalendarXmlizer.setAttr(dayElement, "count", dayEventsCount);
            }

            DayInfo dayInfo = diHandler.getDayInfo(dayDate);
            Option<String> nameO = dayInfo.getNameO();
            if (dayInfo.isDayOff() || dayInfo.isTransfer() || nameO.isPresent()) {
                CalendarXmlizer.setAttr(dayElement, "is-holiday", dayInfo.isDayOff());
                if (nameO.isPresent()) {
                    CalendarXmlizer.setAttr(dayElement, "holiday-name", nameO.get());
                }
            }
            CalendarXmlizer.setAttr(dayElement, "week-of-year", mdt.getWeekOfWeekyear());
            rootElement.addContent(dayElement);

            mdt.addDays(1);
        } // day-iterating while
    } // buildXml method

    private ListF<EventOrReminder> loadEvents() {
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(monthStart.toInstant(), monthEnd.toInstant());

        ListF<EventIndentIntervalAndPerms> eventInfos = eventRoutines.getSortedIndentsIMayView(
                user2InfoO, LayerIdPredicate.list(layerIds, false),
                limits, EventsFilter.DEFAULT, ActionSource.WEB);

        ListF<EventOrReminder> events = eventInfos.map(EventOrReminder.eventF());

        if (!includeReminders || uid2O.exists(PassportUid::isYandexTeamRu)) return events;

        Tuple2<DateTime, DateTime> bounds = DateTimeFormatter.getViewTypeBounds(
                tz, showDate, viewType, startWeekday);

        ListF<RemindersEventInfo> reminderInfos = remindersClient.findEvents(
                uid2O.get(), bounds._1.toLocalDate(), bounds._2.minusDays(1).toLocalDate(), tz);
        ListF<EventOrReminder> reminders = reminderInfos.map(EventOrReminder.reminderF());

        return reminders.isEmpty() ? events : events.plus(reminders).sorted(EventOrReminder.comparator(tz));
    }

    private void addEventOrReminderToDaysGroups(EventOrReminder e) {
        Instant start = ObjectUtils.max(monthStart.toInstant(), e.getStart(tz));
        Instant end = ObjectUtils.min(monthEnd.toInstant(), e.getEnd(tz));

        for (InstantInterval interval : AuxDateTime.splitByDays(new InstantInterval(start, end), tz)) {
            LocalDate date = new LocalDate(interval.getStart(), tz);
            ListF<EventsGroup> groups = dayGroups.getOrElseUpdate(date, Cf::arrayList);

            Option<EventsGroup> groupO = groups.find(EventsGroup.matchesF(interval.getStart()));
            Option<EventsGroup> under = groups.filter(EventsGroup.endsAfterF(interval.getStart())).lastO();

            if (!groupO.isPresent() && under.isPresent()) {
                groups.add(new EventsGroup(interval, under.get().stack, under.get().height + 1));
                under.get().stack.width = Math.max(under.get().height + 1, under.get().stack.width);

            } else if (!groupO.isPresent()) {
                groups.add(new EventsGroup(interval, new EventsStack(interval.getStart()), 0));
            }
            groupO.getOrElse(groups.last()).put(interval, e.getId());

            if (!e.isEvent()) break;
        }
    }

    private void addEventOrReminderToDaysMap(EventOrReminder e, boolean showIsAllDay) {
        if (e.isEvent()) {
            addEventDaysToMap(e.getEvent(), showIsAllDay);
        } else {
            addReminderToDaysMap(e.getReminder());
        }
    }

    private void addEventDaysToMap(EventIndentIntervalAndPerms ei, boolean showIsAllDay) {
        long eiStartMs = ei.getEventInterval().getStart().toInstant(tz).getMillis();
        long eiEndMs = ei.getEventInterval().getEnd().toInstant(tz).getMillis();
        long eiBndStartMs = Math.max(monthStart.getMillis(), eiStartMs);
        long eiBndEndMs = Math.min(monthEnd.getMillis(), eiEndMs);
        LocalDate date = new LocalDate(eiBndStartMs, tz);

        vIndent = -1; // event vindent has not been calculated yet
        DateTime midnight = date.toDateTimeAtStartOfDay(tz);
        int daysOffset = DateTimeFormatter.getDaysBtwMidnights(monthStart.getMillis(), midnight.getMillis(), tz);
        int eLenRemain = DateTimeFormatter.getConsumedGridCells(midnight.getMillis(), eiEndMs, tz);
        do { // we must go through a cycle at least once, even if dayStartMs == eiEndMsBounded (e.length == 0)
            addEventDayToMap(ei, showIsAllDay, midnight.toLocalDate(), daysOffset, eLenRemain);
            midnight = midnight.plusDays(1);
            daysOffset++;
            eLenRemain--;
        } while (midnight.getMillis() < eiBndEndMs);
    }

    // Adds an event element to events map, if needed.
    // NOTE: vIndentHolder != null, new value can be stored there.
    private void addEventDayToMap(
            EventIndentIntervalAndPerms eventInstanceInfo,
            boolean showIsAllDay, LocalDate date, int daysOffset, int eLenRemain)
    {
        long eiStartMs = eventInstanceInfo.getEventInterval().getStart().toInstant(tz).getMillis();
        long eiEndMs = eventInstanceInfo.getEventInterval().getEnd().toInstant(tz).getMillis();
        long eiBndDayStartMs = Math.max(date.toDateTimeAtStartOfDay(tz).getMillis(), eiStartMs);
        long eiBndDayEndMs = Math.min(date.plusDays(1).toDateTimeAtStartOfDay(tz).getMillis(), eiEndMs);

        LocalDateTime showStart = getShowStartAndEnd(eventInstanceInfo.getIndentInterval(), date, tz).get1();

        EventsData eventsData = dayEvents.getOrElseUpdate(date, EventsData::new);
        final int daysOffsetWeek = daysOffset % 7;
        // ssytnik (2008-11-21):
        // Terms and rules:
        // - an event consists of event parts, each of 1-day length;
        // - count of event parts equals count of consumed grid cells;
        // - an event part is called the 'head-of-div' one, if it
        //   starts a <div>. Only 'show-is-all-day' events may have
        //   <div>-s of length > 1 day, that is why parts of only these
        //   events can be treated as non-'head-of-div' ones. Therefore,
        //   parts of the simple events are always 'head-of-div' ones
        //   (NOTE: even if these events span more than one grid cell!) -
        //   this is because simple event parts are all drawn separately;
        // - 'show-is-all-day' event can have more than one long <div>
        //   in case if it spans several weeks in the grid (so its part
        //   is treated a 'head-of-div' one, if daysOffsetWeek == 0);
        // - answer v-block indent should be calculated if and only if
        //   it has not been calculated yet AND a 'head-of-div' part comes
        //   (unless the latter occurs, v-block indent can be kept '-1');
        // Related fixed/outstanding bugs: CAL-913, CAL-947
        boolean isHeadOfDiv = !showIsAllDay || vIndent == -1 || daysOffsetWeek == 0;
        if (isHeadOfDiv) {
            // (Re)calculate 'vindent' of the current event (either 'show-is-all-day' or not)
            vIndent = (eventsData.curVBlockIndent + 1);
            if (eventsData.ansVBlockIndent == -1) {
                eventsData.ansVBlockIndent = vIndent; // this is the top of v-block
            } // if no answer has been calculated yet
        } // if head-of-div
        // NOTE: vIndent > 0 here
        // We need to tell 'eventData' that it is filled up to id.vIndent, inclusive
        eventsData.curVBlockIndent = vIndent;
        // Increase number of event parts in this current day
        eventsData.dayEventsCount++;
//
//        if (indentsOnly && showIsAllDay) {
//            // for indentsOnly, we do not need events with show-is-all-day == 1:
//            // a) we do not like all day event XML-s in get-event-indents output;
//            // b) we took that event into attention by adding eventsData.dayEventsCount
//            return;
//        }
//         // NOW: we can't return here as later we'll need to add head-of-div, div-days attributes

        Element eEvent;
        if (indentsOnly) {
            eEvent = new Element("event");
            CalendarXmlizer.setAttr(eEvent, "id", eventInstanceInfo.getEventId());
            CalendarXmlizer.setAttr(eEvent, "layer-id", eventInstanceInfo.getIndent().getLayerOrResourceId());
            CalendarXmlizer.setAttr(eEvent, "show-is-all-day", showIsAllDay);
            CalendarXmlizer.setAttr(eEvent, "show-start-ts", showStart);
        } else {
            eEvent = new Element("event");
        }
        if (showIsAllDay) { // only such events can have isHeadOfDiv == true
            CalendarXmlizer.setAttr(eEvent, "head-of-div", isHeadOfDiv);
            if (isHeadOfDiv) {
                int divDays = Math.min(7 - daysOffsetWeek, eLenRemain);
                CalendarXmlizer.setAttr(eEvent, "div-days", divDays);
            } // inner if
        } else {
            // If not all-day event, take it into attention and handle indent attributes
            // both for 'get-events' and 'get-event-indents' cases
            // TODO: check whether we need this for WEEK view only.
            /* if (isWeek) { */
            eventsData.handleIndentAttributes(tz, eEvent, eiBndDayStartMs, eiBndDayEndMs);
            /* } */
        }
        if (!showIsAllDay && viewType == ViewType.WEEK) {
            writePositionAttributes(
                    eEvent, EventOrReminder.eventF().apply(eventInstanceInfo),
                    date, new Instant(eiBndDayStartMs));
        }
        if (!viewType.isGrey()) {
            eventsData.eventOrReminders.add(eEvent, Option.of(eventInstanceInfo));
        }
    }

    private void addReminderToDaysMap(RemindersEventInfo r) {
        Element eReminder = new Element("reminder");

        CalendarXmlizer.setAttr(eReminder, "client-id", r.getClientId());
        CalendarXmlizer.setAttr(eReminder, "external-id", r.getExternalId());
        CalendarXmlizer.setAttr(eReminder, "idx", r.getIdx());

        if (!indentsOnly) {
            CalendarXmlizer.appendElm(eReminder, "show-start-ts", new LocalDateTime(r.getStart(), tz));
            CalendarXmlizer.appendElm(eReminder, "show-end-ts", new LocalDateTime(r.getStart().plus(EventsGroup.REMINDER_LENGTH), tz));
            CalendarXmlizer.appendElm(eReminder, "name", r.getName());

            r.getDescription().forEach(desc -> eReminder.addContent(TagReplacement.processText(
                    "description", desc, TagReplacement.Action.CONVERT_XML, linkSignKey, EventRoutines.TAGS)));
        }

        LocalDate date = new LocalDate(r.getStart(), tz);
        if (viewType == ViewType.WEEK) {
            writePositionAttributes(eReminder, EventOrReminder.reminderF().apply(r), date, r.getStart().toInstant());

            Calendar c = DateTimeFormatter.createCalendar(tz);
            c.setTimeInMillis(r.getStart().getMillis());
            CalendarXmlizer.setAttr(eReminder, "vindent", c.get(Calendar.MINUTE) / 5);
        }
        dayEvents.getOrElseUpdate(date, EventsData::new)
                .eventOrReminders.add(eReminder, Option.empty());
    }

    private void writePositionAttributes(Element el, EventOrReminder e, LocalDate date, Instant startTs) {
        EventsGroup group = dayGroups.getOrThrow(date).find(EventsGroup.matchesF(startTs)).getOrThrow(
                "filled stacks expected");

        EventInGroupPosition position = group.getPosition(e.getId());

        CalendarXmlizer.setAttr(el, "g-pos", position.groupPos);
        CalendarXmlizer.setAttr(el, "g-size", position.groupSize);

        if (position.subgroupStart.isPresent() && position.subgroupSize.isPresent()) {
            CalendarXmlizer.setAttr(el, "sg-start", position.subgroupStart.get());
            CalendarXmlizer.setAttr(el, "sg-size", position.subgroupSize.get());
        }
        CalendarXmlizer.setAttr(el, "s-under", group.height);
        CalendarXmlizer.setAttr(el, "s-over", group.stack.width - group.height);
    }

    // If there are selected events, we update 'has_created_events flag'. Returns updated value.
    private boolean handleHasCreatedEvent(int count) {
        if (!sSet) {
            // (anonymous case) true, as if we created event => nothing to do
            return true;
        }
        if (count > 0 && !settings.get().getCommon().getHasCreatedEvent()) {
            PolicyHandle holder = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
            try {
                settingsRoutines.updateHasCreatedEvent(settings.get().getUid(), true);
            } catch (Exception e) {
                logger.warn(e, e);
            } finally {
                holder.popSafely();
            }
        }

        return settings.get().getCommon().getHasCreatedEvent();
    }

    private static Tuple2<LocalDateTime, LocalDateTime> getShowStartAndEnd(
            EventIndentInterval indent, LocalDate date, DateTimeZone tz)
    {
        LocalDateTime start = indent.getEventInterval().getStart().toLocalDateTime(tz);
        LocalDateTime end = indent.getEventInterval().getEnd().toLocalDateTime(tz);

        LocalDateTime showStart = ObjectUtils.max(start, date.toLocalDateTime(LocalTime.MIDNIGHT));
        LocalDateTime showEnd = ObjectUtils.min(end, date.plusDays(1).toLocalDateTime(LocalTime.MIDNIGHT));

        return Tuple2.tuple(showStart, showEnd);
    }
} //~
