package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import java.util.concurrent.atomic.AtomicInteger;

import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.joda.time.MutableDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.ui.CmdGetHolidaysA;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventsFilter;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.grid.GrayedWeekNoAppender;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.event.repetition.EventIndentIntervalAndPerms;
import ru.yandex.calendar.logic.event.repetition.InfiniteInterval;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.Binary;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.holidays.DayInfo;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.commune.holidays.MapDayInfoHandler;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class CmdGetEventsForWidget extends UserXmlCommand {
    private static final String CMD_TAG = "get-events-for-widget";

    private final String showDateStr;
    private final String outCalendar;
    private final String limitStr;
    private final String onlyVisibleInUiStr;
    private final String holidaysForStr;

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventDao eventDao;

    public CmdGetEventsForWidget(
            AuthInfo ai, String showDate, String outCalendar, String limit, String onlyVisibleInUi, String holidaysFor)
    {
        super(CMD_TAG, ai);
        this.showDateStr = showDate;
        this.outCalendar = outCalendar;
        this.limitStr = limit;
        this.onlyVisibleInUiStr = onlyVisibleInUi;
        this.holidaysForStr = holidaysFor;
        this.setMustHaveAuth(true);
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        boolean includeCalendar = StringUtils.isEmpty(outCalendar) || Binary.parseBoolean(outCalendar);
        LocalDate showDate = DateTimeFormatter.toDate(showDateStr, new LocalDate(tz));

        DayOfWeek startWeekDay = settingsRoutines.getSettingsByUidIfExists(uidO.get())
                .map(s -> s.getCommon().getStartWeekday()).getOrElse(DayOfWeek.MONDAY);

        Tuple2<DateTime, DateTime> monthBounds = DateTimeFormatter.getViewTypeBounds(
                tz, showDate, ViewType.MONTH, startWeekDay);
        GrayedWeekNoAppender grayerWeekNoAppender = ViewType.WEEK.createGrayedWeekNoAppender(showDate, startWeekDay, tz);

        LayerIdPredicate layerIds = LayerIdPredicate.allForUser(uidO.get(), Binary.parseBoolean(onlyVisibleInUiStr));

        Instant monthStart = monthBounds.get1().toInstant();
        Instant monthEnd = monthBounds.get2().toInstant();
        InstantInterval monthInterval = new InstantInterval(monthStart, monthEnd);

        ListF<EventIndentIntervalAndPerms> monthEvents = Cf.list();

        if (includeCalendar) {
            monthEvents = eventRoutines.getSortedIndentsIMayView(
                    userInfoO, layerIds, EventLoadLimits.intersectsInterval(monthInterval),
                    EventsFilter.DEFAULT, ActionSource.WEB);
        }

        int eventsLimit = Cf.Integer.parseSafe(limitStr).getOrElse(3);
        InfiniteInterval eventsInterval;

        if (!includeCalendar) {
            Instant start = AuxDateTime.toInstantIgnoreGap(showDate.toLocalDateTime(LocalTime.MIDNIGHT), tz);
            Instant end = AuxDateTime.toInstantIgnoreGap(showDate.plusDays(1).toLocalDateTime(LocalTime.MIDNIGHT), tz);
            eventsInterval = new InfiniteInterval(start, Option.of(end));
        } else {
            eventsInterval = new InfiniteInterval(Instant.now(), Option.<Instant>empty());
        }

        MapF<LocalDate, AtomicInteger> counts = Cf.hashMap();
        ListF<EventIndentIntervalAndPerms> foundEvents = Cf.arrayList();

        for (EventIndentIntervalAndPerms event : monthEvents) {
            InstantInterval interval = event.getEventInterval().toInstantInterval(tz);
            if (interval.overlaps(monthInterval)) {
                for (InstantInterval split: AuxDateTime.splitByDays(interval.overlap(monthInterval), tz)) {
                    counts.getOrElseUpdate(
                            new LocalDate(split.getStart(), tz), () -> new AtomicInteger()).incrementAndGet();
                }
            }
            if (foundEvents.size() < eventsLimit && eventsInterval.overlaps(interval)) {
                foundEvents.add(event);
            }
        }

        if (foundEvents.size() < eventsLimit
            && (!includeCalendar || !eventsInterval.getEnd().exists(i -> !i.isAfter(monthEnd))))
        {
            InfiniteInterval interval = !includeCalendar
                    ? eventsInterval
                    : new InfiniteInterval(monthEnd, eventsInterval.getEnd());

            EventLoadLimits limits = EventLoadLimits.startsInInterval(interval)
                    .withResultSize(eventsLimit - foundEvents.size())
                    .withExcludeIds(foundEvents.map(EventIndentIntervalAndPerms::getEventId));

            foundEvents.addAll(eventRoutines.getSortedIndentsIMayView(
                    userInfoO, layerIds, limits, EventsFilter.DEFAULT, ActionSource.WEB));
        }

        Element rootElement = ctx.getRootElement();

        CalendarXmlizer.setDtfAttr(rootElement, "show-date", showDate, tz);
        CalendarXmlizer.setAttr(rootElement, "start-weekday", startWeekDay.getDbValue());

        if (includeCalendar) {
            Element eCalendar = new Element("calendar");

            int countryId = CmdGetHolidaysA.lookupGeoId(holidaysForStr).orElse(GeobaseIds.RUSSIA);
            MapDayInfoHandler diHandler = new MapDayInfoHandler();
            HolidayRoutines.processDates(
                    monthBounds._1.toLocalDate(), monthBounds._2.toLocalDate().minusDays(1),
                    countryId, OutputMode.ALL, false, diHandler);

            MutableDateTime mdt = new MutableDateTime(monthStart, tz);
            while (mdt.getMillis() < monthEnd.getMillis()) {
                LocalDate dayDate = new LocalDate(mdt.getMillis(), tz);
                Element dayElement = new Element("day");

                CalendarXmlizer.setAttr(dayElement, "date", dayDate);
                grayerWeekNoAppender.append(dayElement, dayDate);

                if (counts.getO(dayDate).isPresent()) {
                    CalendarXmlizer.setAttr(dayElement, "count", counts.getOrThrow(dayDate));
                }
                DayInfo dayInfo = diHandler.getDayInfo(dayDate);
                Option<String> nameO = dayInfo.getNameO();

                if (dayInfo.isDayOff() || dayInfo.isTransfer() || nameO.isPresent()) {
                    CalendarXmlizer.setAttr(dayElement, "is-holiday", dayInfo.isDayOff());
                    if (nameO.isPresent()) {
                        CalendarXmlizer.setAttr(dayElement, "holiday-name", nameO.get());
                    }
                }
                eCalendar.addContent(dayElement);
                mdt.addDays(1);
            }
            rootElement.addContent(eCalendar);
        }

        Element eEvents = new Element("events");

        ListF<Long> foundEventIds = foundEvents.map(EventIndentIntervalAndPerms::getEventId);
        MapF<Long, String> eventNameById = eventDao.findEventsNameByIdsSafe(foundEventIds).toMap();

        for (EventIndentIntervalAndPerms event : foundEvents) {
            EventInterval interval = event.getEventInterval();
            if (eventsInterval.overlaps(interval.toInstantInterval(tz))) {
                Element eEvent = new Element("event");

                CalendarXmlizer.appendElm(eEvent, "id", event.getEventId());
                CalendarXmlizer.appendElm(eEvent, "start-ts", interval.getStart().toLocalDateTime(tz));
                CalendarXmlizer.appendElm(eEvent, "end-ts", interval.getEnd().toLocalDateTime(tz));
                CalendarXmlizer.appendElm(eEvent, "is-all-day", event.getIndent().isAllDay());
                CalendarXmlizer.appendElm(eEvent, "name", eventNameById.getOrThrow(event.getEventId()));

                eEvents.addContent(eEvent);
            }
        }
        CalendarXmlizer.setAttr(eEvents, "count", eEvents.getChildren().size());
        rootElement.addContent(eEvents);
    }
}
