package ru.yandex.calendar.frontend.web.cmd.run.ui.event;

import lombok.val;
import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.validation.Captcha;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.lang.Validate;

public class EventCmdManager {
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Captcha captcha;

    public EventResponse getEvent(EventQuery eventQuery, Option<String> invitationToken, Option<UserInfo> userO,
                                  Option<Instant> startTs) {
        Option<PassportUid> tokenUserUidO = Option.empty();
        EventInstanceInfo eventInstanceInfo;

        val source = ActionSource.WEB;

        if (invitationToken.isPresent()) {
            Option<UserParticipantInfo> participantO = eventInvitationManager
                .getUserParticipantByPrivateToken(invitationToken.get());

            if (participantO.isNotEmpty()) {
                long eventId = participantO.get().getEventId();
                Validate.V.equals(eventQuery.getEventId(), eventId);
                tokenUserUidO = participantO.get().getUid();
                Option<UserInfo> tokenUserO = userManager.getUserInfos(tokenUserUidO).singleO();

                eventInstanceInfo = eventRoutines.getSingleInstance(tokenUserUidO, startTs, eventId, source);
                val eventWithRelations = eventDbManager.getEventWithRelationsById(eventId);

                if (tokenUserO
                        .map(user -> {
                            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, eventWithRelations);
                            return authorizer.canViewEvent(user, eventAuthInfo, source);
                        })
                        .getOrElse(() -> {
                            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(eventWithRelations);
                            return authorizer.canViewEvent(eventAuthInfo, source);
                        })) {
                    return new EventResponse(tokenUserUidO, eventQuery.getLayerId(), eventInstanceInfo);
                }
            }
        }

        Option<PassportUid> uidO = userO.map(UserInfo.getUidF());
        if (userO.isPresent() && !uidO.equals(tokenUserUidO)) {

            eventInstanceInfo = eventRoutines.getSingleInstance(uidO, startTs, eventQuery.getEventId(), source);
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(userO.get(), eventInstanceInfo.getEventWithRelations());
            if (authorizer.canViewEvent(userO.get(), eventAuthInfo, source)) {
                return new EventResponse(tokenUserUidO, eventQuery.getLayerId(), eventInstanceInfo);
            }
        }

        throw new PermissionDeniedUserException();
    }

    CreateInfo createEvent(
            final XmlCmdContext ctx, final EventData eventData, final Option<Captcha.CaptchaData> captchaData,
            final DateTimeZone tz, final PassportUid uid, final ActionInfo actionInfo)
    {
        if (eventData.getInvData().isEventInvitationUpdateData() && captchaData.isPresent()) {
            int newEmailsCount = eventData.getInvData().getEventInvitationUpdateData().getNewEmails().size();
            captcha.check(uid, newEmailsCount, captchaData.get());
        }

        final CreateInfo createInfo = eventWebManager.createUserEvent(
                uid, eventData, InvitationProcessingMode.SAVE_ATTACH_SEND, actionInfo);

        if (createInfo.getEvent().getName().startsWith(SpecialNames.FAIL_ON_CREATE)) {
            throw new RuntimeException(SpecialNames.FAIL_ON_CREATE);
        }

        long id = createInfo.getEvent().getId();

        Element rootElement = ctx.getRootElement();
        CalendarXmlizer.appendElmColl(rootElement, null, "id", Cf.list(id));
        if (createInfo.getLayerId().isPresent()) {
            rootElement.addContent(getLayerElement(createInfo.getLayerId().get()));
        }
        CalendarXmlizer.appendElm(rootElement, EventXmlizer.NAME_NAME, createInfo.getEvent().getName());
        CalendarXmlizer.appendElm(rootElement, EventXmlizer.START_TS_NAME, createInfo.getEvent().getStartTs(), tz);
        CalendarXmlizer.appendElm(rootElement, EventXmlizer.END_TS_NAME, createInfo.getEvent().getEndTs(), tz);
        CalendarXmlizer.appendElm(rootElement, EventXmlizer.IS_ALL_DAY_NAME, createInfo.getEvent().getIsAllDay());
        CalendarXmlizer.appendElm(rootElement, "instance-start-ts",
                createInfo.getEvent().getStartTs(), DateTimeZone.UTC);

        return createInfo;
    }

    private Element getLayerElement(long layerId) {
        Layer l = layerRoutines.getLayerById(layerId);
        Element eLayer = new Element("layer");
        CalendarXmlizer.appendElm(eLayer, "name", layerRoutines.evalLayerName(l, Option.empty()));
        CalendarXmlizer.appendElm(eLayer, "id", l.getId());
        return eLayer;
    }

    public ListF<UidOrResourceId> parseParticipantEmailsAndMaybeMe(String emailsAndMaybeMe, final PassportUid meUid) {
        Validate.V.notEmpty(emailsAndMaybeMe);
        final Option<PassportAuthDomain> defaultDomainO = passportAuthDomainsHolder.getDefaultDomain();

        return Cf.x(emailsAndMaybeMe.split(",")).map(new Function<String, UidOrResourceId>() {
            public UidOrResourceId apply(String s) {
                if (s.equals("me")) {
                    return UidOrResourceId.user(meUid);
                } else if (s.contains("@")) {
                    return userManager.getSubjectIdByEmail(Emails.punycode(s)).get();
                } else if (defaultDomainO.isPresent()) {
                    return userManager.getSubjectIdByEmail(Emails.punycode(s + "@" + defaultDomainO.get().getDomain().getDomain())).get();
                } else {
                    throw new IllegalArgumentException("Can not parse user " + s);
                }
            }
        });
    }

}
